#!/usr/bin/env python

import sys, re, string

supp_types = ["int", "char", "float", "double", "string"]

def make_fill_evt(tlist, class_name):
    ret = "\tvoid fill_event(Event *pub) {\n"    
    fun_indent = "\t\t"
    format = "pub.add_tuple(\"%s\", %s);\n"

    for tuple in tlist:
        type, attr_name, var_name = tuple[0], tuple[1], tuple[2]
        if type not in supp_types:
            raise Exception, "type not supported";

        ret += format % (attr_name, var_name)

    ret += "\nDOMDocument *doc = CREATEDOC(\"%s\");\n" % (class_name)
    ret += "DOMNode *root = serialize_XML(doc);\n"
    ret += "string *str = new string(g_XML_man->serialize(root));\n"

    ret = string.replace(ret, '\n', "\n%s" % fun_indent) 
    ret += format % ("objdump", "*str")      # dont add indent to last line!
    ret += "\t}\n\n"
    
    return ret
        
def make_update_obj(tlist, class_name):
    ret = "\tvoid update_object(Event *pub) {\n"
    fun_indent = "\t\t"
    format = "%s = pub->get_attribute(\"%s\")->get_%s();\n"

    for tuple in tlist:
        type, attr_name, var_name = tuple[0], tuple[1], tuple[2]
        if type not in supp_types:
            raise Exception, "type not supported"
        ret += format % (var_name, attr_name, type)

    ret += "\nDOMDocument *doc = g_XML_man->parse(pub->get_attribute(\"objdump\")->get_string());\n"
    ret += "if (!doc) throw Exception(\"Parsing error while constructing %s\");\n" % class_name
    ret = string.replace(ret, '\n', "\n%s" % fun_indent) 
    ret += "deserialize_XML(doc->getDocumentElement());\n"
    ret += "\t}\n\n"
    
    return ret

def transform_named(attrs, class_name):
    """Takes care of variables which have to exposed to the pubsub system.
    Generates code while things are going down and when things are coming up.
    This boils down to: fill_event(Event *) and update_object(Event *)
    respectively."""

    var_re = re.compile(r"\[(\w+)\s+\"([\w.]+)\"\s+(\w+)\]", re.S|re.MULTILINE)
    tuple_list = var_re.findall(attrs)

    if tuple_list == None:
        return ""

    fill_evt_routine = make_fill_evt(tuple_list, class_name)
    update_obj_routine = make_update_obj(tuple_list, class_name)

    code = '' 
    for tuple in tuple_list:
        code += "\t%s %s;\n" % (tuple[0], tuple[2])

    code += "\n" + fill_evt_routine + update_obj_routine
    return code

def make_serialize_xml(tlist, class_name):
    ret = "\tDOMNode* serialize_XML(DOMDocument *doc) {\n"
    fun_indent = "\t\t"
    format = "APPEND_ELEMENT(doc, root, \"%s\", %s);\n"
    
    ret += "DOMElement *root = doc->getDocumentElement();\n"
    ret += "DOMElement *elem = 0;\n"
    
    for tuple in tlist:
        type = tuple[0]
        if type not in supp_types:
            raise Exception, "type not supported"

        vars = re.split(r",\s*", tuple[1])
        for var in vars:
            if type == "string":
                value = "X((char *) %s.c_str())" % var
            else:
                value = "ENCODE_NUMBER(%s)" % var
            ret += format % (var, value)

    ret = string.replace(ret, '\n', "\n%s" % fun_indent)
    ret += "return root;\n" 

    ret += "\t}\n\n"
    return ret

def make_deserialize_xml(tlist, class_name):
    ret = "\tvoid deserialize_XML(DOMElement *root) {\n"
    fun_indent = "\t\t"
    format = "ASSIGN_VAR(root, \"%s\", %s);\n"

    ret += "DOMElement *elem = 0;\n"
    for tuple in tlist:
        type = tuple[0]
        if type not in supp_types:
            raise Exception, "type not supported"

        vars = re.split(r",\s*",tuple[1])
        for var in vars:
            if type == "string":
                asgn_cmd = "%s = string(Y(valstr))" % var
            else:
                asgn_cmd = "%s = g_XML_man->decode_%s(valstr)" % (var, type)
            ret += format % (var, asgn_cmd)

    ret = string.replace(ret, '\n', "\n%s" % fun_indent)
    ret += "/* everything okay */\n";
    ret += "\t}\n\n"
    return ret
    
def transform_serial(attrs, class_name):
    """Takes care of variables which are not exposed to Mercury but have to be
    serialized, none the less. Right now, it supports only the built-in C types
    along with the string type."""

    ser_re = re.compile(r"(\w+)\s*([^;]*);", re.S|re.MULTILINE)
    tuple_list = ser_re.findall(attrs)

    if tuple_list == None:
        return ""

    ser_XML_routine = make_serialize_xml(tuple_list, class_name)
    deser_XML_routine = make_deserialize_xml(tuple_list, class_name)

    code = ''
    for tuple in tuple_list:
        code += "\t%s %s;\n" % (tuple[0], tuple[1])

    code += "\n" + ser_XML_routine + deser_XML_routine
    return code

def do_magic(match):
    """Main function.  Takes each ANNOTATION block and
    substitutes with various C++ statements.  """
    
    block = match.group()
    cl_name = re.search(r"CLASS\s*([\w]+)", block, re.S|re.MULTILINE)
    if cl_name == None:
        raise Exception, "No class name declaration found!"
    else:
        class_name = cl_name.group(1)
        
    subst_str = ""
    
    exp = re.compile(r"NAMED_ATTR\s*\((.*?)\)\s*;", re.S|re.MULTILINE);
    named_attrs = exp.search(block);
    if named_attrs != None:
        subst_str = subst_str + transform_named(named_attrs.group(1), class_name)

    exp = re.compile(r"SERIAL_ATTR\s*\((.*?)\)\s*;", re.S|re.MULTILINE);
    serial_attrs = exp.search(block);
    if serial_attrs != None:
        subst_str = subst_str + transform_serial(serial_attrs.group(1), class_name)

    return subst_str
    
def process_file(file):
    try:
        f = open(file)
    except:
        sys.stderr.write ("can't open %s: [%s]\n" % (file, sys.exc_value))
        return

    everything = string.join(f.readlines(), '')
    exp = re.compile(r"BEGIN_ANNOTATE(.*)END_ANNOTATE", re.S|re.MULTILINE)
    res = exp.sub(do_magic, everything)
    print res

for arg in sys.argv[1:]:
    process_file(arg)
    
