////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <util/Rect.h>
#include <util/RTree.h>
#include <util/debug.h>
#include <mercury/MercuryID.h>
#include <iostream>
#include <Mercury.h>

int g_VerbosityLevel = 1;

#define Value long long

class Spatobj {
    friend ostream& operator<< (ostream& os, const Spatobj& rect);
    Value x1, y1, x2, y2;
public:
    Spatobj (Value x1, Value y1, Value x2, Value y2) : x1 (x1)
	, y1 (y1), x2 (x2), y2 (y2) 
	{
	    ASSERT (x1 <= x2);
	    ASSERT (y1 <= y2);
	}

    void GetExtent (Rect<Value>* extent) {
	ASSERT (extent->dim == 2);

	extent->min[0] = x1;
	extent->min[1] = y1;
	extent->max[0] = x2;
	extent->max[1] = y2;
    }

    bool Overlaps (const Spatobj& obj) {
	if (x1 > obj.x2 || x2 < obj.x1)
	    return false;
	if (y1 > obj.y2 || y2 < obj.y1)
	    return false;
	return true;
    }
};

ostream& operator<< (ostream& os, const Spatobj& rect) {
    os << "x(" << rect.x1 << "," << rect.x2 << ")";
    os << " cross y(" << rect.y1 << "," << rect.y2 << ")";
    return os;
}

int g_NumItems;

static OptionType g_Options[] =
{
    { 'n', "nitems", OPT_INT, "number of items to insert",
	&g_NumItems, "10", NULL },
	{ 0, 0, 0, 0, 0, 0, 0 }
};

Spatobj **g_Array;

void GetMatches (Spatobj *obj, list<Spatobj *> *res)
{
    res->clear ();
    for (int i = 0; i < g_NumItems; i++) {
	if (g_Array[i]->Overlaps (*obj))
	    res->push_back (g_Array[i]);
    }
}

gmp_randstate_t g_GMPRandState;        // for large random numbers

Value GetRandom (const Value& max)
{
#ifdef Value
    Value ret = (int) (drand48 () * max);
#else
    Value ret = VALUE_NONE;
    unsigned int p = max.getui ();
    mpz_urandomm ((MP_INT *) &ret, g_GMPRandState, &max);
#endif

    return ret;
}

static void grab_list (list<Spatobj *> *plist, Spatobj *obj) { 
    plist->push_back (obj);
}

int main (int argc, char *argv[]) 
{
    DBG_INIT(NULL);
    InitializeMercury(&argc, argv, g_Options, false);

    RTree<Spatobj, Value> *rtree = new RTree<Spatobj, Value> (2);
    g_Array = new Spatobj *[g_NumItems];

    // srand ((int) getpid () ^ (int) time(NULL));
    gmp_randinit_default (g_GMPRandState);
#ifdef Value
    Value amin = -10000000;
    Value amax =  10000000;
    Value htmax =  8000000;
    Value wdmax =  8000000;
#else
    Value amin ("-1000000000", /* base */ 10);
    Value amax ("1000000000", /* base */ 10);
    Value htmax ("800000000", /* base */ 10);
    Value wdmax ("800000000", /* base */ 10);
#endif
    Value diff = amax;
    diff -= amin;


    int v = g_VerbosityLevel;
    g_VerbosityLevel = -10;

    for (int i = 0; i < g_NumItems; i++) {
	Value x1 = GetRandom (diff);  x1 += amin;
	Value y1 = GetRandom (diff);  y1 += amin;

	Value width = GetRandom (wdmax) + 10000;
	Value height = GetRandom (htmax) + 10000;

	// cout << x1 << "," << x1 + width << " " << y1 << "," << y1 + height << endl;

	g_Array[i] = new Spatobj (x1, y1, x1 + width, y1 + height);
	START(INSERT);
	rtree->Insert (g_Array[i]);
	STOP(INSERT);

	rtree->CheckInvariants ();
	ASSERT ((i + 1) == rtree->GetNumRecords ());
    }
    g_VerbosityLevel = v;

    DB(-1) << *rtree;

    // INFO << "sleeping for some rest" << endl;
    // sleep (5);

    // search
    list<Spatobj *> plist, plist_ref;

    for (int i = 0; i < g_NumItems; i++) {
	Value x1 = GetRandom (diff);  x1 += amin;
	Value y1 = GetRandom (diff);  y1 += amin;

	Value width = GetRandom (wdmax) + 10000;
	Value height = GetRandom (htmax) + 10000;

	Spatobj *o = new Spatobj (x1, y1, x1 + width, y1 + height);

	GetMatches (o, &plist_ref);

	plist.clear ();
	START(SEARCH);
	rtree->GetOverlaps (o, wrap (grab_list, &plist));
	STOP(SEARCH);

	NOTE(RESULT_COUNT, plist.size ());
	DB(-2) << "overlapping rects for " << *o << endl;
	for (list<Spatobj *>::iterator it = plist.begin (); it != plist.end (); ++it) 
	    DB(-2) << "\t" << *(*it) << endl;    

	DB(-2) << "REAL overlapping rects for " << *o << endl;
	for (list<Spatobj *>::iterator it = plist_ref.begin (); it != plist_ref.end (); ++it) 
	    DB(-2) << "\t" << *(*it) << endl;    
	ASSERT (plist.size () == plist_ref.size ());
	delete o;
    }


    for (int i = 0; i < g_NumItems; i++) {
	START(ERASE);
	ASSERT (rtree->Erase (g_Array[i]));
	STOP(ERASE);
	rtree->CheckInvariants ();
	delete g_Array[i];
    }

    Benchmark::print ();
    delete [] g_Array;
    delete rtree;
    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
