/* vim: set sw=4 softtabstop=4 ts=8 noet: -*- Mode:c++; c-basic-offset:4; tab-width:8; indent-tabs-mode:t -*- */
#include <util/Rect.h>
#include <util/RTree.h>
#include <util/debug.h>
#include <iostream>
#include <Mercury.h>

int g_VerbosityLevel = 1;

class Spatobj {
    friend ostream& operator<< (ostream& os, const Spatobj& rect);
    int x1, y1, x2, y2;
public:
    Spatobj (int x1, int y1, int x2, int y2) : x1 (x1)
	, y1 (y1), x2 (x2), y2 (y2) 
    {
	ASSERT (x1 <= x2);
	ASSERT (y1 <= y2);
    }

    void GetExtent (Rect<int>* extent) {
	ASSERT (extent->dim == 2);

	extent->min[0] = x1;
	extent->min[1] = y1;
	extent->max[0] = x2;
	extent->max[1] = y2;
    }

    bool Overlaps (const Spatobj& obj) {
	if (x1 > obj.x2 || x2 < obj.x1)
	    return false;
	if (y1 > obj.y2 || y2 < obj.y1)
	    return false;
	return true;
    }
};

ostream& operator<< (ostream& os, const Spatobj& rect) {
    os << "x(" << rect.x1 << "," << rect.x2 << ")";
    os << " cross y(" << rect.y1 << "," << rect.y2 << ")";
    return os;
}

int g_NumItems;

static OptionType g_Options[] =
{
    { 'n', "nitems", OPT_INT, "number of items to insert",
      &g_NumItems, "10", NULL },
    { 0, 0, 0, 0, 0, 0, 0 }
};

Spatobj **g_Array;

void GetMatches (Spatobj *obj, list<Spatobj *> *res)
{
    res->clear ();
    for (int i = 0; i < g_NumItems; i++) {
	if (g_Array[i]->Overlaps (*obj))
	    res->push_back (g_Array[i]);
    }
}

int main (int argc, char *argv[]) {
    DBG_INIT(NULL);
    InitializeMercury(&argc, argv, g_Options, false);
    
    RTree<Spatobj, int> *rtree = new RTree<Spatobj, int> (2);
    g_Array = new Spatobj *[g_NumItems];

    int v = g_VerbosityLevel;
    g_VerbosityLevel = -10;

    for (int i = 0; i < g_NumItems; i++) {
	int width = 20 + (int) (drand48() * 500);
	int height = 30 + (int) (drand48() * 400);
	int x1 = 100 + (int) (drand48() * 300);
	int y1 = 50 + (int) (drand48() * 400);
	
	g_Array[i] = new Spatobj (x1, y1, x1 + width, y1 + height);
	START(INSERT);
	rtree->Insert (g_Array[i]);
	STOP(INSERT);
	
	rtree->CheckInvariants ();
	ASSERT ((i + 1) == rtree->GetNumRecords ());
    }
    g_VerbosityLevel = v;

    DB(-1) << *rtree;
    
    // search
    list<Spatobj *> plist, plist_ref;
    
    for (int i = 0; i < g_NumItems; i++) {
	int width = 20 + (int) (drand48() * 500);
	int height = 30 + (int) (drand48() * 400);
	int x1 = 100 + (int) (drand48() * 300);
	int y1 = 50 + (int) (drand48() * 400);

	Spatobj *o = new Spatobj (x1, y1, x1 + width, y1 + height);

	GetMatches (o, &plist_ref);
	
	plist.clear ();
	START(SEARCH);
	rtree->GetOverlaps (o, &plist);
	STOP(SEARCH);

	NOTE(RESULT_COUNT, plist.size ());
	DB(-2) << "overlapping rects for " << *o << endl;
	for (list<Spatobj *>::iterator it = plist.begin (); it != plist.end (); ++it) 
	    DB(-2) << "\t" << *(*it) << endl;    

	DB(-2) << "REAL overlapping rects for " << *o << endl;
	for (list<Spatobj *>::iterator it = plist_ref.begin (); it != plist_ref.end (); ++it) 
	    DB(-2) << "\t" << *(*it) << endl;    
	ASSERT (plist.size () == plist_ref.size ());
	delete o;
    }

    
    for (int i = 0; i < g_NumItems; i++) {
	START(ERASE);
	ASSERT (rtree->Erase (g_Array[i]));
	STOP(ERASE);
	rtree->CheckInvariants ();
	delete g_Array[i];
    }

    Benchmark::print ();
    delete [] g_Array;
    delete rtree;
    return 0;
}
