////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <Mercury.h>
#include <util/OS.h>
#include <mercury/Message.h>
#include <wan-env/RealNet.h>
#include <util/types.h>
#include <util/TimeVal.h>
#include <util/debug.h>
#include <util/Benchmark.h>
#include <om/Manager.h>
#include <om/OMMessage.h>
#include <om/OMLoadManager.h>
#include <mercury/ID.h>

/*
 * This is an N^2 packet-loss rate stress-test. Bring up one node as the
 * "master" with:
 *
 * ./pkttest -v 0 <host>:<port> m <nservers>
 *
 * And bring up the other nservers as slaves with:
 *
 * ./pkttest -v 0 <host>:<port> <masterhost>:<masterport>
 *
 * After everyone is started, each one will start sending packets at the
 * specified rate to each of the other N servers. Every epoch, it will
 * report the number of packets received and deemed lost.
 */

static SID    self;
static bool   ismaster;
static bool   started;
static SID    master;
static uint32 nservers;
static SIDSet servers;

struct SenderInfo {
    uint32 last_seqno;
    uint32 pkts_recved;
    uint32 pkts_dropped;

    SenderInfo() : last_seqno(0), pkts_recved(0), pkts_dropped(0) {}

    void Reset() {
	pkts_recved  = 0;
	pkts_dropped = 0;
    }
};

typedef map<SID, SenderInfo *, less_SID> SenderInfoMap;
typedef SenderInfoMap::iterator SenderInfoMapIter;

static SenderInfoMap infoMap;

static int send_interval;
static int pkt_size;
static int burst_interval;
static int burst_size;
static int epoch;
static TransportType proto;
static char proto_str[255];

static OptionType options[] = {
    { '/', "send_interval", OPT_INT,
	"interval between each packet group sent (ms)", 
	&send_interval, "100", NULL},
	{ '/', "burst_interval", OPT_INT,
	    "interval between each packet in a group (ms)", 
	    &burst_interval, "1", NULL},
	    { '/', "burst_size", OPT_INT,
		"number of packets in a group (ms)", 
		&burst_size, "1", NULL},
		{ '/', "pkt_size", OPT_INT,
		    "size of packets, excluding header info  (bytes)", 
		    &pkt_size, "128", NULL},
		    { '/', "epoch", OPT_INT,
			"length of each measurement epoch (ms)", 
			&epoch, "5000", NULL},
			{ '/', "proto", OPT_STR,
			    "transport protocol to test",
			    proto_str, "UDP", NULL},
			    { 0, 0, 0, 0, 0, 0, 0 }
};

void ReadJoin(RealNet *net)
{
    while (servers.size() < nservers) {
	RealNet::DoWork();
	Message *msg = NULL;
	SID fromWhom;
	net->GetNextMessage(&fromWhom, &msg);
	if (msg && msg->GetType() == MSG_PING) {
	    INFO << "Joined: " << fromWhom << endl;
	    servers.insert(fromWhom);
	}
    }
}

void WaitForStart(RealNet *net)
{
    MsgPing ping;
    while ( net->SendMessage(&ping, &master, PROTO_TCP) < 0 ) {
	WARN << "ping send to " << master << " failed!" << endl;
    }

    while (true) {
	RealNet::DoWork();
	Message *msg = 0;
	SID fromWhom;
	net->GetNextMessage(&fromWhom, &msg);
	if (msg && msg->GetType() == MSG_SIDLIST) {
	    servers = ((MsgSIDList *)msg)->sidlist;
	    break;
	}
    }
}

void BroadcastStart(RealNet *net)
{
    SID me = net->GetAppID();
    MsgSIDList msg(me);
    msg.sidlist = servers;

    for (SIDSetIter it = servers.begin(); it != servers.end(); it++) {
	if (*it == master) {
	    continue;
	} else {
	    SID sid = *it;
	    INFO << "Sending start to: " << sid << endl;
	    while ( net->SendMessage(&msg, &sid, PROTO_TCP) < 0 ) {
		WARN << "ping send to " << sid << " failed!" << endl;
	    }
	}
    }
}

void DoIt(RealNet *net)
{
    TimeVal start;
    gettimeofday(&start, NULL);
    TimeVal last_send = start;

    SID me = net->GetAppID();

    uint32  seqno = 1;
    MsgBlob blob(pkt_size, me);

    TimeVal next_burst = TIME_NONE;
    int burst_index = 0;

    while (true) {
	TimeVal now;
	gettimeofday(&now, NULL);

	if (last_send + send_interval <= now) {
	    burst_index = 0;
	    next_burst  = now;

	    last_send = last_send + send_interval;
	}

	if (burst_index < burst_size &&
		next_burst <= now) {

	    for (SIDSetIter it = servers.begin(); it != servers.end(); it++) {
		if (*it == self) continue;
		SID sid = *it;
		blob.nonce = seqno;
		START(RealNet::Send);
		int ret = net->SendMessage(&blob, &sid, proto);
		STOP(RealNet::Send);
		if ( ret < 0 ) {
		    WARN << "send to " << sid << " failed " << endl;
		}
	    }
	    seqno++;

	    burst_index++;
	    next_burst = next_burst + burst_interval;
	}

	RealNet::DoWork(1);

again:
	Message *msg = 0;
	SID fromWhom;
	net->GetNextMessage(&fromWhom, &msg);

	if (msg && msg->GetType() == MSG_BLOB) {
	    MsgBlob *bmsg = (MsgBlob *)msg;
	    SenderInfoMapIter it = infoMap.find(fromWhom);
	    ASSERT(it != infoMap.end());
	    uint32 seqno = bmsg->nonce;
	    if (seqno > it->second->last_seqno) {
		uint32 num_dropped = seqno - it->second->last_seqno - 1;
		it->second->last_seqno = seqno;
		it->second->pkts_recved++;
		it->second->pkts_dropped += num_dropped;
	    } else if (seqno < it->second->last_seqno) {
		WARN << "OUT OF ORDER PACKET FROM: " << fromWhom 
		    << " prev=" << it->second->last_seqno
		    << " this=" << seqno << endl;
	    } else if (seqno == it->second->last_seqno) {
		WARN << "DUPLICATE PACKET FROM: " << fromWhom << endl;
	    }
	}

	if (msg) {
	    net->FreeMessage(msg);
	    goto again;
	}

	PERIODIC(epoch, now, {
		INFO << "IPEndPoint      \tdropped\trecved\ttotal\tdrop_rate" << endl;
		for (SenderInfoMapIter it = infoMap.begin(); 
		    it != infoMap.end(); it++) {

		uint32 total = it->second->pkts_dropped+it->second->pkts_recved;
		float rate = total > 0 ? (float)it->second->pkts_dropped/total : 0;

		INFO << it->first << "\t"
		<< it->second->pkts_dropped << "\t"
		<< it->second->pkts_recved << "\t"
		<< total << "\t"
		<< rate << endl;

		it->second->Reset();
		}

		Benchmark::print();
		Benchmark::restart();
		});
    }
}

int main(int argc, char **argv)
{
    InitializeColyseus(&argc, argv, options);

    if (argc < 3) {
	INFO << "usage: pkttest [options] <ipaddr>:<port> " 
	    << "[m nservers| <masterip>:<masterport>]"
	    << endl;
	exit(1);
    }

    self = IPEndPoint(argv[1]);
    ASSERT(self != SID_NONE);
    g_LocalSID = self;

    DBG_INIT((void *)&self);

    if (!strcmp(argv[2], "m")) {
	ismaster = true;

	ASSERT(argc >= 4);
	nservers = atoi(argv[3]);
	servers.insert(self);
	master = self;
    } else {
	ismaster = false;

	master = SID(argv[2]);
	ASSERT(master != SID_NONE);
    }

    RealNet *net = new RealNet(new SchedulerWithoutEventQueue (), self, false);
    net->StartListening(PROTO_TCP);

    if ( strstr(proto_str, "TCP") ) {
	proto = PROTO_TCP;
    } else if ( strstr(proto_str, "UDP") ) {
	proto = PROTO_UDP;
    } else if ( strstr(proto_str, "CBR") ) {
	proto = PROTO_CBR;
    } else {
	WARN << "UNKNOWN PROTOCOL: " << proto_str << endl;
	ASSERT(0);
    }

    if (proto != PROTO_TCP) {
	net->StartListening(proto);
    }

    ///////////////////////////////////////////////////////////////////////////

    started = 0;

    INFO << "Waiting to start..." << endl;

    if (ismaster) {
	ReadJoin(net);
	BroadcastStart(net);
    } else {
	WaitForStart(net);
    }

    for (SIDSetIter it = servers.begin(); it != servers.end(); it++) {
	if (*it == self) continue;
	infoMap[*it] = new SenderInfo();
    }

    INFO << self << ": Started..." << endl;

    while(true) {
	DoIt(net);
    }

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
