////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:nil -*- */

#include <Mercury.h>
#include <mercury/Scheduler.h>
#include <wan-env/RealNet.h>

#include <mercury/Sampling.h>
#include <rpc/MercRPC.h>
#include <rpc/MercuryNodeClientStub.h>
#include <mercury/Application.h>
#include <rpc/RealNetMarshaller.h>
#include <rpc/MercRPCDispatcher.h>
#include <rpc/Application_MercRPC.h>

#include "RPCTerminal.h"

//
// This is a dummy test client
//

/*
 * this just implements TimeNow ()
 */
class SchedulerWithoutEventQueue : public Scheduler {
    TimeVal m_Now;
public:
    SchedulerWithoutEventQueue () {}

    virtual void RaiseEvent (ref<SchedulerEvent> event, IPEndPoint&,
	    u_long millis) {}
    virtual void CancelEvent (ref<SchedulerEvent> event) {}
    virtual void ProcessTill(TimeVal&) {}
    virtual void ProcessFor (u_long millis) {}
    virtual void Reset () {}
    virtual TimeVal& TimeNow () {
	OS::GetCurrentTime(&m_Now);
	return m_Now;
    }

    void FireEvents () { ASSERT (false); }
};

struct DummyApplicationHandler : public Application {
    PubsubStore* GetPubsubStore(int) { return NULL; }
    bool IsLeaveJoinOK() {
	INFO << "()" << endl;
	return true;
    }

    void JoinBegin (const IPEndPoint& succ) {
	INFO << "(" << succ << ")" << endl;
    }
    void JoinEnd (const IPEndPoint& succ) {
	INFO << "(" << succ << ")" << endl;
    }
    void LeaveBegin () {
	INFO << "()" << endl;
    }
    void LeaveEnd () {
	INFO << "()" << endl;
    }
    void RangeExpanded (const NodeRange& oldrange, const NodeRange& newrange) {
	INFO << "(" << oldrange << "," << newrange << ")" << endl;
    }
    void RangeContracted (const NodeRange& oldrange, const NodeRange& newrange) {
	INFO << "(" << oldrange << "," << newrange << ")" << endl;
    }
    int EventRoute (Event * ev, const IPEndPoint& lastHop) {
	INFO << "(" << ev << "," << lastHop << ")" << endl;
	return 1;
    }
    int EventLinear (Event * ev, const IPEndPoint& lastHop) {
	INFO << "(" << ev << "," << lastHop << ")" << endl;
	return 1;
    }
    EventProcessType EventAtRendezvous (Event * ev, const IPEndPoint& lastHop, int nhops) {
	INFO << "(" << ev << "," << lastHop << "," << nhops << ")" << endl;
	return EV_NUKE;
    }
    void EventInterestMatch (const Event * ev, const Interest * in, const IPEndPoint& subscriber) {
	INFO << "(" << ev << "," << in << "," << subscriber << ")" << endl;
    }
    int InterestRoute (Interest * in, const IPEndPoint& lastHop) {
	INFO << "(" << in << "," << lastHop << ")" << endl;
	return 1;
    }
    int InterestLinear (Interest * in, const IPEndPoint& lastHop) {
	INFO << "(" << in << "," << lastHop << ")" << endl;
	return 1;
    }
    InterestProcessType InterestAtRendezvous (Interest * in, const IPEndPoint& lastHop) {
	INFO << "(" << in << "," << lastHop << ")" << endl;
	return IN_NUKE;
    }
};

struct DummySampler : public Sampler {
    float load;

    DummySampler(float load) : load(load) {}

    virtual const char *GetName () const {
	return "dummy_sampler";
    }
    virtual int GetLocalRadius () const {
	return 1;
    }
    virtual u_long GetSampleLifeTime () const {
	return 10000;
    }
    virtual int GetNumReportSamples () const {
	return 10;
    }
    virtual int GetRandomWalkInterval () const {
	return 5000;
    }
    virtual Metric* GetPointEstimate () {
	return new LoadMetric(load);
    }
    virtual Metric* MakeLocalEstimate (vector<Metric* >& samples) {
	float load = 0;
	for (uint32 i=0; i<samples.size(); i++) {
	    LoadMetric *m = dynamic_cast<LoadMetric *>(samples[i]);
	    if (!m) {
		WARN << "ignoring non-load metric" << endl;
		continue;
	    }
	    load += m->GetLoad();
	}
	return new LoadMetric(load/samples.size());
    }
};

RealNetMarshaller *m;
MercuryNodeClientStub *n;
Sampler *s;

uint32 rpcPort;
uint32 termPort;
IPEndPoint rpcServer;
char rpcServerStr[255];

OptionType opts[] = {
    // not really needed; realnet just wants some appid, 
    // we don't actually listen
    { '/', "rpc-port", OPT_INT,
	"port that the rpc server listens on",
	&rpcPort, "14000", NULL },
	{ '/', "rpc-server", OPT_STR,
	    "addr:port of merc rpc server to talk to",
	    &rpcServerStr, "localhost:12000", NULL },
	    { '/', "term-port", OPT_INT,
		"port that the client listens on for commands",
		&termPort, "16000", NULL },
};

int main(int argc, char **argv) {
    InitializeMercury(&argc, argv, opts);
    MercRPC_RegisterTypes();
    MercRPC_SetIsRPCServer(false);

    rpcServer = IPEndPoint(rpcServerStr);
    if (rpcServer == SID_NONE) {
	_die("bad rpc server '%s'", rpcServerStr);
    }

    // set rpc listen address (not used, just an appid)
    struct hostent *entry = gethostbyname(g_Preferences.hostname);
    if (entry)
	g_RPCAddr.m_IP = ((struct in_addr *) entry->h_addr)->s_addr;
    else
	g_RPCAddr.m_IP = inet_addr(g_Preferences.hostname);
    g_RPCAddr.m_Port = rpcPort; 

    DBG_INIT(&g_RPCAddr);

    // start network
    SchedulerWithoutEventQueue *sched = new SchedulerWithoutEventQueue();
    RealNet *net = new RealNet(sched, g_RPCAddr);
    net->StartListening(PROTO_TCP_PASSIVE);

    // startup rpc marshaller
    m = new RealNetMarshaller(net, PROTO_TCP_PASSIVE, rpcServer);

    // create MercuryNode stub
    MercuryNodeClientStub::Init(m);
    n = MercuryNodeClientStub::GetInstance();
    n->RegisterApplication(new DummyApplicationHandler());

    // start the terminal
    RPCTerminal term(termPort, n);
    term.Start();

    // wait for merc to start
    INFO << "waiting for merc to start..." << endl;
    while(! n->AllJoined() )
	sleep(1);
    INFO << "merc is joined!" << endl;

    vector<Constraint> ranges = n->GetHubRanges ();
    for (uint32 i=0; i<ranges.size(); i++) {
	INFO << "hub " << i << ": [" << ranges[i].GetMin()
	    << "," << ranges[i].GetMax() << "]" << endl;
    }

    // try adding a sampler
    //s = new DummySampler(100);
    //n->RegisterSampler(0, s);

    // loop on processing
    //int i = 0;
    while (true) {
	//INFO << i++ << endl; 
	RealNet::DoWork(10);
	MercRPCClientDispatcher::GetInstance()->ProcessRPCs();
    }

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
