////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/***************************************************************************

  RPCTerminal.cpp

  Debugging terminal that can be used to log into the manager and inspect
  stuff online.

begin		   : October 10, 2003
copyright	   : (C) 2003 Jeffrey Pang           ( jeffpang@cs.cmu.edu )

 ***************************************************************************/

#include <Mercury.h>
//#include <rpc/RPCableEvent.h>
//#include <rpc/RPCableInterest.h>
#include <rpc/MercuryNodeClientStub.h>
#include "RPCTerminal.h"
#include <util/Benchmark.h>

typedef map<string, guid_t> StrGUIDMap;
typedef StrGUIDMap::iterator StrGUIDMapIter;
typedef map<guid_t, string, less_GUID> StringMap;
typedef StringMap::iterator StringMapIter;

static char *extract_str(char *buf, int *len);
static char **extract_args(char *buf);
static void CheckTimeouts(struct timeval *now);
static void Cleanup(int sig);

///////////////////////////////////////////////////////////////////////////////

static InputHandler g_DefaultHandlers[] = {
    { "HELP",   "H", RPCTerminalWorker::handleHelp,   "",
	"Show this help message" },
	{ "SENDPUB","P", RPCTerminalWorker::handleSendPub, "<HUBID> <ID>",
	    "Send a publication" },
	    { "SENDSUB","S", RPCTerminalWorker::handleSendSub, "<HUBID> <START> <END>",
		"Send a subscription" },

		{ "QUIT","Q", RPCTerminalWorker::handleQuit, "",
		    "Close connection" },
		    { NULL, NULL, NULL, NULL }
};

static InputHandlers  g_Handlers;
static MercuryNodeClientStub *g_Node;

///////////////////////////////////////////////////////////////////////////////

RPCTerminal::RPCTerminal(uint16 port, MercuryNodeClientStub *node) 
{ 
    g_Node = node;

    try {
	m_LocalSock = ServerSocket(port);
    } catch (SocketException ex) {
	_die("can't create socket: %s", ex.description().c_str() );
    }

    InputHandler *ih = g_DefaultHandlers;
    while (ih->cmd) {
	g_Handlers.push_back(ih);
	++ih;
    }
    //manager->m_Game->InstallRPCTerminalHandlers(&g_Handlers);

    for (int i=0; i<THREAD_POOL_SIZE; i++) {
	pool[i] = NULL;
    }

    DBG << "RPCTerminal started listening on port " << port << endl;
}

RPCTerminal::~RPCTerminal()
{ 
    m_LocalSock.Close();
}

void RPCTerminal::Run()
{
    while (1) {
	ServerSocket sock;

	DBG << "accepting new connections..." << endl;

	try {
	    m_LocalSock.Accept(sock);
	} catch (SocketException ex) {
	    WARN << "accept on local socket failed: " 
		<< ex.description() << "\n";
	    continue;
	}
	DBG << "accepted new client on socket: " << sock.GetSocket() << endl;

	RPCTerminalWorker *worker = NULL;

	// find a thread from the thread pool
	for (int i=0; i<THREAD_POOL_SIZE; i++) {
	    if (pool[i] != NULL && pool[i]->Finished()) {
		delete pool[i];
		pool[i] = NULL;
	    }

	    if (pool[i] == NULL) {
		pool[i] = new RPCTerminalWorker(sock);
		worker = pool[i];
		break;
	    }
	}
	if (worker == NULL) {
	    // not found! -- kill the connection
	    // ignore errors here -- we're killing this client anyway
	    DBG << "too many clients connected, closing socket " 
		<< sock.GetSocket() << "\n";
	    sock << RESP_TOO_MANY_CLIENTS;
	    sock.Close();
	    continue;
	}

	DBG << "starting worker thread on socket " << sock.GetSocket() << "\n";
	worker->Start();
    }
}

RPCTerminalWorker::RPCTerminalWorker(ServerSocket& sock) : m_Sock(sock)
{

}

RPCTerminalWorker::~RPCTerminalWorker()
{
    ASSERT(Finished());
}

void RPCTerminalWorker::Run()
{
    char msg[MAX_MSG_SIZE];
    int pos = 0;

    /*
    // some very minimal security...
    if (strcmp(ManagerParams::TERMINAL_PASSWORD, "")) {
    string passwd("");
    m_Sock << "Password: ";
    do {
    // XXX: should add a timeout here
    int ret = read( m_Sock.GetSocket(), msg, MAX_MSG_SIZE-1 );
    if (ret == 0) break;
    msg[ret] = '\0';
    passwd.append( msg );
    } while (( passwd.length() < 1 || 
    passwd[passwd.length()-1] != '\n' ) &&
    // limit size of passwd to prevent "DoS" type attack
    ( passwd.length() < MAX_MSG_SIZE ));

    while ( passwd.length() > 0 &&
    (passwd[passwd.length()-1] == '\n' ||
    passwd[passwd.length()-1] == '\r') ) {
    passwd = passwd.substr(0,passwd.length()-1);
    }
    if (strcmp(passwd.c_str(), ManagerParams::TERMINAL_PASSWORD)) {
    m_Sock << "Access denied.\n";
    m_Sock.Close();
    return;
    }
    }
     */

    m_Sock << "RPC RPCTerminal." << "\n";

    while (1) {
	int ret = read(m_Sock.GetSocket(), &msg[pos], MAX_MSG_SIZE-pos);
	if (ret == 0) {
	    DBG << "client closed Socket " << m_Sock.GetSocket() << "\n";
	    // EOF
	    break;
	} else if (ret < 0) {
	    if (errno == EINTR) {
		// interrupted, try again...
		continue;
	    } else {
		// some other error, don't really want to recover
		WARN << "read error on client Socket "
		    << m_Sock.GetSocket() << ": " 
		    << strerror(errno) << "\n";
		break;
	    }
	} else {
	    bool quit = false;

	    pos += ret;
	    while (1) {
		char *str = extract_str(msg, &pos);
		if (str == NULL)
		    break;

		DBG << "req: " << str << "\n";

		char **args = extract_args(str);
		char *res;

		m_Sock << "==BEGIN_RESP" << "\n";
		if (! args[0] || args[0] == '\0') {
		    m_Sock << "ERROR: no command provided" << "\n";
		} else {
		    bool done = false;

		    for (int i=0; i < (int)g_Handlers.size(); i++) {
			if ( !strcasecmp(args[0], g_Handlers[i]->cmd) ||
				!strcasecmp(args[0], g_Handlers[i]->abbrev) ) {

			    stringstream out;

			    //g_Manager->Lock();
			    g_Handlers[i]->handler(out, args);
			    //g_Manager->Unlock();

			    m_Sock << out.str();

			    done = true;
			    break;
			}
		    }

		    if (!done) {
			m_Sock << "ERROR: unknown command: " 
			    << args[0] << "\n";
		    }
		}
		m_Sock << "==END_RESP" << "\n";

		// special case for exiting
		if (args[0] &&
			(!strcasecmp(args[0], "QUIT") ||
			 !strcasecmp(args[0], "Q"))) {
		    quit = true;
		}
	    }

	    if (quit) break;
	}
    }

    m_Sock.Close();
}


///////////////////////////////////////////////////////////////////////////////

#define NOT_ENOUGH_ARGS(x) out << "ERROR: need " << x << " arguments" << "\n"
#define DOES_NOT_EXIST(x) out << "ERROR: object does not exist" << "\n";

void RPCTerminalWorker::handleHelp(ostream& out, char **args) {
    out << "Definitions:" << "\n";
    out << "  <ID>       = a name you gave an object or its GUID" << "\n";
    out << "  <attr_key> = a string (no spaces) for an attribute" << "\n";
    out << "  <attr_val> = a string <type>:<value> string for value" << "\n";
    out << "  <type>     = {int,char,byte,real,string,guid,ID}" << "\n";
    out << "  <guid>     = <ip>:<port>:<localOID>, where <ip> is in dotted-tail format" << "\n";
    out << "  <op>       = {==,!=,<,<=,>,>=,*}, where * means 'any'" << "\n";

    out << "\n" << "Commands:" << "\n";

    for (int i=0; i < (int)g_Handlers.size(); i++) {
	out << "  [" << g_Handlers[i]->abbrev << "] "
	    << g_Handlers[i]->cmd << " " 
	    << g_Handlers[i]->usage  << "\n";
	out << "    " << g_Handlers[i]->help << "\n";
    }
}

void RPCTerminalWorker::handleSendPub(ostream& out, char **args)
{
    if (args[1] == NULL || args[2] == NULL) {
	NOT_ENOUGH_ARGS(2);
	return;
    }

    uint32 hubid = atoi(args[1]);
    MercuryID id(args[2], 10);

    MercuryEvent ev;
    Constraint cons(hubid, id, id);
    ev.AddConstraint(cons);

    g_Node->SendEvent(&ev);

    return;
}

void RPCTerminalWorker::handleSendSub(ostream& out, char **args)
{
    // xxx todo
    out << "ERROR: XXX TODO" << "\n";
}


void RPCTerminalWorker::handleQuit(ostream& out, char **args)
{
    out << "OK: closing connection\n";
    return;
}

///////////////////////////////////////////////////////////////////////////////

guid_t RPCTerminalWorker::parse_guid(char *str)
{
    char *ip   = strtok(str, ":");
    char *port = strtok(NULL, ":");
    char *localOID = strtok(NULL, ":");
    if (ip == NULL || port == NULL || localOID == NULL) {
	return GUID_NONE;
    }
    uint32 _ip   = inet_addr(ip);
    uint16 _port = (uint16)atoi(port);
    uint32 _localOID = (uint32)atoi(localOID);
    if (_ip == INADDR_NONE || _port == 0) {
	return GUID_NONE;
    }

    guid_t guid(_ip, _port, _localOID);

    return guid;
}

sid_t RPCTerminalWorker::parse_sid(char *str)
{
    char *ip   = strtok(str, ":");
    char *port = strtok(NULL, ":");
    if (ip == NULL || port == NULL) {
	return SID_NONE;
    }
    uint32 _ip   = inet_addr(ip);
    uint16 _port = (uint16)atoi(port);
    if (_ip == INADDR_NONE || _port == 0) {
	return SID_NONE;
    }

    sid_t sid(_ip, _port);

    return sid;
}

///////////////////////////////////////////////////////////////////////////////

static char str_buf[512];
static char *arg_buf[512];

char *extract_str(char *buf, int *len) 
{
    int i;
    bool found = 0;

    // parse one command terminated by a newline into the static buf
    for (i=0; i<*len; i++) {
	str_buf[i] = buf[i];
	if (str_buf[i] == '\n') {
	    str_buf[i] = '\0';
	    // if client writes \r\n, then need to get rid of two chars
	    if (str_buf[i-1] == '\r')
		str_buf[i-1] = '\0';
	    found = 1;
	    i++;
	    break;
	}
    }

    // if there is stuff left in the buffer, move it to the front
    int j = 0;
    for ( ; i<*len; i++, j++) {
	buf[j] = buf[i];
    }
    *len = j;

    if (found) {
	return str_buf;
    } else {
	return NULL;
    }
}

char **extract_args(char *buf) {
    int i;
    char *last = buf;
    int index = 0;

    // separate command string into an array of string args
    for (i=0; i<512; i++) {
	bool end = (buf[i] == '\0');

	if (end || isspace(buf[i])) {
	    buf[i] = '\0';
	    if (last < buf+i) {
		arg_buf[index] = last;
		++index;
		if (index > 512)
		    break;
	    }
	    last = buf + i + 1;
	}

	if (end) break;
    }
    arg_buf[index] = NULL;

    return arg_buf;
}

///////////////////////////////////////////////////////////////////////////////
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
