/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include "Barrier.h"
#include <mercury/MercMessage.h>

Barrier::Barrier(const IPEndPoint& master, 
		const IPEndPoint& me,
		uint32 nonce, uint32 count) :
master(master), nonce(nonce), count(count), isMaster(master == me),
isDone(false), signal_repeater(NULL), waitcb(NULL)
{
	net = new ARealNet(me);
	net->StartListening(PROTO_TCP);

	net->RegisterGetNextMessageCB(wrap(this, &Barrier::ProcessMessage));
}

Barrier::~Barrier()
{
	if (signal_repeater) {
		timecb_remove(signal_repeater);
		signal_repeater = NULL;
	}
	net->StopListening();
	delete net;
}

void Barrier::signal()
{
	if (isMaster) {
		ProcessSignal(net->GetAppID());
	} else {
		SendSignal();
	}
}

void Barrier::wait(callback<void,void>::ref cb)
{
	if (isDone) {
		delaycb(0, cb);
		return;
	}

	waitcb = cb;
}

void Barrier::SendSignal()
{
	MsgPing sig;
	sig.sender = net->GetAppID();
	sig.pingNonce = nonce;

	net->SendMessage(&sig, &master, PROTO_TCP, NULL);

	// must repeat signal in case master is not up yet
	signal_repeater = delaycb(5, wrap(this, &Barrier::SendSignal));
}

void Barrier::ProcessSignal(const IPEndPoint& from)
{
	ASSERT(isMaster);
	done.insert(from);

	//INFO << "from: " << from << endl;

	if (done.size() >= count) {
		//INFO << "done!" << endl;

		// signal everyone
		MsgPing pong;
		pong.sender = net->GetAppID();
		pong.pingNonce = nonce;

		for (SIDSet::iterator it = done.begin(); it != done.end(); it++) {
			if (*it == master)
				continue;
			net->SendMessage(&pong, (IPEndPoint *)&*it, PROTO_TCP, NULL);
		}

		// signal me
		isDone = true;
		if (waitcb != NULL)
			(*waitcb)();
	}
}

void Barrier::ProcessMessage(ConnStatusType t,IPEndPoint from,MercMessage *msg)
{
	if (!msg)
		return;

	if (!isDone) {
		if (msg->GetType() == MSG_PING) {
			MsgPing *ping = dynamic_cast<MsgPing *>(msg);
			if (ping->pingNonce == nonce) {
				if (isMaster) {
					// got a signal message
					ProcessSignal(from);
				} else {
					// got a done message
					if (signal_repeater) {
						timecb_remove(signal_repeater);
						signal_repeater = NULL;
					}
					isDone = true;
					if (waitcb != NULL)
						(*waitcb)();
				}
			} else {
				WARN << "Got unknown nonce: " << ping->pingNonce << endl;
			}
		} else {
			WARN << "Got unknown message type: " << msg << endl;
		}
	}

	net->FreeMessage(msg);
}

