/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include "AsyncUtil.h"
#include <util/debug.h>
#include <util/Benchmark.h>
#include <rpc/MercuryNodeClientStub.h>
#include <wan-env/RealNet.h>
#include <map>
#include <list>

typedef pair<int, selop> pic;
ostream& operator<<(ostream& out, const pic& v) {
	out << "(" << v.first << "," << (v.second==selread?"r":"w") << ")";
	return out;
}

struct fdcb_t {
	pic key;
	callback<void>::ref cb;
	fdcb_t(const pic& key, callback<void>::ref cb) : key(key), cb(cb) {}
};

struct less_pic {
	bool operator()(const pic& a, const pic& b) const {
		if (a.first < b.first)
			return true;
		if (a.first > b.first)
			return false;
		return a.second < b.second;
	}
};

typedef map< pic, list<fdcb_t *> *, less_pic > fdcb_map_t;
fdcb_map_t fdcb_map;

static void fdcb_once_cb(pic key)
{
	DBG << "fdcb_once_cb: " << key << endl;

	fdcb_map_t::iterator it = fdcb_map.find(key);
	if (it == fdcb_map.end()) {
		WARN << "no callbacks in fdcb_map for: " << key << endl;
		ASSERT(0);
		return;
	}
	list<fdcb_t *> *s = it->second;
	if (s->size() == 0) {
		WARN << "empty callback list in fdcb_map for: " << key << endl;
		return;
	}
	fdcb_t *k = s->front();
	s->pop_front();
	if (s->size() == 0) {
		// nothing left, unregister the cb
		DBG << "nothing left, unregistering cb" << endl;
		delete s;
		fdcb_map.erase(it);
		fdcb(key.first, key.second, NULL);
	}

	(*k->cb)();
	delete k;
	return;
}

fdcb_t *fdcb_once(int socket, selop operation, callback<void>::ref cb)
{
	pic key(socket, operation);

	DBG << "called: " << key << endl;

	fdcb_map_t::iterator it = fdcb_map.find(key);
	list<fdcb_t *> *s;
	if (it == fdcb_map.end()) {
		s = new list<fdcb_t *>();
		fdcb_map[key] = s;
		fdcb(socket, operation, wrap(&fdcb_once_cb, key));
	} else {
		s = it->second;
	}
	fdcb_t *k = new fdcb_t(key, cb);
	s->push_back(k);
	return k;
}

void fdcb_once_remove(fdcb_t *k)
{
	pic key = k->key;

	DBG << "called: " << key << endl;

	fdcb_map_t::iterator it = fdcb_map.find(key);
	if (it == fdcb_map.end()) {
		return;
	}
	list<fdcb_t *> *s = it->second;
	for (list<fdcb_t *>::iterator i = s->begin(); i != s->end(); i++) {
		if (*i == k) {
			s->erase(i);
			delete k;
			break;
		}
	}
	if (s->size() == 0) {
		// nothing left, unregister the cb
		delete s;
		fdcb_map.erase(it);
		fdcb(key.first, key.second, NULL);
	}
	return;
}

///////////////////////////////////////////////////////////////////////////////

static MercuryNodeInterface *merc_node;
static uint32 realnet_maxfd;
static fd_set realnet_fds;

static void realnet_dowork();
static void register_realnet_fdcb();
static void merc_dowork();

static void realnet_dowork()
{
	TimeVal now;
	OS::GetCurrentTime(&now);                                           

	START(realnet_dowork::RealNet);
	RealNet::DoWork(20 /* xxx how long ? */);
	STOP(realnet_dowork::RealNet);

	// process packets immediately
	START(realnet_dowork::Merc);
	merc_node->DoWork(MERCURY_DOWORK_INTERVAL);
	STOP(realnet_dowork::Merc);

	register_realnet_fdcb();
}

static void register_realnet_fdcb()
{
	uint32 old_maxfd = realnet_maxfd;
	fd_set old_fds   = realnet_fds;

	FD_ZERO(&realnet_fds);
	realnet_maxfd = RealNet::FillReadSet(&realnet_fds);

	for (uint32 i=0; i<=MAX(old_maxfd,realnet_maxfd); i++) {
		if (FD_ISSET(i, &old_fds) && !FD_ISSET(i, &realnet_fds)) {
			// unregister fd
			fdcb(i, selread, NULL);
		}
		if (!FD_ISSET(i, &old_fds) && FD_ISSET(i, &realnet_fds)) {
			// register fd
			fdcb(i, selread, wrap(&realnet_dowork));
		}
	}
}

static void merc_dowork()
{
	START(merc_dowork::Merc);
	merc_node->DoWork(10 /* xxx how long? */);
	STOP(merc_dowork::Merc);

	// mercury might close some connections, in which case those fd's become
	// invalid and we should not select on them anymore
	//realnet_maxfd = RealNet::FillReadSet(&realnet_fds);
	register_realnet_fdcb();

	delaycb(MERCURY_DOWORK_INTERVAL/MSEC_IN_SEC, // sec
			(MERCURY_DOWORK_INTERVAL%MSEC_IN_SEC)*1000*1000, // nsec
			wrap(&merc_dowork));
}

void amerc_init(MercuryNodeInterface *node)
{
	ASSERT(node);
	merc_node = node;
	realnet_maxfd = 0;
	FD_ZERO(&realnet_fds);

	delaycb (MERCURY_DOWORK_INTERVAL/MSEC_IN_SEC, // sec
			(MERCURY_DOWORK_INTERVAL%MSEC_IN_SEC)*1000*1000, // nsec
			wrap(&merc_dowork));

	register_realnet_fdcb();
}
