/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:nil -*- */

#include <util/Utils.h>
#include <util/OS.h>
#include <util/Environment.h>
#include "ARealNet.h"
#include "AUDPTransport.h"

int AUDPTransport::MAX_UDP_MSGSIZE = OS::GetMaxDatagramSize();

void AUDPTransport::StartListening()
{
    struct sockaddr_in	server_address;

    int err = ARealNet::CreateSocket(&m_ListenSocket, (int) PROTO_UDP ); 
    if (err < 0) {
	perror("socket");
	ASSERT(0);
    }

    memset((char *)&server_address, 0, sizeof(server_address));

    server_address.sin_family = AF_INET;
    server_address.sin_addr.s_addr = m_ID.m_IP; // already in network order
    server_address.sin_port = htons(m_ID.m_Port);

    if (bind(m_ListenSocket, (struct sockaddr *)&server_address, 
		sizeof(server_address)) < 0 ) {
	perror("bind");
	_die ("could not bind server socket to port [%d]\n", m_ID.m_Port);
    }

    RegisterCBs();

    DB(1) << "Started [PROTO_UDP] server at port " 
	<< m_ID.m_Port << " successfully..." << endl;
    return;
}

void AUDPTransport::StopListening()
{
    if (m_ListenSocket > 0)
	OS::CloseSocket(m_ListenSocket);

    UnregisterCBs();
    _ClearConnections();
}

void AUDPTransport::RegisterCBs()
{
    DBG << "listening on " << m_ListenSocket << endl;
    fdcb(m_ListenSocket, selread, 
	    wrap(this, &AUDPTransport::ReadyConnectionCB));
    lazycb_t *l = lazycb(5, wrap(static_cast<ATransport *>(this), 
		&ATransport::_CleanupConnections));
    m_LazyCBs.push_back(l);
}

void AUDPTransport::UnregisterCBs()
{
    fdcb(m_ListenSocket, selread, NULL);
    for (list<lazycb_t *>::iterator it = m_LazyCBs.begin();
	    it != m_LazyCBs.end(); it++) {
	lazycb_remove(*it);
    }
    m_LazyCBs.clear();
}

AUDPConnection *AUDPTransport::CreateConnection(Socket sock, 
	IPEndPoint *otherEnd)
{
    return new AUDPConnection(this, sock, otherEnd);
}

void AUDPTransport::AddConnection(AConnection *connection)
{
    Lock();
    m_ConnectionList.push_back(connection);
    // I know the other end point here, since I am actively connecting. 
    // So this is the ONLY address I will ever use to identify "this" 
    // particular connection 
    m_AppConnHash.Insert(connection->GetAppPeerAddress(), connection);
    Unlock();
}

void AUDPTransport::RemoveConnection(AConnection *connection)
{
    Lock();
    m_AppConnHash.Flush( connection->GetAppPeerAddress() );
    m_ConnectionList.remove(connection);
    Unlock();
}

void AUDPTransport::GetConnection(IPEndPoint *toWhom, GetConnectionCB::ref cb)
{
    Lock();
    AUDPConnection *connection = 
	(AUDPConnection *)m_AppConnHash.Lookup(toWhom);
    Unlock();

    if (!connection || 
	    connection->GetStatus() == CONN_CLOSED || 
	    connection->GetStatus() == CONN_ERROR) {

	// remove old connection from hash and list
	if (connection)
	    RemoveConnection(connection);

	// create new connection and insert into hash and list
	connection = CreateConnection(m_ListenSocket, toWhom);

	DB(20) << "new connection to: " << toWhom << endl;
	AddConnection(connection);
    }

    (*cb)(connection, 0);
}

void AUDPTransport::ReadyConnectionCB()
{
    ASSERT ( RealNet::IsDataWaiting(m_ListenSocket) );

    ConnStatusType ret = CONN_NOMSG;

    IPEndPoint fromWhom;
    Packet *pkt = new Packet(MAX_UDP_MSGSIZE);

    int len = ARealNet::ReadDatagram(m_ListenSocket, 
	    &fromWhom, 
	    pkt->GetBuffer (),
	    pkt->GetMaxSize ());

    pkt->ResetBufPosition ();
    pkt->IncrBufPosition (len);

    if (len < 0) {
	WARN << "UDP Socket on port " << m_ID.m_Port
	    << " error: " << strerror(errno) << endl;
	delete pkt;
	return;
    }

    Lock();
    AUDPConnection *conn = (AUDPConnection *)m_AppConnHash.Lookup(&fromWhom);
    Unlock();

    if (conn == NULL) {
	// XXX -- currently we assume that the app-level ID is the
	// ipaddr:port that this packet was sent from...
	conn = CreateConnection(m_ListenSocket, &fromWhom);
	conn->SetStatus(CONN_NEWINCOMING);
	AddConnection(conn);

	DBG << "new connection from: " << fromWhom << endl;
    }

    // this is useless in the async version unless the app doesn't
    // process packets when calledback
    if (conn->Size() > APP_QUEUE_SIZE) {
	switch (QUEUING_DISCIPLINE) {
	case Q_DROPTAIL: {
	    delete pkt;
	    return;
	}
	case Q_DROPHEAD: {
	    ref<PacketInfo> info = conn->Pop();
	    delete info->pkt;
	    break;
	}
	default:
	    WARN << "Unknown queuing discipline: "
		<< QUEUING_DISCIPLINE << endl;
	    ASSERT(0);
	}
    }

    DB(20) << "servicing: " << *conn->GetAppPeerAddress() << endl;

    TimeVal timestamp = OS::GetSockTimeStamp(m_ListenSocket);
    conn->Insert(pkt, timestamp);

    GetNetwork()->ProcessMessageCB(this, conn->GetStatus(), conn);  
}

void AUDPTransport::CloseConnection(IPEndPoint *target) {
    Lock();
    AConnection *connection = m_AppConnHash.Lookup(target);
    Unlock();
    if (!connection)
	return;

    connection->SetStatus(CONN_CLOSED);
}
