/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:nil -*- */

#include <async.h>
#include <netinet/tcp.h>
#include <util/Utils.h>
#include <util/OS.h>
#include <awan-env/tcpconnect.h>
#include <util/Environment.h>
#include <awan-env/ATCPTransport.h>
#include <signal.h>

void ATCPTransport::StartListening()
{
    struct sockaddr_in	server_address;
    Socket sock;

    // catch SIGPIPE so that error gets returned on write system calls
    if ( signal(SIGPIPE, SIG_IGN) == SIG_ERR ) {
	WARN << "could not ignore SIGPIPE handler" << endl;
	ASSERT(0);
    }

    int err = GetNetwork()->CreateSocket(&sock, (int) PROTO_TCP);
    if (err < 0) {
	ASSERT(0);
    }
    m_ListenSocket = sock;

    // make re-starting the server easier
    int restart = 1;
    if (OS::SetSockOpt(sock, SOL_SOCKET, SO_REUSEADDR, 
		(char *) &restart, sizeof(int)) < 0 )
	WARN << "StartListening: could not configure server socket properly: " 
	    << strerror(errno) << endl;

    // turn off nagle for real-time sending
    int nonagle = 1;
    if (OS::SetSockOpt(sock, IPPROTO_TCP, TCP_NODELAY, 
		(char *) &nonagle, sizeof(int)) < 0 )
	WARN << "StartListening: could not configure server socket properly: " 
	    << strerror(errno) << endl;

    memset((char *)&server_address, 0, sizeof(server_address));

    server_address.sin_family = AF_INET;
    server_address.sin_addr.s_addr = m_ID.m_IP; // already in network order
    server_address.sin_port = htons(m_ID.m_Port);

    int retries = 3;
    while ( bind(sock, (struct sockaddr *)&server_address, 
		sizeof(server_address)) < 0 && retries-- > 0) {
	perror("bind");
	WARN << "StartListening: bind error for reliable port ["
	    << m_ID.m_Port << "]" << endl;
	sleep(3);
    }
    if (retries <= 0) {
	ASSERT(0);
    }

    // get ready to accept connection requests
    if (listen(sock, MAX_PENDING_REQUESTS) < 0) {
	perror("listen");
	WARN << " could not listen at " << m_ID << endl;
	ASSERT(0);
    }
    RegisterCBs();

    DB(1) << "Started [PROTO_TCP] server at port " 
	<< m_ID.m_Port << " successfully..." << endl;
    return;
}

void ATCPTransport::StopListening()
{
    if (m_ListenSocket > 0) {
	UnregisterCBs();
	OS::CloseSocket(m_ListenSocket);
    }
    m_ListenSocket = -1;

    _ClearConnections();
}

void ATCPTransport::RegisterCBs()
{
    DBG << "listening on " << m_ListenSocket << endl;
    fdcb(m_ListenSocket, selread, 
	    wrap(this, &ATCPTransport::RegisterNewTCPConnectionCB));
    lazycb_t *l = lazycb(5, wrap(static_cast<ATransport *>(this), &ATransport::_CleanupConnections));
    m_LazyCBs.push_back(l);
}

void  ATCPTransport::UnregisterCBs()
{
    fdcb(m_ListenSocket, selread, NULL);
    for (list<lazycb_t *>::iterator it = m_LazyCBs.begin();
	    it != m_LazyCBs.end(); it++) {
	lazycb_remove(*it);
    }
    m_LazyCBs.clear();
}

///////////////////////////////////////////////////////////////////////////////

void ATCPTransport::GetConnection(IPEndPoint *toWhom, GetConnectionCB::ref cb)
{
    DBG << "called" << endl;

    Lock();
    AConnection *connection = m_AppConnHash.Lookup(toWhom);
    Unlock();

    if (!connection || 
	    connection->GetStatus() == CONN_CLOSED || 
	    connection->GetStatus() == CONN_CLOSING ||
	    connection->GetStatus() == CONN_ERROR) {

	// see if the connection is already pending
	PendingConnectMapIter it = m_PendingConnects.find(*toWhom);
	if (it != m_PendingConnects.end()) {
	    // already waiting for this connect, add the cb to the waiting list
	    it->second->push_back(cb);
	    return;
	} else {
	    // mark it as pending
	    list<GetConnectionCB::ref> *l = new list<GetConnectionCB::ref>();
	    l->push_back(cb);
	    m_PendingConnects[*toWhom] = l;
	}

	// remove old connection from hash and list
	m_AppConnHash.Flush(toWhom);

	// too many connections open, so close some connections (in lru order)
	if (m_MaxOpenConnections > 0) {
	    while (m_AppConnHash.size() > m_MaxOpenConnections) {
		TimeVal min;
		IPEndPoint argmin = SID_NONE;
		// xxx this is inefficient if we have lots of connections
		for (AConnectionHash::iterator it = m_AppConnHash.begin();
			it != m_AppConnHash.end(); it++) {
		    TimeVal last = ((ATCPConnection *)it->second)->GetLastAccess();
		    if (argmin == SID_NONE || min > last) {
			min = last;
			argmin = it->first;
		    }
		}
		ASSERT(argmin != SID_NONE);
		ActiveCloseConnection((ATCPConnection *)m_AppConnHash.Lookup(&argmin));
	    }
	}

	/* Jeff: This will be garbage collected later, no need to delete now
	   if (connection)
	   m_ConnectionList.remove(connection);
	   delete connection; // xxx todo reuse this
	 */

	DoConnect(*toWhom, cb);
    } else {
	(*cb)(connection, 0);
    }
}

void ATCPTransport::DoConnect(IPEndPoint otherEnd,
	bool init, int maxTrials, Socket sock) {
    DBG << "called: " << otherEnd << endl;

    if (init || sock < 0) {
	if (maxTrials <= 0) {
	    WARN << "could not connect to (" 
		<< otherEnd << ") giving up..." << endl;
	    DoConnectReturnCB(otherEnd, NULL, sock);
	    return;
	}

	if (init)
	    DB(4) << "connecting to: " << otherEnd.ToString()
		<< " (TCP:" << m_ID.m_Port << ")..." << endl;
	else
	    DB(4) << maxTrials << ".." << endl; 

	// try to connect or try again (xxx wrap this in OS::)
	in_addr addr;
	addr.s_addr = otherEnd.GetIP();

	// bind our own socket to be sure we get the source addr we want
	int sock;
	struct sockaddr_in srcaddr;

	GetNetwork()->CreateSocket(&sock, (int) PROTO_TCP);

	srcaddr.sin_family = AF_INET;
	srcaddr.sin_addr.s_addr = m_ID.m_IP;
	srcaddr.sin_port = 0; // let kernel choose port

	if (bind(sock, (struct sockaddr *)&srcaddr, sizeof(srcaddr)) < 0) {
	    WARN << "bind failed in connecting to (" 
		<< otherEnd << ") giving up..." << endl;
	    DoConnectReturnCB(otherEnd, NULL, sock);
	    return;
	}

	tcpconnect(sock, addr, otherEnd.GetPort(), 
		wrap(this, &ATCPTransport::DoConnect, otherEnd,
		    false, maxTrials-1));
    } else {
	// connected successfully!

	// The first thing to transmit is our AppLevel ID so that the remote
	// host can identify this connection as a target of return packets

	_tcp_addr a(m_ID);

	//DBG << "write" << endl;
	//DumpBuffer((byte *)&a, sizeof(a));
	ARealNet::Write(sock, a.buf, 6, 
		wrap(this, &ATCPTransport::DoConnectCB,
		    otherEnd, sock));
    }
}

void ATCPTransport::DoConnectCB(IPEndPoint otherEnd, Socket sock, int err) {
    DBG << "called" << endl;

    if (err < 0) {
	// failed
	DB(4) << "Connect to " << otherEnd << " failed" << endl; 

	OS::CloseSocket(sock);
	DoConnectReturnCB(otherEnd, NULL, err);
	return;
    }

    DB(4) << "Connect to " << otherEnd << " succeeded (" << sock << ")!" << endl; 

    // so that dead tcp connections don't fill up the file descriptor table
    int restart = 1;
    if (OS::SetSockOpt(sock, SOL_SOCKET, SO_REUSEADDR, 
		(char *) &restart, sizeof(int)) < 0 )
	WARN << "could not configure server socket properly: " 
	    << strerror(errno) << endl;

    // turn off nagle for real-time sending
    int nonagle = 1;
    if (OS::SetSockOpt(sock, IPPROTO_TCP, TCP_NODELAY, 
		(char *) &nonagle, sizeof(int)) < 0 )
	WARN << "could not configure server socket properly: " 
	    << strerror(errno) << endl;

    // create new connection and insert into hash and list
    ATCPConnection *connection = new ATCPConnection(this, sock, &otherEnd);

    Lock();
    m_ConnectionList.push_back(connection);
    // I know the other end point here, since I am actively connecting. 
    // So this is the ONLY address I will ever use to identify "this" 
    // particular connection 
    m_AppConnHash.Insert(connection->GetAppPeerAddress(), connection);
    Unlock();

    // register to read from it
    fdcb(sock, selread, wrap(this, &ATCPTransport::ReadyConnectionCB, connection));

    // callback the app
    DoConnectReturnCB(otherEnd, connection, err);
}

void ATCPTransport::DoConnectReturnCB(IPEndPoint otherEnd,
	AConnection *conn, int err)
{
    PendingConnectMapIter it = m_PendingConnects.find(otherEnd);
    ASSERT(it != m_PendingConnects.end());

    list<GetConnectionCB::ref> *l = it->second;
    m_PendingConnects.erase(it);
    // callback everyone who was waiting for this guy
    for (list<GetConnectionCB::ref>::iterator it = l->begin();
	    it != l->end(); it++) {
	(**it)(conn, err);
    }
    delete l;
}

///////////////////////////////////////////////////////////////////////////////

//
// if the listen socket has data on it, accept the connection and register
// it in our connection hashtable
//
void ATCPTransport::RegisterNewTCPConnectionCB() {
    ASSERT (m_ListenSocket >= 0);

    DBG << "Registering new connection..." << endl;

    struct sockaddr_in address;
    int addrlen = sizeof(sockaddr_in);

    Socket newsock = 0;
    int ret = OS::Accept(&newsock, m_ListenSocket, 
	    (struct sockaddr *) &address, 
	    (socklen_t *) &addrlen);
    if (ret < 0 && errno == EINTR) {
	// interrupted, just ignore this and try again
	return;
    } else if (ret < 0) {
	WARN << "Error while accepting a connection... " << endl;
	return;
    } else {
	// The first things on the connection will be the app level ID
	_tcp_addr *addr = new _tcp_addr;
	ARealNet::Read(newsock, addr->buf, 
		6,
		wrap(this, 
		    &ATCPTransport::RegisterNewTCPConnectionCB2,
		    newsock, addr));
    }
}

void ATCPTransport::RegisterNewTCPConnectionCB2(Socket sock,_tcp_addr *addr,int ret)
{
    DBG << "called" << endl;

    if (ret < 0) {
	DB(1) << "Failed registering new connection (addr)..." << endl;

	OS::CloseSocket(sock);
	// failed to read app ID!
	delete addr;
	return;
    }
    IPEndPoint otherEnd;
    addr->to(otherEnd);
    delete addr;

    // so the file descriptor table doesn't fillup with stable tcp conns
    int restart = 1;
    if (OS::SetSockOpt(sock, SOL_SOCKET, SO_REUSEADDR, 
		(char *) &restart, sizeof(int)) < 0 )
	WARN << "could not configure server socket properly: " 
	    << strerror(errno) << endl;

    // turn off nagle for real-time sending
    int nonagle = 1;
    if (OS::SetSockOpt(sock, IPPROTO_TCP, TCP_NODELAY, 
		(char *) &nonagle, sizeof(int)) < 0 )
	WARN << "StartListening: could not configure server socket properly: " 
	    << strerror(errno) << endl;

    DBG << "Registering new connection from " << otherEnd << endl;

    ATCPConnection *connection = 
	new ATCPConnection(this, sock, &otherEnd);

    // register to read from this
    fdcb(sock, selread, wrap(this, &ATCPTransport::ReadyConnectionCB, connection));
    DBG << "readable: (" << sock << ") " << ARealNet::IsDataWaiting(sock) << endl;

    Lock();
    AConnection *old = m_AppConnHash.Lookup(&otherEnd);
    if (old != NULL) {
	// This can happen if two nodes try to connect to each
	// other simultaneously
	DB(1) << "duplicate connection open for: " << otherEnd << endl;

	// We CAN NOT close either connection, because then we
	// risk losing data (e.g., the stuff that is in the
	// other guy's buffer or in our kernel buffer).
	//
	// For now, I'm just going to hack this and add the
	// new connection to the connectionlist but not the
	// hash. I hope this means we will just use one
	// connection for sending stuff and the other for
	// receiving, which should be fine.

	m_ConnectionList.push_back(connection);
    } else {
	m_ConnectionList.push_back(connection);
	// Make this connection bidirectional -- i.e., when we have
	// data to send to this guy, reuse this connection. No point
	// to initiate a new TCP connection if they aready connected
	// to us
	m_AppConnHash.Insert(&otherEnd, connection);
    }
    Unlock();
}

void ATCPTransport::CloseConnection(IPEndPoint *target) {
    DBG << "called" << endl;

    Lock();
    AConnection *connection = m_AppConnHash.Lookup(target);
    Unlock();
    CloseConnection(connection);
}

void ATCPTransport::CloseConnection(AConnection *connection) {
    if (!connection)
	return;

    DB(1) << "forceably closing connection: " << connection << endl;

    OS::CloseSocket(connection->GetSocket()); // do it ourselves.
    connection->SetStatus(CONN_CLOSED);
    m_AppConnHash.Flush(connection->GetAppPeerAddress());

    // unregister callbacks on socket
    fdcb(connection->GetSocket(), selread, NULL);
}

// This initiates a close which doesn't lose any data (assuming no failures)
void ATCPTransport::ActiveCloseConnection(ATCPConnection *connection) {
    if (!connection)
	return;

    DBG << "active close connection: " << connection << endl;

    // remove from map so we don't send any more data on this
    m_AppConnHash.Flush(connection->GetAppPeerAddress());
    connection->Close(true /* waiting for remote end */);
}

void ATCPTransport::ActiveCloseConnectionCB(ATCPConnection *connection) {
    if (!connection)
	return;

    DBG << "active close finished: " << connection << endl;

    ASSERT(connection->GetStatus() == CONN_CLOSED);

    // unregister callbacks on socket
    fdcb(connection->GetSocket(), selread, NULL);
    // will be garbage collected later
}

void ATCPTransport::PassiveCloseConnection(ATCPConnection *connection) {
    if (!connection)
	return;

    DBG << "passive close connection: " << connection << endl;

    // remove from map so we don't send any more data on this
    m_AppConnHash.Flush(connection->GetAppPeerAddress());
    connection->Close(false /* not waiting for remote end */);

    // unregister callbacks on socket (remote end already closed it)
    fdcb(connection->GetSocket(), selread, NULL);
}

///////////////////////////////////////////////////////////////////////////////

void ATCPTransport::ReadyConnectionCB(ATCPConnection *conn)
{
    ASSERT(!(conn->GetStatus() == CONN_CLOSED ||
		conn->GetStatus() == CONN_ERROR));

    DBG << "connection ready: " << conn << endl;

    // new packet arriving on connection! read it!
    conn->PerformRead(wrap(this, &ATCPTransport::ReadCompleteCB, conn));
}

void ATCPTransport::ReadCompleteCB(ATCPConnection *connection, int status)
{
    // now that read completed, must reregister the read callback to us
    fdcb(connection->GetSocket(), selread, 
	    wrap(this, &ATCPTransport::ReadyConnectionCB, connection));

    DBG << "read complete from: " << connection << " status=" << status << endl;

    switch(status) {
    case NetworkLayer::READ_CLOSE:
	//CloseConnection(connection);
	if (connection->GetStatus() == CONN_CLOSED) {
	    // connection is fully closed, can stop listening on it now
	    ActiveCloseConnectionCB(connection);
	} else {
	    // remote end asked us to close connection, finish sending
	    PassiveCloseConnection(connection);
	}
	GetNetwork()->ProcessMessageCB(this, CONN_CLOSED, connection);
	break;
    case NetworkLayer::READ_ERROR:
	DB(1) << "connection error!" << endl;
	CloseConnection(connection);
	connection->SetStatus(CONN_ERROR);
	GetNetwork()->ProcessMessageCB(this, CONN_ERROR, connection);
	break;
    case NetworkLayer::READ_COMPLETE: {
	ConnStatusType ret = connection->GetStatus();
	if (connection->GetStatus() != CONN_CLOSING)
	    connection->SetStatus(CONN_OK);
	DBG << "process message: " << connection << endl;
	GetNetwork()->ProcessMessageCB(this, CONN_OK, connection);  
	break;
    }
    default:
	// should never get READ_INCOMPLETE
	ASSERT(0);
    }
}
