/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include <awan-env/ATCPTransport.h>
#include <awan-env/ATCPConnection.h>
#include <mercury/MercMessage.h>
#include <awan-env/VPacket.h>
#include <awan-env/ARealNet.h>

ATCPConnection::ATCPConnection(ATransport *t, Socket sock, IPEndPoint *otherEnd) :
AUnbufferedConnection(t, sock, otherEnd), m_IsPendingSend(false),
m_PendingClose(false), m_WaitingClose(false)
{
	//DB(-1) << "called" << endl;
	OS::GetCurrentTime(&m_LastAccess);
	//SetSocketPeerAddress(); -- not needed?
}

ATCPConnection::~ATCPConnection() {
	ASSERT(GetStatus() == CONN_CLOSED || GetStatus() == CONN_ERROR);
}

void ATCPConnection::SendCB(StatusCB::ptr cb, int status) {
	ASSERT(m_IsPendingSend);
	m_IsPendingSend = false;

	// fireoff the next send first before calling back the app
	// because the app callback might do another send, which would
	// go at the front of the queue
	if (m_PendingSends.size() > 0) {
		pair<VPacket *,StatusCB::ptr> p = m_PendingSends.front();
		m_PendingSends.pop_front();
		Send(p.first, p.second);
	}

	if (cb) (*cb)(status);

	// this was marked for closure, try to close now
	if (m_PendingClose)
		DoClose();
}

void ATCPConnection::Send(VPacket *pkt, StatusCB::ptr cb) {
	// no more data should arrive on a connection marked for closure
	//ASSERT(!m_PendingClose); -- this can be called back by the connection...

	OS::GetCurrentTime(&m_LastAccess);

	if (m_IsPendingSend) {
		// wait until the last send is complete
		pair<VPacket *,StatusCB::ptr> p(pkt, cb);
		m_PendingSends.push_back(p);
		return;
	}

	ref<suio> uio = New refcounted<suio>();
	uint32 len = htonl((uint32)pkt->GetSize());
	uio->copy((void *)&len, sizeof(len));
#if 0
	ref<suio> puio = New refcounted<suio>();
	puio->copy(pkt->m_Buffer, pkt->GetSize());
#else
	ref<suio> puio = pkt->GetBuffer();
#endif
	ref< list<ref<suio> > > todo = New refcounted< list<ref<suio> > >();
	todo->push_back(uio);
	todo->push_back(puio);

	m_IsPendingSend = true;
	DBG << "write: " << (uio->resid() + puio->resid()) << " bytes" << endl;
	ARealNet::Write(GetSocket(), todo, wrap(this, &ATCPConnection::SendCB, cb));

	delete pkt;
}

void ATCPConnection::PerformRead(StatusCB::ptr cb) {
	// should not try to read a socket that was already closed
	ASSERT(GetStatus() != CONN_CLOSED && GetStatus() != CONN_ERROR);

	OS::GetCurrentTime(&m_LastAccess);

	// first read how long the packet is
	uint32 *len = new uint32;

	DBG << "called" << endl;

	ARealNet::Read(GetSocket(), (byte *)len, sizeof(uint32),
			wrap(this, &ATCPConnection::PerformReadCB1, len, cb));
}

void ATCPConnection::PerformReadCB1(uint32 *nlen, StatusCB::ptr cb, int status)
{
	DBG << "called: " << status << endl;

	if (status < 0) {
		// error
		SetStatus(CONN_ERROR);
		if (cb) (*cb)(NetworkLayer::READ_ERROR);
		return;
	} else if (status >= 0 && status < (int)sizeof(uint32)) {
		if (status != 0) {
			WARN << "premature EOF, read " << status << " bytes" << endl;
			// hit premature eof
			SetStatus(CONN_CLOSED);
			if (cb) (*cb)(NetworkLayer::READ_ERROR);
		} else {
			// hit aligned eof -- remote close
			m_WaitingClose = false;
			if (GetStatus() == CONN_CLOSING) {
				//DB(-1) << "fin close" << endl;
				// both ends are now closed
				FinClose();
			} else
				SetStatus(CONN_CLOSING);
			if (cb) (*cb)(NetworkLayer::READ_CLOSE);
		}
		return;
	} else {
		// finished read
		ASSERT(GetStatus() != CONN_CLOSED);
		ASSERT(GetStatus() != CONN_ERROR);
		ASSERT(GetStatus() != CONN_CLOSING || m_WaitingClose);

		// init packet buffer
		if (status > ATCPTransport::MAX_TCP_MSGSIZE) {
			WARN << "Got a packet length that is too big: " 
				<< status << endl;

			// Should close connection here... someone got desync'd
			if (cb) (*cb)(NetworkLayer::READ_ERROR);
			return;
		}

		uint32 len = ntohl(*nlen);
		delete nlen;
		DBG << "reading msg size: " << len << endl;

		_InitPacket(len);
		// now read the data
		ARealNet::Read(GetSocket(), m_Packet->GetBuffer(), (int)len,
				wrap(this, &ATCPConnection::PerformReadCB2, len, cb));
	}
}

void ATCPConnection::PerformReadCB2(uint32 len, StatusCB::ptr cb, int status)
{
	ASSERT(GetStatus() != CONN_CLOSED);
	ASSERT(GetStatus() != CONN_ERROR);
	ASSERT(GetStatus() != CONN_CLOSING || m_WaitingClose);

	if (status < 0) {
		// error
		SetStatus(CONN_ERROR);
		if (cb) (*cb)(NetworkLayer::READ_ERROR);
		return;
	} else if (status >= 0 && status < (int)len) {
		// hit premature eof
		SetStatus(CONN_CLOSED);
		if (cb) (*cb)(NetworkLayer::READ_ERROR);
		return;
	} else {
		if (cb) (*cb)(NetworkLayer::READ_COMPLETE);
	}
}

void ATCPConnection::Close(bool waiting)
{
	//DB(-1) << "called: " << waiting << endl;
	m_PendingClose = true;
	m_WaitingClose = waiting;
	DoClose();
}

void ATCPConnection::DoClose()
{
	if (m_IsPendingSend) {
		//DB(-1) << "is pending" << endl;
		// finish sending all the pending data first
		return;
	}

	/*
	   DB(-1) << "called: " << m_WaitingClose << endl;
	   if (m_WaitingClose)
	   DB(-1) << "start close" << endl;
	   if (!m_WaitingClose)
	   DB(-1) << "pass close" << endl;
	 */

	ASSERT(!m_IsPendingSend);
	ASSERT(m_PendingSends.size() == 0);
	shutdown(GetSocket(), SHUT_WR);
	if (m_WaitingClose) {
		// waiting for other end to close
		SetStatus(CONN_CLOSING);
	} else {
		FinClose();
	}
}

void ATCPConnection::FinClose()
{
	ASSERT(!m_IsPendingSend);
	ASSERT(m_PendingSends.size() == 0);
	ASSERT( read(GetSocket(), NULL, 1) == 0 );

	//DB(-1) << "called" << this << endl;
	// both of us have finished closing now
	SetStatus(CONN_CLOSED);
	OS::CloseSocket(GetSocket());
}
