/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:nil -*- */

#ifndef __AREALNET__H
#define __AREALNET__H

#include <map>
#include <list>
#include <async.h>
#include <util/debug.h>
//#include <om/common.h>

#include <util/Environment.h>
#include <util/IPEndPoint.h>
#include <mercury/RoutingLogs.h>
#include <util/ExpLog.h>

#include <wan-env/RealNet.h>
#include <awan-env/ANetUtils.h>

class VPacket;
class ATransport;
class AConnection;

typedef map<ProtoID, ATransport *, less_ProtoID> ATransportMap;
typedef ATransportMap::iterator ATransportMapIter;

typedef callback<void,ConnStatusType,IPEndPoint,MercMessage *> GetNextMessageCB;

///////////////////////////////////////////////////////////////////////////////

//class Scheduler;
class ATransport;
class AConnection;

class ARealNet : public NetworkLayer {
    friend class AConnection;
    friend class ATransport;

    static ATransportMap m_Transports;

    static ATransport *_LookupTransport(const ProtoID& proto);

private:

    inline static void Lock() {} // xxx legacy code
    inline static void Unlock() {} // xxx legacy code

    ///////////////////////////////////////////////////////////////////////////

    //Scheduler            *m_Scheduler;
    IPEndPoint            m_AppID;  // ID for this RealNet instance
    ProtoIDSet            m_Protos; // the set of transports used

    GetNextMessageCB::ptr m_GetNextMessageCB;

    ///////////////////////////////////////////////////////////////////////////

    bool m_RecordBandwidthUsage;
    const uint32 m_WindowSize;    // the size of the bandwidth window in msec
    Window m_InboundWindow;       // the bandwidth usage tracking windows
    Window m_OutboundWindow;

    bool m_EnableMessageLog;

    MeasurementMap m_InboundAggregates, m_OutboundAggregates;

    //Scheduler *GetScheduler () { return m_Scheduler; }
public:

    /**
     * @param the application-level ID to associate with this RealNet instance.
     *
     * @param recordBwidthUsage track bandwidth usage over the last
     * windowSize milliseconds.
     *
     * @param windowSize the window to track bandwidth usage over (ms).
     */
    ARealNet(/*Scheduler *sched,*/ IPEndPoint appID, 
	    bool recordBwidthUsage = false, 
	    uint32 windowSize = 1000);
    virtual ~ARealNet();

    IPEndPoint GetAppID() {
	return m_AppID;
    }

    ///////////////////////////////////////////////////////////////////////////

    //
    // Implement the NetworkLayer methods
    //
    void StartListening(TransportType proto);
    void StopListening();

    void FreeMessage(Message *msg);
    void CloseConnection(IPEndPoint *otherEnd, TransportType proto);

    /** legacy method (xxx not implemented) */
    ConnStatusType GetNextMessage(IPEndPoint *ref_fromWhom, Message **ref_msg);
    /** legacy method (blocking) */
    int SendMessage(Message *msg, IPEndPoint *toWhom, TransportType proto);

    ///////////////////////////////////////////////////////////////////////////
    // Asynchronous RealNet methods

    /**
     * Like NetworkLayer::SendMessage(), but send asynchronously and
     * call a callback with the send status when sending completes
     * (if statuscb is non-NULL).
     */
    void SendMessage(Message *msg, IPEndPoint *toWhom, TransportType proto,
	    StatusCB::ptr statuscb);
    /**
     * Register a callback which will be called when a new message is
     * received. Only one callback may be registered right now.
     *
     * @param cb called with (ConnStatusType t, IPEndPoint from, Message *msg)
     *           msg will be NULL unless t == CONN_NEWINCOMING or CONN_OK.
     */
    void RegisterGetNextMessageCB(GetNextMessageCB::ptr cb) {
	m_GetNextMessageCB = cb;
    }
    /**
     * Called back from ATransport when we get a new message.
     */
    void ProcessMessageCB(ATransport *fromWhom,
	    ConnStatusType status, AConnection *conn);

    ///////////////////////////////////////////////////////////////////////////

    bwidth_t GetOutboundUsage(TimeVal& now); /* in bytes/sec */
    bwidth_t GetInboundUsage(TimeVal& now);  /* in bytes/sec */

    /// Log messages sent and received by this network layer
    void EnableLog() { 
	ASSERT(g_MeasurementParams.enabled);
	m_EnableMessageLog = true; 
    }
    void DoAggregateLogging ();

    ///////////////////////////////////////////////////////////////////////////

    /** creates a non-blocking socket */
    static int CreateSocket(Socket *sockp, int proto);

    static void   ExecWhenWritable (AConnection *conn, callback<void>::ref cb);
    static void   ExecWhenWritable (Socket sock, callback<void>::ref cb);
    static void   ExecWhenReadable (AConnection *conn, callback<void>::ref cb);
    static void   ExecWhenReadable (Socket sock, callback<void>::ref cb);

    static bool   IsDataWaiting (Socket sock);
    static bool   IsWritable    (Socket sock);
    static bool   IsException   (Socket sock);

    /** warning: this will block if sock is not readable! */
    static int    ReadDatagram  (Socket sock, IPEndPoint *fromWhom,
	    byte *buffer, int length);
    /** warning: this will block if sock is not writable! */
    static int    WriteDatagram (Socket sock, IPEndPoint *toWhom,
	    byte *buffer, int length);

    /** non-blocking datagram write */
    static void   WriteDatagram (Socket sock, IPEndPoint toWhom, 
	    ref<suio> uio, StatusCB::ptr cb);

    /** warning: this overrides the callback on sock */
    static void Write (Socket sock, ref<list<ref<suio> > > uios, StatusCB::ptr cb);
    /** warning: this overrides the callback on sock */
    static void Write (Socket sock, ref<suio> uio, StatusCB::ptr cb);
    /** warning: this overrides the callback on sock */
    static void Write (Socket sock, byte *buffer, int length, StatusCB::ptr cb);
    /** warning: this overrides the callback on sock */
    static void  Read (Socket sock, ref<suio> uio, int len, StatusCB::ptr cb);
    /** warning: this overrides the callback on sock */
    static void  Read (Socket sock, byte *buffer, int length, StatusCB::ptr cb);

private:
    void SendMessageCB(VPacket *pkt, StatusCB::ptr statuscb, 
	    AConnection *connection, int status);

    void RecordOutbound(uint32 size, TimeVal& now);
    void RecordInbound(uint32 size, TimeVal& now);

    static void WriteDatagramCB(Socket sock, IPEndPoint toWhom, 
	    ref<suio> uio, StatusCB::ptr cb);

    static void WriteMultiCB(Socket sock, ref<list<ref<suio> > > uios, StatusCB::ptr cb, int stat);

    static void WriteCB(Socket sock, ref<suio> uio, int written, StatusCB::ptr cb);
    static void ReadCB(Socket sock, ref<suio> uio, int len, int read, StatusCB::ptr cb);
};

#endif // __AREALNET__H
