/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include <awan-env/AsyncUtil.h>
#include <util/OS.h>
#include <util/debug.h>

#include <netinet/tcp.h>
#include <util/IPEndPoint.h>
#include <util/Environment.h>
#include <util/Utils.h>
#include <mercury/MercuryNode.h>
#include <mercury/Event.h>
#include <mercury/RoutingLogs.h>
#include <util/Packet.h>
#include <mercury/MercMessage.h>
#include <mercury/options.h>
#include <util/Timer.h>

//#include <awan-env/DelayedTransport.h>
#include <mercury/ObjectLogs.h>

#include "VPacket.h"
#include "ARealNet.h"
#include "AConnection.h"
#include "ATransport.h"

static IPEndPoint s_HackyBootstrapIP;

static MsgType __GetSubTypeForLogging (MercMessage *msg) {
	Interest *in = ((MsgSubscription *) msg)->GetInterest ();

	if (msg->hopCount == 1) 
		return MSG_SUB_NOTROUTING;
	else if (((MsgSubscription *) msg)->IsRoutingModeLinear ())
		return MSG_SUB_LINEAR;
	else
		return MSG_SUB;
}

static MsgType __GetPubTypeForLogging(MercMessage *msg) {
	Event *ev = ((MsgPublication *) msg)->GetEvent();

	// EventType t = ev->GetType();  // cant use this since app can override PointEvent

	// dynamic_cast looks ugly here; but it's okay since we 
	// do this for logging only

	if (dynamic_cast<PointEvent *>(ev)) {
		if (ev->IsMatched())
			return MSG_MATCHED_PUB;
		else 
			return MSG_PUB;
	}
	else {
		if (ev->IsMatched())
			return MSG_RANGE_MATCHED_PUB;
		else {
			if (msg->hopCount == 1)
				return MSG_RANGE_PUB_NOTROUTING;
			else if (((MsgPublication *) msg)->IsRoutingModeLinear ()) 
				return MSG_RANGE_PUB_LINEAR;
			else
				return MSG_RANGE_PUB;
		}
	}

	// Should never come here!
	return MSG_PUB;
}

static byte __GetMsgType (MercMessage *msg)
{
	byte type = msg->GetType ();

	// We want to know the type of the original message for compressed ones
	if (type == MSG_COMPRESSED)
		type = ((MsgCompressed *)msg)->orig->GetType();

	if (type == MSG_PUB)
	{
		if (msg->GetType() == MSG_COMPRESSED)
			type = __GetPubTypeForLogging(((MsgCompressed *) msg)->orig);
		else
			type = __GetPubTypeForLogging(msg);
	}
	if (type == MSG_SUB)
	{
		if (msg->GetType() == MSG_COMPRESSED)
			type = __GetSubTypeForLogging(((MsgCompressed *) msg)->orig);
		else
			type = __GetSubTypeForLogging(msg);

	}

	return type;
}

///////////////////////////////////////////////////////////////////////////////

ATransportMap ARealNet::m_Transports;

ATransport *ARealNet::_LookupTransport(const ProtoID& proto)
{
	ATransport *t = NULL;
	Lock();
	ATransportMapIter it = m_Transports.find(proto);
	if (it != m_Transports.end())
		t = it->second;
	Unlock();
	return t;
}

///////////////////////////////////////////////////////////////////////////////

void ARealNet::DoAggregateLogging ()
{
	AggregateMessageEntry ent (m_InboundAggregates, m_OutboundAggregates);
	g_AggregateMessageLog->Log (ent);

	m_InboundAggregates.clear ();
	m_OutboundAggregates.clear ();

	// reschedule
	delaycb(g_MeasurementParams.aggregateLogInterval/MSEC_IN_SEC,
			(g_MeasurementParams.aggregateLogInterval%MSEC_IN_SEC)*1000000,
			wrap(this, &ARealNet::DoAggregateLogging));
}

///////////////////////////////////////////////////////////////////////////////

//
// We assume here that socket handles returned by the kernel are never "0".
//
ARealNet::ARealNet(IPEndPoint appID, bool recordBwidthUsage, uint32 windowSize) :
m_AppID (appID), m_RecordBandwidthUsage (recordBwidthUsage), 
m_WindowSize (windowSize), m_EnableMessageLog (false)
{
	s_HackyBootstrapIP = IPEndPoint (g_Preferences.bootstrap);

	if (g_MeasurementParams.enabled) {
		IPEndPoint mercid(appID.m_IP, g_Preferences.port);
		InitRoutingLogs(mercid);

		///* xxx FUCK. why the FUCK is this broken?
		if (g_MeasurementParams.aggregateLog) {
			ASSERT(g_AggregateMessageLog != NULL);
			ASSERT(g_AggregateMessageLog != (void *)0xFFFFFFFF);

			// schedule measurement
			delaycb(g_MeasurementParams.aggregateLogInterval/MSEC_IN_SEC,
					(g_MeasurementParams.aggregateLogInterval%MSEC_IN_SEC)*1000000,
					wrap(this, &ARealNet::DoAggregateLogging));
		}
		//*/
	}
}

//
// clear the hash table and the connection list.
// delete the connections only once - since the entries in the hash and list are the same!
// 
ARealNet::~ARealNet() {
	StopListening();
}

//
// create socket depending on what protocol the user wants (tcp/udp)
// bind to a port; and start listening
//
void ARealNet::StartListening(TransportType p) {
	ProtoID proto(m_AppID, p);

	Lock();
	ATransportMapIter it = m_Transports.find(proto);
	ASSERT(it == m_Transports.end());

	ATransport *t = ATransport::Create(this, proto.proto, proto.id);

	/* xxx todo
	   if (g_Preferences.latency) {
	// enable artificial latency?
	t = new DelayedTransport(t);
	}
	 */

	m_Transports[proto] = t;
	Unlock();

	m_Protos.insert(proto);

	t->StartListening();
}

void ARealNet::StopListening() {
	for (ProtoIDSetIter it = m_Protos.begin();
			it != m_Protos.end(); it++) {
		ATransport *t = _LookupTransport(*it);
		ASSERT(t);

		Lock();
		m_Transports.erase(*it);
		Unlock();

		t->StopListening();
		delete t;
	}
	m_Protos.clear();
}

///////////////////////////////////////////////////////////////////////////////

static void SendMessageBlockingCB(bool *rdy, int *fill, int retval) {
	*fill = retval;
	*rdy = true;
}

int ARealNet::SendMessage(Message *amsg, IPEndPoint *toWhom, TransportType p) {
	MercMessage *msg = (MercMessage *) amsg;
	
	bool rdy = false;
	int retval;
	SendMessage(msg, toWhom, p, wrap(&SendMessageBlockingCB, &rdy, &retval));
	while (!rdy) {
		acheck();
	}
	return retval;
}

void ARealNet::SendMessage(Message *amsg, IPEndPoint *toWhom, TransportType p,
		StatusCB::ptr statuscb) {
	//START( SendMessage::1 );

	MercMessage *msg = (MercMessage *) amsg;

	ProtoID proto(m_AppID, p);

	// Jeff 6/7/2004: HACK... This will break non-merc stuff if not set.
	if (msg->IsMercMsg())
		msg->sender = m_AppID;	

	/*
	   ASSERT( connection->GetProtocol() == p );
	 */

	msg->hopCount++;
	DB (7) << "sending msg " << msg << endl;

	//STOP( SendMessage::1 );

	VPacket *pkt;

	//START( SendMessage::2::GetLength );
	int len = msg->GetLength();
	//STOP( SendMessage::2::GetLength );
	int ret;
	///// MercMessage compression
	if (g_Preferences.msg_compress &&
			msg->sender != *toWhom && 
			len > g_Preferences.msg_compminsz) {
		//START( SendMessage::2::Compress );
		MsgCompressed cmsg = MsgCompressed(msg);
		int clen = cmsg.GetLength();
		//STOP( SendMessage::2::Compress );
		// only send compressed if compressed message is smaller
		if (len > clen)
			pkt = new VPacket(&cmsg);
		else
			pkt = new VPacket(msg);
	} else {
		pkt = new VPacket(msg);
	}
	/////

	///// Logging : If we don't want to include processing time,
	///// then we should place this code in StatusCB2
	bool logworthy = 		// Ignore messages from the bootstrap server
		*toWhom != s_HackyBootstrapIP &&
		// Ignore messages we send to ourselves
		*toWhom != m_AppID;

	if (logworthy) {
		if (m_EnableMessageLog) {
			if (!g_MeasurementParams.aggregateLog) {
				MessageEntry ent(MessageEntry::OUTBOUND, 
						p == PROTO_TCP ?  MessageEntry::PROTO_TCP : MessageEntry::PROTO_UDP,
						msg->nonce, __GetMsgType (msg), pkt->GetSize(), msg->hopCount);

				g_MessageLog->Log(ent);
			}
			else {
				byte type = __GetMsgType (msg);

				AggMeasurement *msr = NULL;

				MeasurementMap::iterator it = m_OutboundAggregates.find (type);
				if (it == m_OutboundAggregates.end ()) {
					m_OutboundAggregates.insert (MeasurementMap::value_type (type, AggMeasurement ()));			
					msr = &( (m_OutboundAggregates.find (type))->second );
				}
				else {
					msr = & (it->second);
				}

				msr->hopcount += msg->hopCount;
				msr->size += pkt->GetSize();
				msr->nsamples++;
			}
		}
	}

	/// Ideally, we should be really logging this JUST when the packet goes into the
	/// kernel, but since we are not using CBR, we should be happy. -- ASHWIN [10/11/2004]
	if (g_MeasurementParams.enabled && !g_MeasurementParams.aggregateLog) {
		int type = msg->GetType();
		MercMessage *omsg = msg;

		// We want to know the type of the original message for compressed ones
		if (type == MSG_COMPRESSED) {
			type = ((MsgCompressed *)msg)->orig->GetType();
			omsg = ((MsgCompressed *)msg)->orig;
		}

		if (type == MSG_PUB)
		{
			type = __GetPubTypeForLogging(omsg);
			if (type != MSG_MATCHED_PUB && type != MSG_RANGE_MATCHED_PUB)
			{
				DiscoveryLatEntry ent(DiscoveryLatEntry::PUB_ROUTE_SEND, 
						omsg->hopCount, ((MsgPublication *) omsg)->GetEvent()->GetNonce());
				g_DiscoveryLatLog->Log(ent);
			}
		}
		else if (type == MSG_SUB)
		{
			DiscoveryLatEntry ent(DiscoveryLatEntry::SUB_ROUTE_SEND, 
					omsg->hopCount, ((MsgSubscription *) omsg)->GetInterest()->GetNonce());
			g_DiscoveryLatLog->Log(ent);
		}
	}
	/// MEASUREMENT

	TimeVal now;
	OS::GetCurrentTime(&now);

	///// Bandwidth usage tracking
	if (m_RecordBandwidthUsage) {
		RecordOutbound(pkt->GetSize(), now);
	}

	DBG << "getting connection: " << *toWhom << endl;

	ATransport *t = _LookupTransport(proto);
	ASSERT(t);

	t->GetConnection(toWhom, wrap(this, &ARealNet::SendMessageCB, 
				pkt, statuscb));

	if (msg->GetType() != MSG_PUB || msg->GetType() != MSG_SUB) {
		// XXX HACK: for message structs sent multiple times
		// msg->hopCount--;
		msg->nonce = (uint32)(drand48()*0xFFFFFFFFUL);
	}
	return;
}

void ARealNet::SendMessageCB(VPacket *pkt, StatusCB::ptr statuscb,
		AConnection *connection, int status) {

	if (connection == NULL) {
		// connect failed
		if (statuscb) (*statuscb)(status);
		return;
	}

	DBG << "sending message with: " << connection << endl;

	connection->SendMessage(pkt, statuscb);
}

///////////////////////////////////////////////////////////////////////////////

void ARealNet::ProcessMessageCB(ATransport *t,
		ConnStatusType status, AConnection *connection)
{
	ASSERT(t);
	ASSERT(connection);

	DBG << "Processing message from: " << connection 
		<< " status=" << g_ConnStatusStrings[status] << endl;

	ConnStatusType ret = CONN_NOMSG;

	ASSERT(connection);
	IPEndPoint fromWhom = *connection->GetAppPeerAddress();
	MercMessage *msg = 0;

	// Try to read a message from this connection
	switch (status) {
	case CONN_CLOSED:
	case CONN_CLOSING:
		ret = CONN_CLOSED;
		// Not needed -- transport does this already
		//t->CloseConnection(connection->GetAppPeerAddress());
		break;
	case CONN_ERROR:
		ret = CONN_ERROR;
		// Not needed -- transport does this already
		//t->CloseConnection(connection->GetAppPeerAddress());
		break;
	case CONN_NEWINCOMING:
	case CONN_OK: {
		msg = connection->GetLatestMessage();

		DBG << "Got msg: " << msg << endl;

		if (msg == NULL) {
			DB(1) << "Invalid packet read" << endl;
			ret = CONN_ERROR;
			break;
		}

		uint32 len = (msg)->GetLength();
		bool logworthy = // Ignore messages from the bootstrap server
			s_HackyBootstrapIP != (msg)->sender &&
			// Mercury Messages (pubs, subs) can be sent to ourselves...
			m_AppID != (msg)->sender;

		///// Logging
		if (logworthy) {
			if (m_EnableMessageLog) {

				if (!g_MeasurementParams.aggregateLog) {
					MessageEntry ent(MessageEntry::INBOUND, 
							connection->GetProtocol() == PROTO_TCP ?  MessageEntry::PROTO_TCP : MessageEntry::PROTO_UDP,
							msg->nonce, __GetMsgType (msg), len, msg->hopCount);
					g_MessageLog->Log(ent);
				}
				else {
					byte type = __GetMsgType (msg);
					AggMeasurement *msr = NULL;

					MeasurementMap::iterator it = m_InboundAggregates.find (type);
					if (it == m_InboundAggregates.end ()) {
						m_InboundAggregates.insert (MeasurementMap::value_type (type, AggMeasurement ()));			
						msr = &( (m_InboundAggregates.find (type))->second );
					}
					else {
						msr = & (it->second);
					}

					msr->hopcount += msg->hopCount;
					msr->size += len;
					msr->nsamples++;
				}
			}
		}
		/////

		if (m_RecordBandwidthUsage) {
			TimeVal now;
			OS::GetCurrentTime(&now);
			RecordInbound(len, now);
		}

		//// Decompression
		if ((msg) && (msg)->GetType() == MSG_COMPRESSED) {
			MsgCompressed *cmsg = (MsgCompressed *)msg;
			msg = cmsg->orig;
			(msg)->recvTime = cmsg->recvTime;
			delete cmsg;
		}
		/////

		DB(6) << "Read Complete: " << msg << endl;

		ret = status;
		break;
	}
	default:
		WARN << "BUG: should never be here..." << endl;
		ASSERT(0);
	}

	if (m_GetNextMessageCB) {
		(*m_GetNextMessageCB)(ret,fromWhom,msg);
	} else {
		// otherwise this is a memory leak
		if (msg) {
			delete msg;
		}
	}
}

ConnStatusType ARealNet::GetNextMessage(IPEndPoint *ref_fromWhom, Message **ref_msg) 
{
	// XXX TODO: this is a legacy function...
	return CONN_NOMSG;
}

///////////////////////////////////////////////////////////////////////////////

//
// lookup the socket corresponding to the endpoint
// and close the socket.
//
void ARealNet::CloseConnection(IPEndPoint *otherEnd, TransportType p) {
	ProtoID proto(m_AppID, p);

	ATransport *t = _LookupTransport(proto);
	if (t)
		t->CloseConnection(otherEnd);
}

//
// One could implement a message buffer pool as well! 
//
void ARealNet::FreeMessage(Message *msg) {
	delete msg;
}

///////////////////////////////////////////////////////////////////////////////

int ARealNet::CreateSocket(Socket *sockp, int proto) {
	int err = OS::CreateSocket(sockp, proto);
	if (err < 0) {
		return -1;
	}
	make_async(*sockp);
	if (proto == PROTO_TCP)
		tcp_nodelay(*sockp);
	return err;
}

void ARealNet::ExecWhenWritable(Socket sock, callback<void>::ref cb) {
	if (IsWritable(sock)) {
		// ready right now: execute!
		(*cb)();
		return;
	} else {
		// not writable now: put in queue to execute when writable
		fdcb_once(sock, selwrite, cb);
	}
}

void ARealNet::ExecWhenWritable(AConnection *conn, callback<void>::ref cb) {
	ExecWhenWritable(conn->GetSocket(), cb);
}

void ARealNet::ExecWhenReadable(Socket sock, callback<void>::ref cb) {
	if (IsDataWaiting(sock)) {
		// ready right now: execute!
		(*cb)();
		return;
	} else {
		// not readable now: put in queue to execute when readable
		fdcb_once(sock, selread, cb);
	}
}

void ARealNet::ExecWhenReadable(AConnection *conn, callback<void>::ref cb) {
	ExecWhenReadable(conn->GetSocket(), cb);
}

bool ARealNet::IsDataWaiting(Socket sock) {
	static TimeVal t = { 0, 0 };

	fd_set ready;
	FD_ZERO(&ready);
	FD_SET(sock, &ready);

	select((int) sock + 1, &ready, 0, 0, &t);
	if (FD_ISSET(sock, &ready) != 0)
		return true;
	else
		return false;
}

bool ARealNet::IsWritable(Socket sock) {
	static TimeVal t = { 0, 0 };

	fd_set ready;
	FD_ZERO(&ready);
	FD_SET(sock, &ready);

	select((int) sock + 1, 0, &ready, 0, &t);
	if (FD_ISSET(sock, &ready) != 0)
		return true;
	else
		return false;
}

bool ARealNet::IsException(Socket sock) {
	static TimeVal t = { 0, 0 };

	fd_set ready;
	FD_ZERO(&ready);
	FD_SET(sock, &ready);

	select((int) sock + 1, 0, 0, &ready, &t);
	if (FD_ISSET(sock, &ready) != 0)
		return true;
	else
		return false;
}

int ARealNet::ReadDatagram(Socket sock, IPEndPoint *fromWhom,
		byte *buffer, int length) {
	int error, junklen = sizeof(struct sockaddr_in);
	struct sockaddr_in junk; // can not be static, otherwise not thread safe!

	//START(REALNET::recvfrom);
	// XXX stick in OS::
	error = recvfrom(sock, (char *) buffer, length, 0, 
			(struct sockaddr *) &junk, (socklen_t *) &junklen);
	//STOP(REALNET::recvfrom);

	if (error >= 0) {
		fromWhom->m_IP   = junk.sin_addr.s_addr;
		fromWhom->m_Port = ntohs(junk.sin_port);
	}
	return error;
}

int ARealNet::WriteDatagram(Socket sock, IPEndPoint *toWhom,
		byte *buffer, int length) {
	int error;
	struct sockaddr_in addr; // can not be static, otherwise not thread safe!
	memset((void *) &addr, 0, sizeof(struct sockaddr_in));

	toWhom->ToSockAddr(&addr);

	//START(REALNET::sendto);
	// XXX stick this in OS::
	error = sendto(sock, (char *) buffer, length, 0, 
			(struct sockaddr *) &addr, sizeof(struct sockaddr_in));	
	//STOP(REALNET::sendto);

	return error;
}

void ARealNet::WriteDatagramCB(Socket sock, IPEndPoint toWhom, 
		ref<suio> uio, StatusCB::ptr cb)
{
	struct sockaddr_in addr;
	//memset((void *) &addr, 0, sizeof(struct sockaddr_in));
	toWhom.ToSockAddr(&addr);

	struct msghdr hdr;
	hdr.msg_name = &addr;
	hdr.msg_namelen = sizeof(struct sockaddr_in);
	hdr.msg_iov = (struct iovec *)uio->iov();
	hdr.msg_iovlen = uio->iovcnt();
	hdr.msg_control = NULL;
	hdr.msg_controllen = 0;
	hdr.msg_flags = 0;

	int ret = sendmsg(sock, &hdr, 0);
	if (cb) (*cb)(ret);
}

void ARealNet::WriteDatagram (Socket sock, IPEndPoint toWhom, 
		ref<suio> uio, StatusCB::ptr cb)
{
	ExecWhenWritable(sock, wrap(&ARealNet::WriteDatagramCB, sock, toWhom,
				uio, cb));
}

void ARealNet::WriteCB (Socket sock, ref<suio> uio, 
		int written, StatusCB::ptr cb)
{
	DBG << "writing: " << uio->resid() << endl;

	int start = uio->resid();

	/*
	// check if socket is borked first... dunno why we need to do this
	// but otherwise writev seems to return SIGBUS
	if (IsException(sock)) {
	if (cb) (*cb)(-1);
	return;
	}
	 */

	// assumes sock is non-blocking!
	int n = uio->output(sock);
	if (n < 0 && errno != EAGAIN && errno != EINTR) {
		if (cb) (*cb)(n);
		return;
	}
	written += start - uio->resid();

	DBG << "written: " << written  << " left: " << uio->resid() << endl;
	if (uio->resid()) {
		ExecWhenWritable(sock, wrap(&ARealNet::WriteCB, sock, uio,
					written, cb));
	} else {
		if (cb) (*cb)(written);
	}
}

void ARealNet::Write(Socket sock, ref<suio> uio, StatusCB::ptr cb)
{
	ExecWhenWritable(sock, wrap(&ARealNet::WriteCB, sock, uio, 0, cb)); 
}

void ARealNet::WriteMultiCB(Socket sock, ref<list<ref<suio> > > uios, StatusCB::ptr cb, int stat)
{
	DBG << "stat: " << stat << endl;
	if (stat < 0 || uios->size() == 0)
		(*cb)(stat);
	else {
		ref<suio> fst = uios->front();
		uios->pop_front();
		Write(sock, fst, wrap(&ARealNet::WriteMultiCB, sock, uios, cb));
	}
}

void ARealNet::Write(Socket sock, ref<list<ref<suio> > > uios, StatusCB::ptr cb)
{
	WriteMultiCB(sock, uios, cb, 0);
}

void ARealNet::Write (Socket sock, byte *buffer, int length, StatusCB::ptr cb)
{	
	/*
	   DBG << "Write: len=" << length << endl;
	   DBG_DO {
	   DumpBuffer(buffer, length);
	   }
	 */
	ref<suio> uio = New refcounted<suio>();
	uio->copy(buffer, length);
	Write(sock, uio, cb);
}

void ARealNet::ReadCB(Socket sock, ref<suio> uio, 
		int len, int read, StatusCB::ptr cb)
{
	// assumes sock is non-blocking!
	int n = uio->input(sock, len - read);
	if (n < 0 && errno != EAGAIN && errno != EINTR) {
		if (cb) (*cb)(n);
		return;
	} else if (n == 0) {
		// eof
		if (cb) (*cb)(read);
		return;
	}
	if (n > 0)
		read += n;

	if (len > read)
		ExecWhenReadable(sock, wrap(&ARealNet::ReadCB, sock, uio, 
					len, read, cb));
	else
		if (cb) (*cb)(read);
}

void ARealNet::Read (Socket sock, ref<suio> uio, int len, StatusCB::ptr cb)
{
	ExecWhenReadable(sock, wrap(&ARealNet::ReadCB, sock, uio, len, 0, cb));
}

static void CopyBufferCB(ref<suio> uio, byte *buf, StatusCB::ptr cb, int status)
{
	if (status > 0)
		uio->copyout(buf);

	/*
	   DBG << "Read end: len=" << uio->resid() << endl;
	   DBG_DO {
	   DumpBuffer(buf, uio->resid());
	   }
	 */

	if (cb) (*cb)(status);
}

void ARealNet::Read (Socket sock, byte *buffer, int length, StatusCB::ptr cb)
{
	/*
	   DBG << "Read begin: len=" << length << endl;
	   DBG_DO {
	   DumpBuffer(buffer, length);
	   }
	 */
	ref<suio> uio = New refcounted<suio>();
	Read(sock, uio, length, wrap(&CopyBufferCB, uio, buffer, cb));
}

///////////////////////////////////////////////////////////////////////////////
///// BANDWIDTH PREDICTION

void ARealNet::RecordOutbound(uint32 size, TimeVal& now) 
{
	Measurement last;
	if (m_OutboundWindow.size() != 0) {
		last = m_OutboundWindow.back();
	}
	if (m_OutboundWindow.size() != 0 && now - last.time < BUCKET_SIZE) {
		m_OutboundWindow.back().size += size;
	} else {
		Measurement m;
		m.time = now;
		m.size = size;
		m_OutboundWindow.push_back(m);
	}
}

void ARealNet::RecordInbound(uint32 size, TimeVal& now) 
{
	Measurement last;
	if (m_InboundWindow.size() != 0) {
		last = m_InboundWindow.back();
	}
	if (m_InboundWindow.size() != 0 && now - last.time < BUCKET_SIZE) {
		m_InboundWindow.back().size += size;
	} else {
		Measurement m;
		m.time = now;
		m.size = size;
		m_InboundWindow.push_back(m);
	}
}

bwidth_t ARealNet::GetOutboundUsage(TimeVal& now)
{
	// XXX This should be in "DoWork()"
	while ( m_OutboundWindow.size() > 0 && 
			m_OutboundWindow.front().time + m_WindowSize < now ) {
		m_OutboundWindow.pop_front();
	}

	double used = 0;
	for (WindowIter i = m_OutboundWindow.begin(); 
			i != m_OutboundWindow.end(); i++) {
		used += i->size;
	}

	return (bwidth_t)(used/m_WindowSize)*MSEC_IN_SEC;
}

bwidth_t ARealNet::GetInboundUsage(TimeVal& now)
{
	// XXX This should be in "DoWork()"
	while ( m_InboundWindow.size() > 0 && 
			m_InboundWindow.front().time + m_WindowSize < now ) {
		m_InboundWindow.pop_front();
	}

	double used = 0;
	for (WindowIter i = m_InboundWindow.begin(); 
			i != m_InboundWindow.end(); i++) {
		used += i->size;
	}

	return (bwidth_t)(used/m_WindowSize)*MSEC_IN_SEC;
}
