/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:nil -*- */

#ifndef __ACONNECTION__H
#define __ACONNECTION__H

#include <util/GPL/refcnt.h>
#include <mercury/common.h>
#include <util/IPEndPoint.h>
#include <wan-env/Connection.h>
#include <awan-env/ANetUtils.h>

class ATransport;
class MercMessage;
class Packet;

/**
 * Base class for Transport connections.
 */
class AConnection {
    //friend class DelayedTransport;
    friend ostream& operator<<(ostream& out, AConnection *con);

    ATransport     *m_Transport;             // owning transport

    Socket         m_Socket;                 // Socket for this connection
    IPEndPoint     m_SocketPeerAddress;      // 'getpeername' 
    IPEndPoint     m_AppPeerAddress;         // "App level" ID of the other end

    ConnStatusType m_Status;                 // My status: closed/error/etc...

    // for debugging
    uint32 m_Nonce;

protected:

    AConnection(ATransport *t, Socket sock, IPEndPoint *otherEnd);

    /**
     * This is called to send a data packet filled by the application.
     * The connection becomes the owner of the packet.
     *
     * @return error status
     */
    virtual void Send(VPacket *tosend, StatusCB::ptr cb) = 0;

    /**
     * This is called to get the next message when a READ_COMPLETE is
     * returned from Read() (or anytime after). NULL if none.
     *
     * @param aux packet auxiliary information (filled in)
     */
    virtual Packet *GetNextPacket(PacketAuxInfo *aux) = 0;

    /**
     * Free a packet returned by GetNextPacket();
     */
    virtual void FreePacket(Packet *pkt) = 0;

    virtual void SetSocketPeerAddress();
    virtual void SetSocketPeerAddress(IPEndPoint *addr) { 
	m_SocketPeerAddress = *addr;
    }

public:

    virtual ~AConnection();

    /**
     * Get the creating transport protocol.
     */
    ATransport *GetTransport() {
	return m_Transport;
    }

    /**
     * Get the appID of the remote endpoint of this connection. This is
     * the IPEndPoint that should be used to communicate to this end point.
     */
    IPEndPoint *GetAppPeerAddress() {
	return &m_AppPeerAddress;
    }

    /**
     * Get the actual IP:Port endpoint of the remote end of this connection.
     * This may be different from the appID (e.g., may be just a random
     * port assigned via the kernel).
     */
    IPEndPoint *GetSocketPeerAddress() {
	return &m_SocketPeerAddress;
    }

    /**
     * Set the appID of the remote endpoint of this connection
     */
    void SetAppPeerAddress(IPEndPoint *otherEnd) {
	m_AppPeerAddress = *otherEnd;
    }

    /**
     * Get the associated socket.
     */
    Socket GetSocket() {
	return m_Socket;
    }

    /**
     * Get the associated transport protocol.
     */
    int GetProtocol();

    /**
     * Get the status of this connection.
     */
    ConnStatusType GetStatus() {
	return m_Status;
    }

    /**
     * Set the status of this connection.
     */
    void SetStatus(ConnStatusType status) {
	m_Status = status;
    }

    /**
     * Public interface to send packets via this connection.
     */
    void SendMessage(VPacket *tosend, StatusCB::ptr cb);

    /**
     * If this connection is ready (e.g., the transport returned it
     * as the next ready connection), then call this to retrieve the
     * latest message received. Do NOT call it otherwise, as it may
     * block.
     */
    MercMessage *GetLatestMessage();

    void Print(FILE* stream);
};

ostream& operator<<(ostream& out, AConnection *con);

#endif // __CONNECTION__H
