////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/***************************************************************************
  main.cpp

  SampleApp - simple test application where bots make a random walk along
  a 1D line.

usage: sampleapp [port]

... where [port] is the port this program should run on. (We assume
the bootstrap server is running on 15000)

To see the game in action, you probably want to do the following to
get rid of debugging messages:

sampleapp [port] 2> /dev/null


begin		   : October 10, 2003
copyright	   : (C) 2003 Jeffrey Pang           ( jeffpang@cs.cmu.edu )

 ***************************************************************************/

#include <cstdlib>
#include <cstdio>
#include <typeinfo>

#include <sys/types.h>
#include <unistd.h>

#include <framework2/common.h>
#include <framework2/GObject.h>
#include <framework2/GEvent.h>
#include <framework2/Server.h>
#include <framework2/MercuryProtocolLayer.h>
#include <framework2/Policy.h>

#ifndef MAX
#define MAX(x,y) ((x)>(y)?(x):(y))
#endif
#ifndef MIN
#define MIN(x,y) ((x)<(y)?(x):(y))
#endif


///////////////////////////////////////////////////////////////////////////////
////////// Application Classes

const attrkey_t CharAttrs[]  = { "sprite", "x_coordinate" };
const int NumCharAttrs = 2;

/**
 * An entity bot in our game. Represented by a random capital letter.
 */
class Char:public GObject {

    friend GObject *OCF_Char(Packet *);
    friend GObject *ECF_Char(GEvent *);

public:
    /* how far our character sees in either direction */
    static const int sight = 20;

    /**
     * Local constructor
     */
    Char(GUID guid) : GObject(guid, CharAttrs, NumCharAttrs) {
	setAttribute(string("sprite"), Value((uint32)('A' + (rand() % 26))));
	setAttribute(string("x_coordinate"), Value((uint32)(rand() % 79)));
    }

    virtual ~Char() {
    }

    /**
     * @return the location of the entity
     */
    int GetLoc() {
	return (int)getAttribute(string("x_coordinate")).m_Ival;
    }

    /**
     * Have the entity make a random move (left, right, or none)
     */
    void MakeMove() {
	int move = rand() % 3;
	switch (move) {
	case 0:
	    break;
	case 1:
	    MoveLeft();
	    break;
	case 2:
	    MoveRight();
	    break;
	}
    }

    void MoveLeft() {
	if (GetLoc() > 0) {
	    setAttribute(string("x_coordinate"), Value((uint32)(GetLoc()-1)));
	}
    }

    void MoveRight() {
	if (GetLoc() < 78) {
	    setAttribute(string("x_coordinate"), Value((uint32)(GetLoc()+1)));
	}
    }

    int GetSprite() {
	return (int)getAttribute(string("sprite")).m_Ival;;
    }

    ////////// Implements GObject //////////

protected:

    virtual GUpdate *createUpdate() {
	return NULL; // XXX TODO
    }

    virtual void handleUpdate(GUpdate *delta) {
	// XXX TODO
    }

public:
    virtual bool isClient() { 
	return false; 
    }

    virtual gtype_t typeCode() {
	return 0;
    }

    Char(Packet *pkt) : GObject(pkt) {

    }
};

///////////////////////////////////////////////////////////////////////////////
////////// Mercury Globals

// TODO: this can probably be generated automatically some how
GObject *OCF_Char(Packet *pkt) {
    return new Char(pkt);
}

// TODO: This is *really* yucky. An application writer should NOT have
// to do this... maybe make this automatic? We have to decide what stuff
// goes into events: everything? a delta? how to reconstruct from a delta?
GObject *ECF_Char(GEvent *evt) {

    Char *obj = new Char(evt->getGUID());
    obj->handleEvent(evt);

    return obj;
}

///////////////////////////////////////////////////////////////////////////////
////////// Application Globals

SID sid = { 0, 0 };
ServerParams params;

Server *server;
MercuryProtocolLayer *proto;
Policy *policy;

/**
 * The gameworld -- a 1D array of 79 spaces (fits in a console :)
 */
char *gameworld;

/**
 * The bot character we own.
 */
Char *me;


///////////////////////////////////////////////////////////////////////////////
////////// Application Code

/**
 * Initialize the local game world and mercury framework.
 *
 * @param port the port to start the local mercury rounter on
 */
void InitGame(int port)
{
    // init rand seed
    srand((int) getpid());

    // start merc controller
    sid.port = port;
    proto = new MercuryProtocolLayer(port);
    policy = new StaticPlacementPolicy();
    server = new Server(sid, &params, proto, policy);

    proto->start();

    // setup game world
    gameworld = new char[80];
    gameworld[79] = '\0';

    // set up our character
    GUID gid = { 0, 0, 0 };
    gid.port = port;
    me = new Char(gid);

    // register my character
    server->handleAddObject(me);
}

/**
 * Make a subscription for our current character's state.
 *
 * @return the subscription
 */
Interest *MakeSub()
{
    string attr("x_coordinate");
    uint32 min = MAX(0, me->GetLoc() - Char::sight);
    uint32 max = MIN(78, me->GetLoc() + Char::sight);

    Interest *in = new Interest();
    in->AddConstraint(attr, OP_GREATER_EQUAL, Value(min));
    in->AddConstraint(attr, OP_LESS_EQUAL, Value(max));
    // in->SetUnsubscribe(false);

    return in;
}

/**
 * Start the game loop (infinite loop)
 */
void RunGame()
{
    InterestSet s;
    Interest *in;

    while (1) {
	////////// update state
	// ask our entity to move
	me->MakeMove();

	////////// update subs
	// unregister old sub
	if (in) {
	    // XXX This needs to be fixed, this should be soft state
	    // in->SetUnsubscribe(true);
	    s.insert(in);
	    me->setInterests(s);
	    s.clear();
	}
	// register new sub
	s.insert(MakeSub());
	in = MakeSub();
	me->setInterests(s);
	s.clear();

	// update local gameworld
	for (int i = 0; i < 79; i++) {
	    gameworld[i] = '.';
	}
	ObjectStore store = server->getObjectStore();
	for (ObjectStoreIter it = store.begin();
		it != store.end(); it++) {
	    Char *obj = dynamic_cast < Char * >((*it).second);
	    gameworld[obj->GetLoc()] = obj->GetSprite();
	}
	if (me->GetLoc() - Char::sight >= 0)
	    gameworld[me->GetLoc() - Char::sight] = '[';
	if (me->GetLoc() + Char::sight < 79)
	    gameworld[me->GetLoc() + Char::sight] = ']';

	////////// render frame
	printf("%s\n", gameworld);

	////////// sleep
	usleep(250000);
    }

}

/**
 * Entry point
 */
int main(int argc, char **argv)
{

    ////////// read arguments
    if (argc != 2) {
	fprintf(stderr, "usage: sampleapp [port]\n");
	return 1;
    }
    int port = atoi(argv[1]);

    ////////// set mercury preferences
    strcpy(g_Preferences.bootstrap, "127.0.0.1:15000");
    strcpy(g_Preferences.hostname, "127.0.0.1");
    g_Preferences.port = port;
    g_Preferences.pub_delay = 100;
    g_Preferences.send_pubs = true;
    g_Preferences.send_subs = true;
    g_VerbosityLevel = 5;

    ////////// initialize the game and mercury
    InitGame(port);

    ////////// run the game
    RunGame();

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
