////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/***************************************************************************

begin		   : October 10, 2003
copyright	   : (C) 2003 Jeffrey Pang           ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <Colyseus.h>
#include <om/Manager.h>
#include <om/test/TestAdaptor.h>
#include <om/test/TestObject.h>
#include <util/Benchmark.h>
#include <util/Utils.h>

struct VirtualNode {

    SID sid;
    TestAdaptor *t;
    Manager *m;
    ObjectStore *s;

    VirtualNode() : t(NULL), m(NULL), s(NULL) {}

    void Init(int port)
    {
	sid.m_IP = inet_addr("127.0.0.1");
	sid.m_Port = port;

	TestAttributes attrs;
	TestAttribute attr1 = { "test1", true };
	TestAttribute attr2 = { "test2", true };
	TestAttribute attr3 = { "test3", true };
	attrs.push_back(attr1);
	attrs.push_back(attr2);
	attrs.push_back(attr3);

	t = new TestAdaptor();
	t->SetAttributes(attrs);

	m = new Manager(t, false);
	s = t->GetObjectStore();
    }

    void Switch() {
	g_LocalSID = sid;
	TestObject::m_TestAdaptor = t;
    }
};

MsgAppUpdateAck *MakeAck(ReplicaConnectionReceiver *conn);
void PingPong(VirtualNode& senderNode, ReplicaConnectionSender *sender, 
	      VirtualNode& receiverNode, ReplicaConnectionReceiver *receiver,
	      int count, float loss_rate = 0, float ack_loss_rate = 0,
	      TestObject *touch_obj = NULL, char *field = NULL);
MsgAppUpdateAck *Ping(VirtualNode& senderNode,
		      ReplicaConnectionSender *sender, 
		      VirtualNode& receiverNode,
		      ReplicaConnectionReceiver *receiver,
		      float loss_rate = 0, float ack_loss_rate = 0);
void DoAck(VirtualNode& senderNode, ReplicaConnectionSender *sender, 
	   VirtualNode& receiverNode, ReplicaConnectionReceiver *receiver,
	   MsgAppUpdateAck *ack);

///////////////////////////////////////////////////////////////////////////////

MsgAppUpdateAck *MakeAck(ReplicaConnectionReceiver *conn)
{
    MsgAppUpdateAck *ack = new MsgAppUpdateAck();
    ack->epoch    = conn->GetEpoch();
    ack->ackSeqno = conn->GetLastSeqNoAck();
    return ack;
}

void PingPong(VirtualNode& senderNode, ReplicaConnectionSender *sender, 
	      VirtualNode& receiverNode, ReplicaConnectionReceiver *receiver,
	      int count, float loss_rate, float ack_loss_rate,
	      TestObject *touch_obj, char *field)
{
    for (int i=0; i<count; i++) {
	senderNode.Switch();

	if (touch_obj) {
	    string attr;
	    if (field) {
		attr = string(field);
	    } else {
		double choice = drand48();
		if (choice <= 0.333) {
		    attr = string("test1");
		} else if (choice > 0.333 && choice <= 0.666) {
		    attr = string("test2");
		} else {
		    attr = string("test3");
		}
	    }

	    DBG << "MOD ATTR: " << attr << endl;

	    touch_obj->SetAttribute(attr, Value((uint32)1));
	}

	MsgAppUpdateAck *ack = Ping(senderNode,
				    sender,
				    receiverNode,
				    receiver,
				    loss_rate,
				    ack_loss_rate);

	senderNode.Switch();
	if (touch_obj)
	    touch_obj->ClearDeltaMask(); 

	DoAck(senderNode,
	      sender,
	      receiverNode,
	      receiver,
	      ack);
    }
}

MsgAppUpdateAck *Ping(VirtualNode& senderNode,
		      ReplicaConnectionSender *sender, 
		      VirtualNode& receiverNode,
		      ReplicaConnectionReceiver *receiver,
		      float loss_rate, float ack_loss_rate)
{
    senderNode.Switch();
    MsgAppUpdate *update = scopy<MsgAppUpdate>(sender->CreateUpdate());
    ASSERT(update);

    if (drand48() < loss_rate) {
	DBG << "DROPPED MSG" << endl;
	delete update;
	return NULL;
    }

    receiverNode.Switch();
    receiver->HandleUpdate(update);
    delete update;

    MsgAppUpdateAck *ack_orig = MakeAck(receiver);
    MsgAppUpdateAck *ack      = scopy<MsgAppUpdateAck>(ack_orig);
    delete ack_orig;

    if (drand48() < ack_loss_rate) {
	DBG << "DROPPED ACK" << endl;
	delete ack;
	return NULL;
    }

    return ack;
}

void DoAck(VirtualNode& senderNode, ReplicaConnectionSender *sender, 
	   VirtualNode& receiverNode, ReplicaConnectionReceiver *receiver,
	   MsgAppUpdateAck *ack)
{
    if (ack) {
	senderNode.Switch();
	sender->HandleUpdateAck(ack);
	delete ack;
    }
}

int main(int argc, char **argv)
{
    // init rand seed
    srand48((int) getpid());
    InitializeMercury(&argc, argv);

    VirtualNode n1, n2;
    n1.Init(1);
    n2.Init(2);

    n1.Switch();
    ReplicaConnectionSender *sender = 
	new ReplicaConnectionSender(n2.sid, n1.m);

    n2.Switch();
    ReplicaConnectionReceiver *receiver =
	new ReplicaConnectionReceiver(n1.sid, 0, 0, n2.m);

    n1.Switch();
    TestObject *obj = new TestObject(n1.m->CreateGUID());
    n1.s->_ManagerAdd(obj);
    sender->RegisterGUID(obj->GetGUID());


#define PP n1, sender, n2, receiver

    MsgAppUpdateAck *ack1, *ack2, *ack3;

    ///////////////////////////////////////////////////////////////////////////

    DBG << "================" << endl;
    DBG << "(1) send a couple messages" << endl;
    DBG << "================" << endl;

    PingPong(PP, 5);

    DBG << "sender: " << sender << endl;
    DBG << "recver: " << receiver << endl;

    ///////////////////////////////////////////////////////////////////////////

    DBG << "================" << endl;
    DBG << "(2) drop a couple" << endl;
    DBG << "================" << endl;

    PingPong(PP, 1, 1, 0, obj, "test1");
    PingPong(PP, 1, 1, 0, obj, "test2");
    PingPong(PP, 1, 1, 0, obj, "test3");
    PingPong(PP, 3);

    DBG << "sender: " << sender << endl;
    DBG << "recver: " << receiver << endl;

    ///////////////////////////////////////////////////////////////////////////

    DBG << "================" << endl;
    DBG << "(3) reset connection" << endl;
    DBG << "================" << endl;

    PingPong(PP, 101, 1);
    PingPong(PP, 2);

    DBG << "sender: " << sender << endl;
    DBG << "recver: " << receiver << endl;

    ///////////////////////////////////////////////////////////////////////////

    DBG << "================" << endl;
    DBG << "(4) multiple in flight" << endl;
    DBG << "================" << endl;

    ack1 = Ping(PP, 0, 0);
    ack2 = Ping(PP, 0, 0);
    ack3 = Ping(PP, 0, 0);

    DoAck(PP, ack1);
    DoAck(PP, ack2);
    DoAck(PP, ack3);

    PingPong(PP, 1);

    DBG << "sender: " << sender << endl;
    DBG << "recver: " << receiver << endl;

    ///////////////////////////////////////////////////////////////////////////

    DBG << "================" << endl;
    DBG << "(5) drop in middle" << endl;
    DBG << "================" << endl;

    ack1 = Ping(PP, 0, 0);
    Ping(PP, 1, 0);
    ack2 = Ping(PP, 0, 0);
    ack3 = Ping(PP, 0, 0);

    DoAck(PP, ack1);
    DoAck(PP, ack2);
    DoAck(PP, ack3);

    PingPong(PP, 2);

    DBG << "sender: " << sender << endl;
    DBG << "recver: " << receiver << endl;

    ///////////////////////////////////////////////////////////////////////////

    DBG << "================" << endl;
    DBG << "(5) dropped acks" << endl;
    DBG << "================" << endl;

    PingPong(PP, 5, 0, 0.5);

    PingPong(PP, 2);

    DBG << "sender: " << sender << endl;
    DBG << "recver: " << receiver << endl;

    ///////////////////////////////////////////////////////////////////////////

    DBG << "================" << endl;
    DBG << "(5) dropped both" << endl;
    DBG << "================" << endl;

    PingPong(PP, 10, 0.5, 0.5);

    PingPong(PP, 2);

    DBG << "sender: " << sender << endl;
    DBG << "recver: " << receiver << endl;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
