////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#ifndef __VALUE__H
#define __VALUE__H

#include <util/ID.h>
#include <util/common.h>
#include <string>

using namespace std;

typedef enum {
    ATTR_INVALID,
    ATTR_CHAR,
    ATTR_UINT,
    ATTR_FLOAT,
    ATTR_STRING,
    ATTR_GUID,
    ATTR_SID,
    ATTR_IDBUNDLE,
    ATTR_BLOB,
} AttrType;

static const char *g_AttrStrings[] = 
    {
	"INVALID TYPE",
	"CHAR",
	"UINT",
	"FLOAT",
	"STRING",
	"GUID",
	"SID",
	"IDBUNDLE",
	"BLOB",
    };

extern const char* g_AttrStrings[];

typedef enum {
    OP_EQUAL,
    OP_NOT_EQUAL,
    OP_LESS_THAN,
    OP_LESS_EQUAL,
    OP_GREATER_THAN,
    OP_GREATER_EQUAL,
    OP_SUFFIX,
    OP_PREFIX,
    OP_ANY
} OperatorType;

///////////////////////////////////////////////////////////////////////////////

//
// XXX Careful about memory leaks here...
//
class TestValue : public Serializable {
 public:
    AttrType  m_Type;
    union {
	uint32 	m_Ival;
	char 	m_Cval;
	float 	m_Fval;
	string *m_Sval;
	guid_t *m_GUIDval;
	sid_t  *m_SIDval;
	struct {
	    guid_t *guid;
	    sid_t  *sid;
	} m_IDval;
	struct {
	    int     len;
	    void   *data;
	} m_Bval;
    };

    TestValue() : m_Type(ATTR_INVALID), m_Ival(0) {}

    TestValue(uint32);
    TestValue(char);
    TestValue(float);
    TestValue(string);
    TestValue(char *s);
    TestValue(guid_t);
    TestValue(sid_t);
    TestValue(guid_t, sid_t);      /* "ID bundle" so the recvr can locate guid */
    TestValue(size_t, void *);     /* caller retains ownership of data */ 

    TestValue(const TestValue&);

    TestValue(Packet *pkt);
    virtual ~TestValue();

    AttrType GetType() { return m_Type; }

    //
    // Methods of the Serializable interface...
    //
    void   Serialize(Packet *pkt);
    uint32 GetLength();
    void   Print(FILE *stream);

    TestValue& operator=	( const TestValue& );
    int	   operator<	( TestValue& );
    int	   operator>	( TestValue& );
    int	   operator==	( TestValue& );
    int	   operator>=	( TestValue& );
    int	   operator<=	( TestValue& );

    static float ComputeDistance(TestValue&, TestValue&, TestValue&, TestValue& );
    static double CalcAbsDistance( TestValue* a, TestValue* b );

 private:
    void  _AssignMembers(const TestValue& );
};

// I/O utilities
istream& operator>>(istream& in, TestValue& val);
ostream& operator<<(ostream& out, TestValue& val);

#endif // __VALUE__H
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
