////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/***************************************************************************

  TestObject.cpp

begin		   : April 5, 2004
copyright	   : (C) 2003 Jeffrey Pang   ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include "TestObject.h"
#include "TestAdaptor.h"
#include <om/Manager.h>

extern Manager *g_Manager;
static DeltaMask DELTA_MASK_ALL;

static void InitDeltaMaskAll() {
    static bool inited = false;

    if (!inited) {
	for (int i=0; i<MAX_ATTRS; i++)
	    DELTA_MASK_ALL.Set(i);
	inited = true;
    }
}

TestAdaptor *TestObject::m_TestAdaptor = &g_TestAdaptor;

void TestObject::GetRefs(GObjectRefList *tofill) {
    for (RefMapIter it = m_Refs.begin(); it != m_Refs.end(); it++) {
	test_ref_t *ref = it->second;
	tofill->push_back(ref);
    }
}

void TestObject::ResolveRef(GObjectRef *ref, GObject *obj) {
    GUID ptr;

    if (obj == NULL) {
	ptr = GUID_NONE;
    } else {
	ptr = obj->GetGUID();
    }

    string key = ((test_ref_t *)ref)->attr;
    RefMapIter p = m_Refs.find(key);
    delete p->second;
    if (p == m_Refs.end()) {
	WARN << "trying to resolve non-existent ref!" << endl;
	return;
    }
    m_Refs.erase(p);
    m_Attributes[key] = TestValue(ptr);
}

TestObject::TestObject(GUID id) : 
    GObject(id),
    m_FixedCost(1), m_DeltaCost(1), m_IsNew(true), m_Migrating(false) 
{
    InitDeltaMaskAll();
}

TestObject::TestObject(GObjectInfoIface *info,
		       Packet *pkt, const DeltaMask& mask,
		       SIDMap *unresolved) :
    GObject(info), m_IsNew(false), m_Migrating(false)
{
    InitDeltaMaskAll();
    ASSERT(mask == DELTA_MASK_ALL);
    UnpackUpdate(pkt, mask, unresolved);
}

void TestObject::SetAttribute(string key, TestValue new_val)
{
    DBG << "called guid=" << GetGUID() << " " << key << "=" << new_val << endl;

    TestAttributes *attrs = m_TestAdaptor->GetAttributes();
    int index = -1;
    for (uint32 i=0; i<attrs->size(); i++) {
	//DBG << "comparing test=" << (*attrs)[i].key << " with " << key << endl;
	if ( (*attrs)[i].key == key ) {
	    index = i;
	}
    }
    // must be a known attribute
    ASSERT(index >= 0);

    AttributeMapIter p = m_Attributes.find(key);

    // see if we replaced an unresolved ref...
    RefMapIter rp = m_Refs.find(key);
    if (rp != m_Refs.end()) {
	// if so, assume its no longer unresolved...
	delete rp->second;
	m_Refs.erase(rp);
    }

    /*
      if (p != m_Attributes.end() && p->second.m_Type == ATTR_GUID) {
      // we're removing a reference...
      GObjectRefMapIter ref = m_Refs.find(*p->second.m_GUIDval);
      ASSERT(ref != m_Refs.end());
      delete ref->second;
      m_Refs.erase(*p->second.m_GUIDval);
      }

      if (new_val.m_Type == ATTR_GUID) {
      // we're adding a reference...
      guid_t guid = *new_val.m_GUIDval;
      GObject *obj = m_TestAdaptor->GetObjectStore()->Find(guid);
      ASSERT(obj != NULL);
      m_Refs[guid] = new test_ref_t(obj);
      }
    */

    // set the new value
    m_Attributes[key] = new_val;

    // set the bit in the delta mask
    SetDeltaMaskField(index);
}

TestValue& TestObject::GetAttribute(string key)
{
    AttributeMapIter iter = m_Attributes.find(key);
    ASSERT(iter != m_Attributes.end());

    return iter->second;
}

void TestObject::SetInterest(OMInterest& sub)
{
    ASSERT(sub.begin() != sub.end());
    m_Interest = OMInterest(sub);
    // HACK!!!
    m_Interest.AddTag(GetGUID());
    ASSERT(m_Interest.begin() != m_Interest.end());

    DBG << "called: " << m_Interest << endl;
}

extern Value test_val_to_val(int index, const TestValue& tv);

void TestObject::FillEvents(EventList *reg,
			    EventList *unreg,
			    EventSet  *curr)
{
    bool hasPub = false;
    bool same   = true;

    OMEvent *ev  = g_Manager->CreateEvent(GetGUID());
    OMEvent *old = NULL;
    if (curr)
	old = *(curr->begin());

    NamedAttributes *attrs = m_TestAdaptor->GetNamedAttributes();
    int index = 0;
    for (NamedAttributes::iterator it = attrs->begin();
	 it != attrs->end(); it++, index++) {
	string key = *it;
	AttributeMapIter p = m_Attributes.find(key);
	if (p != m_Attributes.end()) {
	    const Value *ov = NULL;
	    if (old) ov = old->GetValue(index);
	    Value v = test_val_to_val(index, p->second);

	    ev->AddTuple(Tuple(index, v));
	    hasPub = true;

	    if (!old || !ov || !(*ov == v)) {
		same = false;
	    }
	}
    }

    if (same)
	hasPub = false;

    /*
      if (!hasPub) {
      ostream& out = (WARN << "no attributes? " << GetGUID() << ": ");
      for (AttributeMapIter i= m_Attributes.begin(); 
      i != m_Attributes.end(); i++) {
      if (i != m_Attributes.begin()) out << ",";
      out << i->first << ":" << i->second;
      }
      out << endl;
      }
    */

    if (!hasPub) {
	g_Manager->FreeEvent(ev);
    } else {
	ev->SetLifeTime( g_TestAdaptor.GetSendInterval() );
	reg->push_back(ev);
	if (old)
	    unreg->push_back(old);
    }
}

uint32 TestObject::InterestTime(OMEvent *ev, InterestMap *curr) 
{
    if (m_Interest.Overlaps(ev)) {
	// XXX: The current regression tests just set the min TTL 
	// very long :(
	return 1;
    } else {
	DB(0) << "Asked if we are interested in an event that we are "
	      << " not intereted in" << endl;
	return 0;
    }

    return 0;
}

uint32 TestObject::InterestTime(GObject *obj, InterestMap *curr)
{
    // XXX This is hacky!!! We desparately need to update this framework
    // so that it knows about all the functionality we use!

    EventList reg, unreg;
    obj->FillEvents(&reg, &unreg, NULL);
    uint32 interestTime = 0;
    if (reg.size() > 0) {
	for (EventListIter it = reg.begin(); it !=  reg.end(); it++) {
	    (*it)->SetMatched();
	    interestTime = MAX(interestTime, InterestTime(*it, NULL));
	    g_Manager->FreeEvent(*it);
	}
    }

    return interestTime;
}

uint32 TestObject::FillInterest(OMInterest *in, OMInterest *old)
{
    if (old && m_Interest.GetGUID() == old->GetGUID()) {
	DB(1) << "Interest didn't change from last: " << old->GetGUID()
	      << endl;
	// same sub, don't need to resend
	return 0;
    }

    for (Interest::iterator it = m_Interest.begin(); it != m_Interest.end(); it++) {
	in->AddConstraint( **it );
    }

    m_Interest.SetGUID(in->GetGUID());

    return m_Interest.begin() != m_Interest.end() ? g_TestAdaptor.GetSendInterval() : 0;
}

void TestObject::PackUpdate(Packet *pkt,
			    const DeltaMask& mask)
{
    DBG << "called guid=" << GetGUID() << " mask=" << mask << endl;

    if (mask == DELTA_MASK_ALL) {
	DBG << "serializing new obj stuff" << endl;
	pkt->WriteFloat(m_FixedCost);
	pkt->WriteFloat(m_DeltaCost);
	m_Interest.Serialize(pkt);
    }

    TestAttributes *attrs = m_TestAdaptor->GetAttributes();
    ObjectStore *store = m_TestAdaptor->GetObjectStore();

    for (uint32 i=0; i < attrs->size(); i++) {
	if (mask.IsSet(i)) {
	    string key = (*attrs)[i].key;

	    if (m_Attributes.find(key) == m_Attributes.end()) {
		// hacky way to signal field is missing
		pkt->WriteByte((byte)false);
	    } else {
		pkt->WriteByte((byte)true);
		TestValue v = m_Attributes[key];
		if (v.m_Type == ATTR_GUID) {
		    GObject *target = store->Find(*v.m_GUIDval);
		    ASSERT(target != NULL);

		    v = TestValue(target->GetGUID(), target->GetSID());
		}
		v.Serialize(pkt);

		DBG << "serializing attr: " << key << "=" << v << endl;
	    }
	}
    }
}

const DeltaMask& TestObject::GetInitDeltaMask()
{
    return DELTA_MASK_ALL;
}

void TestObject::UnpackUpdate(Packet *pkt, 
			      const DeltaMask& mask,
			      SIDMap *unresolved)
{
    DBG << "called guid=" << GetGUID() << endl;

    if (mask == DELTA_MASK_ALL) {
	// treat this case specially to pack in extra junk that is
	// required for reconstituting the entire object.
	m_FixedCost = pkt->ReadFloat();
	m_DeltaCost = pkt->ReadFloat();
	m_Interest  = OMInterest(pkt);
    }

    TestAttributes *attrs = m_TestAdaptor->GetAttributes();

    for (uint32 i=0; i < attrs->size(); i++) {
	if (mask.IsSet(i)) {
	    bool present = (bool)pkt->ReadByte();
	    if (!present) continue;

	    string key = (*attrs)[i].key;
	    TestValue v = TestValue(pkt);
	    if (v.m_Type == ATTR_IDBUNDLE) {
		guid_t guid = *v.m_IDval.guid;
		GObject *obj = m_TestAdaptor->GetObjectStore()->Find(guid);
		if (obj == NULL) {
		    (*unresolved)[guid] = *v.m_IDval.sid;
		    m_Refs[key] = new test_ref_t(key, guid);
		}
		m_Attributes[key] = TestValue(guid);
	    } else {
		m_Attributes[key] = v;
	    }
	    DBG << "Setting attribute " << key << "=" << v << endl;
	}
    }
}

bool TestObject::IsMigratable()
{
    return true;
}

uint32 TestObject::BeginMigration(sid_t target)
{
    m_Migrating = true;
    return 10000;
}

void TestObject::CancelMigration(sid_t target)
{
    ASSERT(m_Migrating);
    m_Migrating = false;
    return;
}

void TestObject::CommitMigration(sid_t target)
{
    ASSERT(IsReplica() || m_Migrating);
    m_Migrating = false;
    return;
}

///////////////////////////////////////////////////////////////////////////////

ostream& operator<<(ostream& out, TestObject *objp)
{
    out << "(GObject guid=" << objp->GetGUID() 
	<< " sid=" << objp->GetSID() 
	<< " status=" << (objp->IsReplica()?"R":"P")
	<< " dirty=" << (objp->IsDirty()?"Y":"N")
	<< " migrating=" << (objp->m_Migrating?"Y":"N")
	<< " attributes=[";
    for (AttributeMapIter i= objp->m_Attributes.begin(); 
	 i != objp->m_Attributes.end(); i++) {
	if (i != objp->m_Attributes.begin()) out << ",";
	out << i->first << ":" << i->second;
    }
    out << "] sub=" << &objp->m_Interest << ")";
    return out;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
