////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/***************************************************************************

  om_test.cpp

begin		   : October 10, 2003
copyright	   : (C) 2003 Jeffrey Pang           ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <pthread.h>
#include <cstdlib>
#include <cstdio>
#include <readline/readline.h>
#include <readline/history.h>

#include <Colyseus.h>
#include <om/Manager.h>
#include "TestAdaptor.h"
#include "TestObject.h"
#include "TestValue.h"
#include <util/Benchmark.h>
#include <util/ServerSocket.h>
#include <mercury/options.h>

typedef map<string, guid_t> StrGUIDMap;
typedef StrGUIDMap::iterator StrGUIDMapIter;
typedef map<guid_t, string, less_GUID> StringMap;
typedef StringMap::iterator StringMapIter;

void InitGame();
void RunGame();
void RunNetGame(ServerSocket& sock);
void UnNewAllObjects();
int HandleInput(FILE *file, struct timeval *now, bool echoInput = false);

void handleHelp(char **args);
void handleScript(char **args);
void handleSend(char **args);
void handleRecv(char **args);
void handleMerc(char **args);
void handleAdd(char **args);
void handleRemove(char **args);
void handleSet(char **args);
void handleSetIn(char **args);
void handleSetCost(char **args);
void handleMigrate(char **args);
void handleShow(char **args);
void handleStore(char **args);
void handleSleep(char **args);
void handleGraph(char **args);
void handleGraphVis(char **args);
void handleLoad(char **args);
void handleLoadMap(char **args);
void handleReplicaConn(char **args);
void handleObjectInfo(char **args);
void handleBenchmark(char **args);

Value test_val_to_val(int index, const TestValue& tv);
int parse_val(TestValue *val, char *str);
guid_t parse_guid(char *str);
sid_t parse_sid(char *str);
TestObject *find_obj(char *name);
char *extract_str(char *buf, int *len);
char **extract_args(char *buf);
void CheckTimeouts(struct timeval *now);
void Cleanup(int sig);

extern char g_ProgramName[];
static bool g_NoColorize;
static bool g_NoUseReadLine;
static bool g_Periodic;
static char g_Script[512];
static bool g_Network;
static int  g_Port;

static char g_Attributes[512];
static char g_NamedAttributes[512];

///////////////////////////////////////////////////////////////////////////////

OptionType g_AppOptions[] = {
    { '#', "Test application parameters", OPT_SEP, "", NULL, "", NULL },

    {'R', "noreadline", OPT_BOOL | OPT_NOARG,
     "don't use GNU readline for input", 
     &(g_NoUseReadLine), "0", (void *)"1"},
    {'C', "nocolor", OPT_BOOL | OPT_NOARG,
     "don't colorize output", 
     &(g_NoColorize), "0", (void *)"1"},
    {'S', "script", OPT_STR,
     "run a series of scripted commands", g_Script, "", NULL},
    {'P', "periodic", OPT_BOOL | OPT_NOARG,
     "periodically call Recv() automatically", 
     &g_Periodic, "0", (void *)"1"},
    {'A', "attrs", OPT_STR,
     "non-named object attributes", g_Attributes, "", NULL},
    {'N', "nattrs", OPT_STR,
     "named object attributes", g_NamedAttributes, "", NULL},
    {'/', "network", OPT_BOOL | OPT_NOARG,
     "read input from network instead of stdin",
     &g_Network, "0", (void *)"1"},
    {'/', "testport", OPT_INT,
     "port if listening from network",
     &g_Port, "60000", NULL},
    {0, 0, 0, 0, 0, 0, 0}
};

typedef struct {
    const char *cmd;
    const char *abbrev;
    void (*handler)(char **args);
    const char *usage;
    const char *help;
} TAInputHandler;

static TAInputHandler g_Handlers[] = {
    { "HELP",   "H",  handleHelp,   "",
      "Show this help message" },
    { "RUN",    "R",  handleScript, "<script-file>", 
      "Run commands in the file <script-file>" },
    { "SEND",   "SD", handleSend,   "", 
      "Call Manager::Send()" },
    { "RECV",   "RC", handleRecv,   "", 
      "Call Manager::Recv()" },
    { "MERC",   "MC", handleMerc,   "",
      "Call Manager::Merc()" },
    { "ADD",    "A",  handleAdd,    "<name>", 
      "Add a new object identified with <name> to the Store" },
    { "REMOVE", "R",  handleRemove, "<ID>",
      "Remove object <ID> from the Store" },
    { "SET",    "S",  handleSet,    "<ID> <attr_key> <attr_val>",
      "Set an attribute in object <ID>" },
    { "SETIN",  "I",  handleSetIn,  "<ID> [<attr_key> <op> <attr_val>]*",
      "Set the interest of object <ID>. Each triple is a constraint" },
    { "SETCOST",  "C",  handleSetCost,  "<ID> <fixed> <delta>*",
      "Set the cost of object <ID> (should only be called on prims)." },
    { "MIGRATE",  "M",  handleMigrate,  "<ID> <SID>",
      "Migrate the primary <ID> to the node <SID>" },
    { "SHOW",   "D",  handleShow,   "<ID>",
      "Show the object <ID>" },
    { "STORE",  "O",  handleStore,  "", 
      "Show the entire object store" },
    { "SLEEP",  "P",  handleSleep,  "<msec>", 
      "Sleep for <msec> milliseconds" },
    { "GRAPH",  "G", handleGraph, "",
      "Dump the interest-graph in format <src>,<tgt>,<link_time>" },
    { "GRAPHVIS", "GV", handleGraphVis, "",
      "Dump the interest-graph and display in GraphVis" },
    { "OBJINFO",  "OI", handleObjectInfo, "",
      "Dump the object info map" },
    { "LOAD",   "L", handleLoad, "",
      "Obtain the load value of the node" },
    { "LOADMAP","LM", handleLoadMap, "",
      "Dump the load map of the node in format <tgt>,<toLow>,<toHigh>" },
    { "CONN","CO", handleReplicaConn, "",
      "Dump the replica connections" },
    { "BENCHMARK","BM", handleBenchmark, "",
      "Dump benchmark numbers" },
    { NULL, NULL, NULL, NULL }
};

extern Manager *g_Manager;
static StrGUIDMap  g_Names;
static StringMap g_GUIDs;
static ObjectStore *g_Store;
static ostream *g_Out = NULL;
static ServerSocket g_Listen, g_Sock;

static pthread_mutex_t g_ManagerLock;

#define FRAME_INTERVAL 100000

class PeriodicThread : public Thread
{
    void Run() {
	g_TestAdaptor.SetSendInterval(FRAME_INTERVAL);

	while (1) {
	    pthread_mutex_lock(&g_ManagerLock);
	    g_Manager->Send();
	    UnNewAllObjects();
	    g_Manager->Recv();
	    pthread_mutex_unlock(&g_ManagerLock);
	    usleep(FRAME_INTERVAL);
	}
    }
};

///////////////////////////////////////////////////////////////////////////////

void UnNewAllObjects()
{
    g_Store->Begin();
    TestObject *obj;
    while ( (obj = (TestObject *)g_Store->Next()) != NULL ) {
	obj->SetIsNew(false);
    }
}

void InitGame()
{
    // init rand seed
    srand((int) getpid());

    // if read input from network, wait for connect
    if (g_Network) {
	try {
	    g_Listen = ServerSocket(g_Port);
	} catch (SocketException ex) {
	    Debug::die("can't create socket: %s", ex.description().c_str() );
	}

	try {
	    g_Listen.Accept(g_Sock);
	} catch (SocketException ex) {
	    WARN << "accept on local socket failed: " 
		 << ex.description() << "\n";
	    Debug::die("");
	}
    }

    // start merc manager
    g_Manager = Manager::GetInstance(&g_TestAdaptor);
    g_Store = g_TestAdaptor.GetObjectStore();

    TestAttributes attrs;

    // set the object attributes
    char *str = strtok(g_Attributes, ",;:");
    while (str != NULL) {
	TestAttribute attr;
	attr.key = str;
	attr.isNamed = false;
	attrs.push_back(attr);
	str = strtok(NULL, ",;:");
    }	
    str = strtok(g_NamedAttributes, ",;:");
    if (str == NULL) Debug::die("No named attributes specified");
    while (str != NULL) {
	TestAttribute attr;
	attr.key = str;
	attr.isNamed = true;
	attrs.push_back(attr);
	str = strtok(NULL, ",;:");
    }
    g_TestAdaptor.SetAttributes(attrs);
    g_TestAdaptor.SetSendInterval(g_Preferences.sub_lifetime);

    pthread_mutex_init(&g_ManagerLock, NULL);
    if (g_Periodic) {
	(new PeriodicThread())->Start();
    }

    // HACK: do a couple rounds of Send(), Recv() to initialize Mercury
    for (int i=0; i<5; i++) {
	g_Manager->Send();
	UnNewAllObjects();
	g_Manager->Recv();
    }
}

void RunGame()
{
    struct timeval now;

    if ( strcmp(g_Script, "") ) {
	char *dummy[2];
	dummy[0] = "SCRIPT";
	dummy[1] = g_Script;
	handleScript(dummy);
    }

    if (g_Network) {
	RunNetGame(g_Sock);
    } else {

	while(true) {
	    gettimeofday(&now, NULL);
	    if ( HandleInput(stdin, &now) < 0 ) {
		Debug::warn("stdin was closed!");
		Cleanup(0);
	    }
	    fflush(stdout);
	}	

    }
}

void RunNetGame(ServerSocket& sock)
{
    char msg[MAX_MSG_SIZE];
    int pos = 0;

    while (1) {
	int ret = read(sock.GetSocket(), &msg[pos], MAX_MSG_SIZE-pos);
	if (ret == 0) {
	    DBG << "client closed Socket " << sock.GetSocket() << "\n";
	    // EOF
	    break;
	} else if (ret < 0) {
	    // XXX FIXME: errno is not thread-safe!
	    if (errno == EINTR) {
		// interrupted, try again...
		continue;
	    } else {
		// some other error, don't really want to recover
		WARN << "read error on client Socket "
		     << sock.GetSocket() << ": " 
		     << strerror(errno) << "\n";
		break;
	    }
	} else {
	    pos += ret;
	    while (1) {
		char *str = extract_str(msg, &pos);
		if (str == NULL)
		    break;

		DB(1) << "req: " << str << "\n";

		char **args = extract_args(str);
		char *res;

		sock << "==BEGIN_RESP" << "\n";
		DBG << "==BEGIN_RESP" << "\n";
		if (! args[0] || args[0] == '\0') {
		    sock << "ERROR: no command provided" << "\n";
		} else {
		    bool done = false;

		    for (int i=0; g_Handlers[i].cmd != NULL; i++) {
			if ( !strcasecmp(args[0], g_Handlers[i].cmd) ||
			     !strcasecmp(args[0], g_Handlers[i].abbrev) ) {

			    stringstream out;
			    g_Out = &out;

			    pthread_mutex_lock(&g_ManagerLock);
			    g_Handlers[i].handler(args);
			    pthread_mutex_unlock(&g_ManagerLock);

			    sock << out.str();
			    DBG << out.str();
			    g_Out = &cout;

			    done = true;
			    break;
			}
		    }

		    if (!done) {
			sock << "ERROR: unknown command: " 
			     << args[0] << "\n";
			DBG << "ERROR: unknown command: " 
			    << args[0] << "\n";
		    }
		}
		sock << "==END_RESP" << "\n";
		DBG << "==END_RESP" << "\n";
	    }
	}
    }

    sock.Close();
}

int HandleInput(FILE *in /* only works without readline! */, 
		struct timeval *now,
		bool echoInput)
{
    char _line_buf[512];
    bool free_cmdline = false; // HACK!
    char *cmdline;

    if (!g_NoUseReadLine) {
	HISTORY_STATE *state;

	cmdline = readline("> ");

	if (cmdline == NULL) {
	    return -1;
	}

	state = history_get_history_state();
	if (state->length > 1000) {
	    HIST_ENTRY *ent = remove_history(0);
	    free(ent->line);
	    free(ent);
	}
	HIST_ENTRY *last = history_get(MAX(history_base + state->length-1, 0));
	if (last == NULL || strcmp(cmdline, last->line)) {
	    add_history(cmdline);
	} else {
	    free_cmdline = true;
	}
    } else {
	errno = 0;
	if ( fgets(_line_buf, 512, in) == NULL ) {
	    if (errno) {
		Debug::warn("fgets error on input: %s", strerror(errno));
	    }
	    return -1;
	}
	_line_buf[strlen(_line_buf)-1] = '\0'; // remove trailing newline
	cmdline = _line_buf;
    }

    if (echoInput)
	(*g_Out) << "> " << cmdline << endl;
    DBG << "> " << cmdline << endl;

    char **args = extract_args(cmdline);

    (*g_Out) << (!g_NoColorize?"[31m":"") 
	     << "==BEGIN_RESP" 
	     << (!g_NoColorize?"[m":"")  << endl;
    if (args[0] == '\0') {
	(*g_Out) << "ERROR: no command provided" << endl;
    } else {
	bool done = false;

	for (int i=0; g_Handlers[i].cmd != NULL; i++) {
	    if ( !strcasecmp(args[0], g_Handlers[i].cmd) ||
		 !strcasecmp(args[0], g_Handlers[i].abbrev) ) {

		pthread_mutex_lock(&g_ManagerLock);
		g_Handlers[i].handler(args);
		pthread_mutex_unlock(&g_ManagerLock);

		done = true;
		break;
	    }
	}

	if (!done) {
	    (*g_Out) << "ERROR: unknown command: " << args[0] << endl;
	}
    }
    (*g_Out) << (!g_NoColorize?"[31m":"") 
	     << "==END_RESP" 
	     << (!g_NoColorize?"[m":"")  << endl;

    if (free_cmdline) free(cmdline);

    return 0;
}

///////////////////////////////////////////////////////////////////////////////

#define NOT_ENOUGH_ARGS(x) (*g_Out) << "ERROR: need " << x << " arguments" << endl
#define DOES_NOT_EXIST(x) (*g_Out) << "ERROR: object does not exist" << endl;


void handleHelp(char **args) {
    (*g_Out) << "Definitions:" << endl;
    (*g_Out) << "  [35m<ID>[m       = a name you gave an object or its GUID" << endl;
    (*g_Out) << "  [35m<attr_key>[m = a string (no spaces) for an attribute" << endl;
    (*g_Out) << "  [35m<attr_val>[m = a string <type>:<value> string for value" << endl;
    (*g_Out) << "  [35m<type>[m     = {int,char,byte,real,string,guid,ID}" << endl;
    (*g_Out) << "  [35m<guid>[m     = <ip>:<port>:<localOID>, where <ip> is in dotted-tail format" << endl;
    (*g_Out) << "  [35m<op>[m       = {==,!=,<,<=,>,>=,*}, where * means 'any'" << endl;

    (*g_Out) << endl << "Commands:" << endl;

    for (int i=0; g_Handlers[i].cmd != NULL; i++) {
	(*g_Out) << "  [35m[" << g_Handlers[i].abbrev << "] "
		 << g_Handlers[i].cmd << " " 
		 << g_Handlers[i].usage  << "[m" << endl;
	(*g_Out) << "    " << g_Handlers[i].help << endl;
    }
}

void handleScript(char **args) {
    if (args[1] == NULL) {
	NOT_ENOUGH_ARGS(1);
	return;
    }

    struct timeval now;
    FILE *script = fopen(args[1], "r");
    if (script == NULL) {
	(*g_Out) << "ERROR: can't open script file: " << args[1] << endl;
	return;
    }

    // HACK!
    bool old_val = g_NoUseReadLine;
    g_NoUseReadLine = true;
    while ( true ) {
	gettimeofday(&now, NULL);
	if ( HandleInput(script, &now, true) < 0 ) {
	    break;
	}
    }
    g_NoUseReadLine = old_val;

    fclose(script);
}

void handleSend(char **args)
{
    g_Manager->Send();
    UnNewAllObjects();
    (*g_Out) << "OK: Send() called" << endl;
}

void handleRecv(char **args)
{
    g_Manager->Recv();
    (*g_Out) << "OK: Recv() called" << endl;
}

void handleMerc(char **args)
{
    g_Manager->Merc();
    (*g_Out) << "OK: Merc() called" << endl;
}

void handleAdd(char **args)
{
    if (args[1] == NULL) {
	NOT_ENOUGH_ARGS(1);
	return;
    }

    string key(args[1]);

    if (g_Names.find(key) != g_Names.end()) {
	(*g_Out) << "ERROR: object with name " << args[1] << 
	    " already exists" << endl;
	return;
    }

    guid_t guid = g_Manager->CreateGUID();
    g_Names[key] = guid;
    g_GUIDs[guid] = key;
    g_Store->ApplicationAdd(new TestObject(guid));
    //g_Manager->AddGUID(guid);

    (*g_Out) << "OK: Added object " << args[1] << " with guid=" << guid << endl;
    return;
}

void handleRemove(char **args)
{
    if (args[1] == NULL) {
	NOT_ENOUGH_ARGS(1);
	return;
    }

    TestObject *obj = find_obj(args[1]);
    if (obj == NULL) {
	DOES_NOT_EXIST(args[1]);
	return;
    }

    guid_t guid = obj->GetGUID();
    g_Names.erase(string(args[1]));
    g_GUIDs.erase(guid);
    g_Store->ApplicationRemove(guid);
    /*
      if (! obj->IsReplica() ) {
      g_Manager->DeleteGUID(guid);
      }
    */
    delete obj;

    (*g_Out) << "OK: removed object with guid=" << guid << endl;
    return;
}

void handleSet(char **args)
{
    if (args[1] == NULL || args[2] == NULL || args[3] == NULL) {
	NOT_ENOUGH_ARGS(3);
	return;
    }

    TestObject *obj = find_obj(args[1]);
    if (obj == NULL) {
	DOES_NOT_EXIST(args[1]);
	return;
    }

    string attr(args[2]);
    TestValue val;
    if (!parse_val(&val, args[3])) {
	return;
    }

    obj->SetAttribute(attr, val);
    (*g_Out) << "OK: object guid=" << obj->GetGUID() << " set attribute " 
	     << attr << " to value: " << val << endl;
    return;
}

void handleSetCost(char **args)
{
    if (args[1] == NULL || args[2] == NULL || args[3] == NULL) {
	NOT_ENOUGH_ARGS(3);
	return;
    }

    TestObject *obj = find_obj(args[1]);
    if (obj == NULL) {
	DOES_NOT_EXIST(args[1]);
	return;
    }

    if (obj->IsReplica()) {
	(*g_Out) << "ERROR: object is a replica" << endl;
	return;
    }

    CostMetric fixed = atof(args[2]);
    CostMetric delta = atof(args[3]);

    obj->SetFixedCost(fixed);
    obj->SetDeltaCost(delta);
    (*g_Out) << "OK: object cost: fixed=" << fixed << " delta=" << delta << endl;
    return;
}

struct info {
    Value min, max;
};

void handleSetIn(char **args) {
    if (args[1] == NULL) {
	NOT_ENOUGH_ARGS(2);
	return;
    }

    TestObject *obj = find_obj(args[1]);
    if (obj == NULL) {
	DOES_NOT_EXIST(args[1]);
	return;
    }

    map<int, info> cons;

    OMInterest in;
    int index = 2;

    while (1) {
	if (! args[index]) {
	    break;
	}
	if (! args[index+1]) {
	    (*g_Out) << "ERROR: missing constraint for attribute key=" 
		     << args[index] << endl;
	    return;
	}
	if (! args[index+2]) {
	    (*g_Out) << "ERROR: missing value for attribute key=" 
		     << args[index] << endl;
	    return;
	}

	string attr(args[index]);
	char *type = args[index+1];
	OperatorType op;

	if (! strcmp(type, "==") ) {
	    op = OP_EQUAL;
	} else if (! strcmp(type, "!=") ) {
	    op = OP_NOT_EQUAL;
	} else if (! strcmp(type, "<") ) {
	    op = OP_LESS_THAN;
	} else if (! strcmp(type, "<=") ) {
	    op = OP_LESS_EQUAL;
	} else if (! strcmp(type, ">") ) {
	    op = OP_GREATER_THAN;
	} else if (! strcmp(type, ">=") ) {
	    op = OP_GREATER_EQUAL;
	} else if (! strcmp(type, "*") ) {
	    op = OP_ANY;
	} else {
	    (*g_Out) << "ERROR: unknown operator: " << type << endl;
	    return;
	}

	TestValue val;
	if (!parse_val(&val, args[index+2])) {
	    return;
	}

	int i = 0;
	NamedAttributes *attrs = g_TestAdaptor.GetNamedAttributes();
	for (NamedAttributes::iterator it = attrs->begin();
	     it != attrs->end(); it++, i++) {
	    if (attr == *it) break;
	}

	Value v = test_val_to_val(i, val);
	if (op == OP_EQUAL) {
	    cons[i].min = v;
	    cons[i].max = v;
	} else if (op == OP_LESS_EQUAL || op == OP_LESS_THAN) {
	    // XXX off by 1 error!
	    cons[i].max = v;
	} else if (op == OP_GREATER_EQUAL || op == OP_GREATER_THAN) {
	    // XXX off by 1 error
	    cons[i].min = v;
	} else {
	    // XXX dunno how to emulate op_any!
	    ASSERT(0);
	}

	index+=3;
    }

    // XXX does not detect il-formed constraints!

    for (map<int, info>::iterator it = cons.begin(); it != cons.end(); it++) {
	Constraint c = Constraint(it->first, it->second.min, it->second.max);
	in.AddConstraint(c);
    }

    obj->SetInterest(in);

    (*g_Out) << "OK: set object guid=" << obj->GetGUID() 
	     << " interest to: " << &in << endl;
}

void handleMigrate(char **args)
{
    if (args[1] == NULL || args[2] == NULL) {
	NOT_ENOUGH_ARGS(2);
	return;
    }

    TestObject *obj = find_obj(args[1]);
    if (obj == NULL) {
	DOES_NOT_EXIST(args[1]);
	return;
    }

    if (obj->IsReplica()) {
	(*g_Out) << "ERROR: can't migrate a replica: " 
		 << obj->GetGUID() << endl;
	return;
    }

    sid_t sid = parse_sid(args[2]);
    if (sid == g_Manager->GetSID()) {
	(*g_Out) << "ERROR: can't migrate object to ourselves" << endl;
	return;
    }
    if (sid == SID_NONE) {
	(*g_Out) << "ERROR: bad sid value: " << args[2] << endl;
	return;
    }

    g_Manager->Migrate(obj, sid);

    (*g_Out) << "OK: started migration for guid=" << obj->GetGUID() 
	     << " to " << sid << endl;
    return;
}

void handleShow(char **args)
{
    if (args[1] == NULL) {
	NOT_ENOUGH_ARGS(1);
	return;
    }

    TestObject *obj = find_obj(args[1]);
    if (obj == NULL) {
	DOES_NOT_EXIST(args[1]);
	return;
    }

    (*g_Out) << "OK: " << obj << endl;
}

void handleStore(char **args)
{
    TestObject *obj;
    g_Store->Begin();
    while ( (obj = (TestObject *)g_Store->Next()) != NULL ) {
	(*g_Out) << "OK: " << obj << endl;
    }
}

void handleSleep(char **args)
{
    if (args[1] == NULL) {
	NOT_ENOUGH_ARGS(1);
	return;
    }
    int msec = atoi(args[1]);
    usleep(msec*USEC_IN_MSEC);
    (*g_Out) << "OK: slept for " << msec << " ms" << endl;
    return;
}

void handleGraph(char **args)
{
    const char *output = g_Manager->DumpInterestGraph();
    (*g_Out) << output;
    return;
}

void handleLoad(char **args)
{
    TimeVal now;
    gettimeofday(&now, NULL);

    (*g_Out) << g_Manager->GetLoad(now) << endl;
    return;
}

void handleLoadMap(char **args)
{
    const char *output = g_Manager->DumpNodeLoadMap();
    (*g_Out) << output;
    return;
}

void handleReplicaConn(char **args)
{
    const char *output = g_Manager->DumpReplicaConnections();
    (*g_Out) << output;
    return;
}

void handleObjectInfo(char **args)
{
    const char *output = g_Manager->DumpObjectInfo();
    (*g_Out) << output;
    return;
}

#define TEMP_DOT "/tmp/TestApp.graphvis.tmp.dot"
#define TEMP_PS  "/tmp/TestApp.graphvis.tmp.ps"

void handleGraphVis(char **args)
{
    const char *output = g_Manager->DumpInterestGraphToGraphViz();

    FILE *tmp = fopen(TEMP_DOT, "w");
    if (!tmp) {
	(*g_Out) << "ERROR: couldn't open temp graph vis file: " 
		 << TEMP_DOT << endl;
    }
    fprintf(tmp, output);
    fclose(tmp);
    (*g_Out) << "OK: generating postscript..." << endl;
    system("neato -Tps " TEMP_DOT " > " TEMP_PS);
    (*g_Out) << "OK: displaying; close window to continue..." << endl;
    system("gv " TEMP_PS);

    return;
}

void handleBenchmark(char **args)
{
    Benchmark::print();
    return;
}

///////////////////////////////////////////////////////////////////////////////

TestObject *find_obj(char *name)
{
    string key(name);
    guid_t guid;

    if (g_Names.find(key) == g_Names.end()) {
	guid = parse_guid(name);
    } else {
	guid = g_Names[key];
    }

    TestObject *obj = (TestObject *)g_Store->Find(guid);
    return obj;
}

Value test_val_to_val(int index, const TestValue& tv)
{
    real64 dTv;
    switch(tv.m_Type) {
    case ATTR_UINT:
	dTv = tv.m_Ival; break;
    case ATTR_FLOAT:
	dTv = tv.m_Fval; break;
    default:
	ASSERT(0);
	dTv = 0; // make g++ happy
    }
    return g_Manager->ConvertToValue(index, dTv);
}

int parse_val(TestValue *val, char *str)
{
    char *type   = strtok(str, ":");
    char *value  = strtok(NULL, " "); // until end

    if (type == NULL || value == NULL) {
	(*g_Out) << "ERROR: invalid value string: " << str << endl;
	return 0;
    }

    if ( !strcasecmp(type, "i") ||
	 !strcasecmp(type, "int") ||
	 !strcasecmp(type, "integer") ) {

	*val = TestValue( (uint32)atoi(value) );

    } else if ( !strcasecmp(type, "c") ||
		!strcasecmp(type, "char") ) {

	*val = TestValue( (char)value[0] );

    } else if ( !strcasecmp(type, "b") ||
		!strcasecmp(type, "byte") ) {

	*val = TestValue( (char)atoi(value) );

    } else if ( !strcasecmp(type, "f") ||
		!strcasecmp(type, "float") ||
		!strcasecmp(type, "real") ) {

	*val = TestValue( (float)atof(value) );

    } else if ( !strcasecmp(type, "s") ||
		!strcasecmp(type, "string") ||
		!strcasecmp(type, "str") ) {

	string s(value);
	*val = TestValue( s );

    } else if ( !strcasecmp(type, "g") ||
		!strcasecmp(type, "guid") ) {

	guid_t guid = parse_guid(value);
	if (guid == GUID_NONE) {
	    (*g_Out) << "ERROR: invalid guid=" << value << endl;
	    return 0;
	}
	if (g_Store->Find(guid) == NULL) {
	    (*g_Out) << "ERROR: guid " << value << " wasn't found in ObjectStore"
		     << endl;
	    return 0;
	}
	*val = TestValue( guid );

    } else if ( !strcasecmp(type, "I") ||
		!strcasecmp(type, "ID") ) {

	string k(value);

	if (g_Names.find(k) == g_Names.end()) {
	    (*g_Out) << "ERROR: can't find object with id " << k << endl;
	    return 0;
	}
	guid_t guid = g_Names[k];
	*val = TestValue( guid );

    } else {
	(*g_Out) << "ERROR: invalid value type: " << type << endl;
	return 0;
    }

    return 1;
}

guid_t parse_guid(char *str)
{
    char *ip   = strtok(str, ":");
    char *port = strtok(NULL, ":");
    char *localOID = strtok(NULL, ":");
    if (ip == NULL || port == NULL || localOID == NULL) {
	return GUID_NONE;
    }
    uint32 _ip   = inet_addr(ip);
    uint16 _port = (uint16)atoi(port);
    uint32 _localOID = (uint32)atoi(localOID);
    if (_ip == INADDR_NONE || _port == 0) {
	return GUID_NONE;
    }

    guid_t guid(_ip, _port, _localOID);

    return guid;
}

sid_t parse_sid(char *str)
{
    char *ip   = strtok(str, ":");
    char *port = strtok(NULL, ":");
    if (ip == NULL || port == NULL) {
	return SID_NONE;
    }
    uint32 _ip   = inet_addr(ip);
    uint16 _port = (uint16)atoi(port);
    if (_ip == INADDR_NONE || _port == 0) {
	return SID_NONE;
    }

    sid_t sid(_ip, _port);

    return sid;
}

///////////////////////////////////////////////////////////////////////////////

static char str_buf[512];
static char *arg_buf[512];

char *extract_str(char *buf, int *len) 
{
    int i;
    bool found = 0;

    // parse one command terminated by a newline into the static buf
    for (i=0; i<*len; i++) {
	str_buf[i] = buf[i];
	if (str_buf[i] == '\n') {
	    str_buf[i] = '\0';
	    // if client writes \r\n, then need to get rid of two chars
	    if (str_buf[i-1] == '\r')
		str_buf[i-1] = '\0';
	    found = 1;
	    i++;
	    break;
	}
    }

    // if there is stuff left in the buffer, move it to the front
    int j = 0;
    for ( ; i<*len; i++, j++) {
	buf[j] = buf[i];
    }
    *len = j;

    if (found) {
	return str_buf;
    } else {
	return NULL;
    }
}

char **extract_args(char *buf) {
    int i;
    char *last = buf;
    int index = 0;

    // separate command string into an array of string args
    for (i=0; i<512; i++) {
	bool end = (buf[i] == '\0');

	if (end || isspace(buf[i])) {
	    buf[i] = '\0';
	    if (last < buf+i) {
		arg_buf[index] = last;
		++index;
		if (index > 512)
		    break;
	    }
	    last = buf + i + 1;
	}

	if (end) break;
    }
    arg_buf[index] = NULL;

    return arg_buf;
}

///////////////////////////////////////////////////////////////////////////////

void Cleanup(int sig)
{
    exit(0);
}

int main(int argc, char **argv)
{
    g_Out = &cout;

    ////////// set mercury preferences
    InitializeColyseus(&argc, argv, g_AppOptions);

    ////////// initialize the game and mercury
    InitGame();

    ////////// run the game
    RunGame();

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
