#!/usr/bin/perl
#
# Simple regression test.
#
# (1) Test that a replica is properly created on a subscriber.
# (2) Test that a replica is modified on a subscriber.
# (3) Update the contents of a primary by modifying the replica.
# (4) Fetching GUID pointers and object resolution.
# (5) Only receive pubs for objects in subscription.
# (6) Primary migration primitive.
#

use strict;
require "TestLib.pl";

my $NUM_NODES = 5;

start_bootstrap( "x" => [0, 10000] );

my %node = start_nodes($NUM_NODES, 
		       { "REPLICA_MAINTAIN_TTL" => 10000,
			 "INTEREST_LINK_TTL" => 10000,
			 "REPLICA_INTEREST_TTL" => 10000 },
		       "-l", 15000, "-v", 5, 
		       "-D 'NONE'", "-f", #"--debugfuncs", "Send,HandleEvent,HandlePub",
		       #"--measurement", 
		       #"--latency", "--latency-file", "~/Merc/localtest10.lat",
		       "--log-flush-interval", 10, "--pubtriggers", "--log-binary",
		       "-N", "x", "-A", "name,nickname,ptr,ptr2,self");

my (@guid, %store, $primary, $replica, $temp);

sub dump_store($) {
    my %store   = store(shift @_);
    foreach my $k (keys %store) {
	print "$k => " . obj_to_str( $store{$k} ) . "\n";
    }
}

#if (0) {

$guid[0] = add($node{1});
set($node{1}, $guid[0], "name", "s:Bobby");
set($node{1}, $guid[0], "x", "f:10");
#print obj_to_str( show($node{1}, $guid[0]) ) . "\n";

$guid[1] = add($node{3});
setin($node{3}, $guid[1], ["x", ">", "f:0"], ["x", "<", "f:20"]);

info("##### TEST 1: Replica Creation");

# need up to NUM_NODES cycles for Merc to fwd message around ring
idle_all(\%node, $NUM_NODES+3);

$primary = show($node{1}, $guid[0]);
info("original: " .  obj_to_str( $primary ) );

$replica = show($node{3}, $guid[0]);
info("replica: " .  obj_to_str( $replica ) );

if (! obj_eq($primary, $replica) ) {
    error("Wrong replica for $guid[0] in store on 3:");
    dump_store($node{3});
    fail();
}

info("##### TEST 1.5: Replica Creation from Pub");

$guid[10] = add($node{1});
set($node{1}, $guid[10], "name", "s:BobbyTwinMonster");
set($node{1}, $guid[10], "x", "f:10");

idle_all(\%node, $NUM_NODES);

$primary = show($node{1}, $guid[10]);
info("original: " .  obj_to_str( $primary ) );

$replica = show($node{3}, $guid[10]);
info("replica: " .  obj_to_str( $replica ) );

if (! obj_eq($primary, $replica) ) {
    error("Wrong replica for $guid[10] in store on 3:");
    dump_store($node{3});
    fail();
}

info("##### TEST 2: Replica Changing");

set($node{1}, $guid[0], "name", "s:Joe");
set($node{1}, $guid[0], "nickname", "s:Jimmy");

# need up to NUM_NODES cycles for Merc to fwd message around ring
idle_all(\%node, $NUM_NODES+3);

$primary = show($node{1}, $guid[0]);
info("original: " .  obj_to_str( $primary ) );

$replica = show($node{3}, $guid[0]);
info("replica: " .  obj_to_str( $replica ) );

if (! obj_eq($primary, $replica) ) {
    error("Wrong replica for $guid[0] in store on 3");
    fail();
}

info("##### TEST 3: Remote Replica Update");

set($node{3}, $guid[0], "nickname", "s:Sally");

# should only have to idle twice since the update propagates directly
idle_all(\%node, 2);
# what might happen is the replica might be updated again with a 
# stale version, in which case we need the new pub to be propagated again
idle_all(\%node, $NUM_NODES+3);

$primary = show($node{1}, $guid[0]);
info("original: " .  obj_to_str( $primary ) );

$replica = show($node{3}, $guid[0]);
info("replica: " .  obj_to_str( $replica ) );

if (! obj_eq($primary, $replica) ) {
    error("Wrong primary for $guid[0] in store on 1");
    fail();
}
if ($primary->{attrs}->{nickname}->[1] ne "Sally" ) {
    error("Wrong primary for $guid[0] in store on 1 nickname=" .
	  $primary->{attrs}->{nickname}->[0] . ":" .
	  $primary->{attrs}->{nickname}->[1]);
    fail();
}

info("##### TEST 4: Pointer Resolution + Fetching");

$guid[2] = add($node{1});
$guid[3] = add($node{1});
# create a pointer cycle
set($node{1}, $guid[0], "ptr", "guid:$guid[2]");
set($node{1}, $guid[2], "ptr", "guid:$guid[3]");
set($node{1}, $guid[3], "ptr", "guid:$guid[0]");
# another smaller cycle, duplicate guids
set($node{1}, $guid[0], "ptr2", "guid:$guid[2]");
set($node{1}, $guid[2], "ptr2", "guid:$guid[0]");
# create self reference
set($node{1}, $guid[0], "self", "guid:$guid[0]");

idle_all(\%node, $NUM_NODES+5);

foreach my $g (0, 2, 3) {
    $primary = show($node{1}, $guid[$g]);
    info("original: " .  obj_to_str( $primary ) );
    $replica = show($node{3}, $guid[$g]);
    info("replica: " .  obj_to_str( $replica ) );
    if (! obj_eq($primary, $replica) ) {
	error("Wrong replica for $guid[$g] in store");
	dump_store($node{3});
	fail();
    }
}

info("##### TEST 5: Subscription Limiting Pubs");

$guid[4] = add($node{1});
set($node{1}, $guid[4], "x", "f:21");

idle_all(\%node, $NUM_NODES+3);

$primary = show($node{1}, $guid[4]);
info("original: " .  obj_to_str( $primary ) );

%store = store($node{3});
$replica = $store{$guid[4]};
info("replica: " .  obj_to_str( $replica ) );

if ( $replica ) {
    error("Node 3 should not have $guid[4]!");
    fail();
}

#}##

info("##### TEST 6: Primary Migration");

$guid[5] = add($node{1});
$guid[6] = add($node{3});
$temp = show($node{3}, $guid[6]);
my $sid = $temp->{sid};

idle_all(\%node, $NUM_NODES);

migrate($node{1}, $guid[5], $sid);

# migrate_req, migrate_resp, migrate_ack, recv
idle_all(\%node, 4);

$primary = show($node{3}, $guid[5]);
info("new_primary: " .  obj_to_str( $primary ) );
if ( ! $primary || $primary->{replica} ) {
    error("Node 3 isn't the owner of $guid[5]!");
    fail();
}

$replica = show($node{1}, $guid[5]);
info("new_replica: " .  obj_to_str( $replica ) );
if ( ! $replica || !$replica->{replica} ) {
    error("Node 1 is still the owner of $guid[5]!");
    fail();
}

#foreach my $n (keys %node) {
#    info("node $n:\n" . benchmark($node{$n}));
#}

stop_all();
info("SUCCESS");
