#!/usr/bin/perl
#
# Simple Game that runs all nodes synchronously...
#
# -n number of nodes
# -f number of frames to execute
# -i frame interval (ms)
# -D debug files
# -v verbosity

use strict;
require "TestLib.pl";
use Time::HiRes qw( usleep time );
use Getopt::Std;
use vars qw($opt_n $opt_f $opt_i $opt_D $opt_v);

getopts("n:f:i:D:v");

my $NUM_NODES   = defined $opt_n ? $opt_n : 10;
my $FRAME_LIM   = defined $opt_f ? $opt_f : 100;
my $FRAME_INT   = defined $opt_i ? $opt_i : 500;
my $DEBUG_FILES = defined $opt_D ? $opt_D : "";
my $VERBOSITY   = defined $opt_v ? $opt_v : 0;

my $MAX_RANGE   = 79;
my $SUB_RANGE   = 20;

start_bootstrap( "x" => ["int", 0, $MAX_RANGE] );

my %node = start_nodes($NUM_NODES, {},
		       "-l '$FRAME_INT'",
		       "-v '$VERBOSITY'", 
		       "-D '$DEBUG_FILES'",
		       "--measurement",
		       "--log-flush-interval", 1000,
		       "-N", "x", "-A", "id");

###############################################################################

my $FRAMENO = 0;
my $frame_time = 0;
my $slack = 0;

our %node_data = ();
our $id_index  = 0;
our @id_array  = qw( q w e r t y u i o p a s d f g h j k l z x c v b n m );

our $REFRESH_INT = 250;
our $last_refresh = 0;
our $SCREEN = '';

# initialize nodes
foreach my $nid (keys %node) {
    $node_data{$node{$nid}->{sid}} = {};
    InitNode($node{$nid});
}

# run game loop
while ($FRAMENO++ < $FRAME_LIM) {
    my $start_time = time();

    ## display to terminal
    if (time() > $last_refresh + $REFRESH_INT/1000) {
	$last_refresh = time();
	system("clear");
	print $SCREEN;
	print "avg. frame_interval\t$frame_time\n";
    }
    $SCREEN = '';

    ## execute a frame on each node
    foreach my $nid (keys %node) {
	RunFrame($node{$nid});
	Send($node{$nid});
    }

    ## spin-process until next frame
    my $stop_time = $start_time + $FRAME_INT/1000;
    my $slack_total = 0;
    my $slack_count = 0;
    do {
	my $start_recv = time();
	foreach my $nid (keys %node) {
	    Recv($node{$nid});
	}
	$slack_total += time() - $start_recv;
	$slack_count++;
    } while ( time() < $stop_time - $slack );
    $slack = $slack_total/$slack_count;

    my $curr_time = time();
    Sleep( $stop_time - $curr_time ) if ($curr_time < $stop_time);

    $frame_time = time() - $start_time;
}

stop_all();
info("FINISHED");

###############################################################################

sub InitNode {
    my $node = shift;
    my $data = $node_data{$node->{sid}};
    
    $data->{obj} = add($node);

    my $pos = int(rand($MAX_RANGE+1));
    my $min = int($pos - $SUB_RANGE/2);
    $min = 0 if $min < 0;
    my $max = int($pos + $SUB_RANGE/2);
    $max = $MAX_RANGE if $max > $MAX_RANGE;
 
    set($node, $data->{obj}, "x", "int:$pos");
    set($node, $data->{obj}, "id", 
	"char:" . $id_array[$id_index++ % scalar(@id_array)] );
    setin($node, $data->{obj}, 
	  ["x", ">=", "int:$min"], ["x", "<=", "int:$max"]);

    $data->{bullet} = undef;
    $data->{bullet_dir} = undef;
}

sub RunFrame {
    my $node = shift;
    my $data = $node_data{$node->{sid}};

    my $move = rand() >= 0.5 ? "left" : "right";

    my $obj  = show($node, $data->{obj});
    error("missing obj on node " . $node->{sid}) if !$obj;

    my($pos) = $obj->{attrs}->{x}->[1];
    
    if ($move eq "left") {
	if ($pos != 0) { --$pos; }
    } else {
	if ($pos != $MAX_RANGE) { ++$pos; }
    }

    my $min = int($pos - $SUB_RANGE/2);
    $min = 0 if $min < 0;
    my $max = int($pos + $SUB_RANGE/2);
    $max = $MAX_RANGE if $max > $MAX_RANGE;
    
    set($node, $data->{obj}, "x", "int:$pos");
    setin($node, $data->{obj}, 
	  ["x", ">=", "int:$min"], ["x", "<=", "int:$max"]);

    if (!$data->{bullet}) {
	if (rand() > 0.95) {
	    $data->{bullet_dir} = rand() >= 0.5 ? "left" : "right";
	    
	    $data->{bullet} = add($node);
	    set($node, $data->{bullet}, "x", "int:$pos");
	    set($node, $data->{bullet}, "id", "char:-" );
	}
    } else {
	my $bullet = show($node, $data->{bullet});
	error("missing bullet on node " . $node->{sid}) if !$bullet;

	my $bpos = $bullet->{attrs}->{x}->[1];
	if ($data->{bullet_dir} eq "left") {
	    $bpos--;
	} else {
	    $bpos++;
	}
	
	if ($bpos < 0 || $bpos > $MAX_RANGE) {
	    remove($node, $data->{bullet});
	    $data->{bullet} = undef;
	    $data->{bullet_dir} = undef;
	} else {
	    set($node, $data->{bullet}, "x", "int:$bpos");
	}
    }

    #####

    my %store = store($node);

    my @view;
    for (my $i=0; $i<=$MAX_RANGE; $i++) {
	$view[$i] = '.';
    }

    foreach my $guid (keys %store) {
	my $obj = $store{$guid};
	my $id  = $obj->{attrs}->{id}->[1];
	my $pos = $obj->{attrs}->{x}->[1];

	if ($guid eq $data->{obj}) {
	    $view[$pos] = "#";
	} elsif ($guid eq $data->{bullet}) {
	    $view[$pos] = '*';
	} else {
	    $view[$pos] = $id;
	}
    }

    $view[$min] = "[";
    $view[$max] = "]";

    $SCREEN .= join('', @view) . "\n";
}
