////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#define NPATBUFS 20
char *patbufs[NPATBUFS];
#include <config.h>

#ifdef DEBIAN
#include <pcre.h>
#else
#include <pcre/pcre.h>
#endif
#include <sys/types.h>
#include <dirent.h>
#include <hash_map.h>

struct eq_string {
    bool operator() (const string& s1, const string& s2) const {
	return strcmp (s1.c_str (), s2.c_str ()) == 0;
    }
};

struct hash_string_1 { 
    size_t operator() (const string& x) const {
	return hash<const char *> () (x.c_str ());
    }
};

struct eqcharptr {
    bool operator() (const char *a, const char *b) const {
	return strcmp (a, b) == 0;
    }
};
typedef hash_map<const char *, pcre *, hash<const char *>, eqcharptr> PatMap;
typedef PatMap::iterator PatMapIter;

char line[1024];
PatMap patterns;

pcre *get_re (const char *pat)
{   
    pcre *re;
    const char *error;
    int erroffset;

    re = pcre_compile (pat, 0, &error, &erroffset, NULL);
    if (!re) {  
	fprintf (stderr, "error in compiling pattern: %s\n", error);
	exit (1);
    }

    return re;
}

void match_capture (pcre *re, const char *str, int nmatches, vector<string> *ret)
{   
    int rc;
    int ovector [60];

    rc = pcre_exec (re, NULL, str, strlen (str), 0, 0, ovector, 60);
    if (rc < 0) { 
	if (rc == PCRE_ERROR_NOMATCH)
	    return;
	fprintf (stderr, "error during matching\n");
	exit (1);
    }

    for (int i = 1; i < nmatches + 1; i++) {
	int si = ovector [2*i], ei = ovector[2*i + 1];

	string s;
	s.append (str + si, ei - si);
	ret->push_back (s);
    }
}

// captured patterns are stored in the buffers 
// it is assumed that the buffers contain enough space
bool match_capture (pcre *re, const char *str, int nm, char **buf)
{
    int rc;
    int ovector [60];

    rc = pcre_exec (re, NULL, str, strlen (str), 0, 0, ovector, 60);
    if (rc < 0) {
	if (rc == PCRE_ERROR_NOMATCH)
	    return false;
	fprintf (stderr, "error during matching\n");
	exit (1);
    }

    for (int i = 1; i < nm + 1; i++) {
	int si = ovector [2*i], ei = ovector[2*i + 1];

	*(buf[i - 1]) = '\0';
	strncat (buf[i - 1], str + si, ei - si);
    }
    return true;
}

pcre *get_cached_pattern (const char *pat) 
{
    PatMapIter it = patterns.find (pat);
    pcre *re = NULL;
    if (it == patterns.end ()) {
	re = get_re (pat);
	patterns.insert (PatMap::value_type (strdup (pat), re));
    }
    else 
	re = it->second;

    return re;
}

bool match_capture (const char *pat, const char *str, int nm, char **buf)
{    
    return match_capture (get_cached_pattern (pat), str, nm, buf);
}

bool matches (pcre *re, const char *str)
{
    int rc;
    int ovector [60];
    rc = pcre_exec (re, NULL, str, strlen (str), 0, 0, ovector, 60);
    if (rc < 0) {
	if (rc == PCRE_ERROR_NOMATCH)
	    return false;
	else {
	    fprintf (stderr, "error during matching\n");
	    exit (1);
	}
    }
    return true;
}

bool matches (const char *pat, const char *str) 
{
    return matches (get_cached_pattern (pat), str);
}

void glob (pcre *re, vector<string> *files)
{
    DIR *dir = opendir (".");
    if (!dir) {
	perror ("opendir");
	exit (1);
    }

    struct dirent *ent;
    while ((ent = readdir (dir))) {
	if (matches (re, ent->d_name))
	    files->push_back (ent->d_name);
    }
    closedir (dir);
}

void glob (const char *pat, vector<string> *files) 
{
    return glob (get_cached_pattern (pat), files);
}

struct less_string {
    bool operator () (const string& a, const string& b) const {
	return a < b;
    }
};

FILE *open_log (string& str) 
{
    FILE *fp = NULL;

    if (strstr (str.c_str (), ".gz")) {
	string s = "/usr/bin/gunzip -c ";
	s.append (str);
	fp = popen (s.c_str (), "r");
    }
    else {
	fp = fopen (str.c_str (), "r");
    }
    return fp;
}

void close_log (FILE *fp, string& str)
{
    if (strstr (str.c_str (), ".gz"))
	pclose (fp);
    else
	fclose (fp);
}

void OpenLogs (vector<FILE *>& vec, vector<string>& files)
{
    sort (files.begin (), files.end (), less_string ());

    for (vector<string>::iterator it = files.begin (); it != files.end (); ++it)
	{
	    FILE *fp = open_log (*it);
	    if (!fp) {
		perror ("fopen or popen");
		exit (1);
	    }

	    vec.push_back (fp);
	}
}

void OpenLogs (vector<FILE *>& vec, char *pat)
{
    vector<string> files;
    glob (pat, &files);
    OpenLogs (vec, files);
}

void CloseLogs (vector<FILE *>& vec, vector<string>& files)
{
    int index = 0;
    for (vector<string>::iterator it = files.begin (); it != files.end (); ++it, ++index)
	{
	    if (strstr (it->c_str (), ".gz")) {
		pclose (vec[index]);
	    }
	    else {
		fclose (vec[index]);
	    }
	}

    vec.clear ();
}

void CloseLogs (vector<FILE *>& vec, char *pat)
{
    vector<string> files;
    glob (pat, &files);
    CloseLogs (vec, files);
}

void show_progress (float done, float total) 
{
    if (total < 1.0e-5)
	return;

    fprintf (stderr, "%.2f %%done\r", (done * 100.0 / total));
}

double GetMinStart (vector<FILE *> logs1, vector<FILE *> logs2)
{
    vector<double> times;
    vector<FILE *> *arr[2];

    arr[0] = &logs1; arr[1] = &logs2;

    for (int i = 0; i < 2; i++) {
	for (vector<FILE *>::iterator it = arr[i]->begin (); it != arr[i]->end (); ++it) {
	    if (!fgets (line, sizeof (line), *it))
		Debug::die ("no lines in log");

	    if (!match_capture ("^(\\d+\\.\\d+)", line, 1, (char **) patbufs)) 
		Debug::die ("no time found in log line");

	    double f;
	    sscanf (patbufs[0], "%lf", &f);
	    times.push_back (f);
	}
    }

    double min = times[0];
    for (vector<double>::iterator it = times.begin (); it != times.end (); ++it)
	if (*it < min)
	    min = *it;

    return min;
}

void ScrollUntilTime (double start, vector<FILE *>& logs, vector<string> *res)
{
    char *p;

    for (vector<FILE *>::iterator it = logs.begin (); it != logs.end (); ++it) {
	fprintf (stderr, "."); fflush (stderr);
	while ((p = fgets (line, sizeof (line), *it))) { 
	    match_capture ("^(\\d+\\.\\d+)", line, 1, (char **) patbufs);
	    double f;
	    sscanf (patbufs[0], "%lf", &f);
	    if (f < start)
		continue;
	    else
		break;
	}

	if (p)
	    res->push_back (line);
    }
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
