#!/usr/bin/perl
use strict;
use Travertine;
use Getopt::Std;
use Statistics::Descriptive;
use vars qw($opt_S $opt_s $opt_l $opt_w);

getopts("S:s:l:w:");

our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 120;
our $LENGTH   = defined $opt_l ? $opt_l : 600;
our $SLOWDOWN = defined $opt_w ? $opt_w : 1;
our @DIMS = ("HDR_ABS", "TMP_ABS", "TOT_ABS", "AVG_SIZE");

our %stats = ProcessInput(@ARGV);

print sprintf ("%-14sMean\tMedian\tStd-dev\n", "Stat");

foreach my $s (@DIMS) {
    print sprintf ("%-14s", $s);
    foreach my $f (
	[ "MEAN",   sub { $_[0]->mean() } ],
	[ "MEDIAN", sub { $_[0]->median() } ],
	[ "STDDEV", sub { $_[0]->standard_deviation() } ]
    )
    {
	print sprintf ("%.3f\t", &{$f->[1]} ($stats{$s}));
    }
    print "\n";
}
	
sub ProcessInput(@)
{
    my @files = @_;
    my($start, $end);
    $start = $START;
    my %stats = ();

    foreach my $s (@DIMS) {
	$stats{$s} = new Statistics::Descriptive::Full ();
    }
    
    foreach my $f (@files) {
        if ($f =~ /\.gz$/) {
	    open(F, "zcat $f |") || die "can't pipe $f";
        } else {
	    open(F, "<$f") || die "can't open $f";
        }

	tinfo ("$f...");

	while (<F>) {
	    # print;
	    chomp $_;
	    m/(\d+\.\d+)\t(\d+)\t(\d+)\t(\d+)\t(\d+)\t(\d+)/ or warn "bad line: $_" && next;
	    
	    my $time = $1;
	    my $n_temp = $2;
	    my $size_temp  = $3;
	    my $n_tot = $4;
	    my $size_tot = $5;
	    my $header = $6;

	    my $total = $header + $size_tot;
	    next if ($total <= 0);
		
	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }
	    
	    $stats{'HDR_ABS'}->add_data ($header);
	    $stats{'TOT_ABS'}->add_data ($total);
	    $stats{"TMP_ABS"}->add_data ($size_temp);

	    if ($n_tot > 0) { 
		$stats{"AVG_SIZE"}->add_data ($size_tot / $n_tot);
	    }
	}	
    }

    return %stats;
}
