#!/usr/bin/perl
#
# Converts a topology file to a waspnet script, including setting up ip
# aliases. This generates a waspnet script on stdout. Run the script as root.
#
# INPUT FORMAT:
#
# node <id> <host> <vaddr> <out> <in>
# ...
#
# <id1>,<id2> <delay>
#
# where:
#
# <id>    - unique identifier for each vhost
# <host>  - real name/addr of the host (attached to the desired interface)
# <vaddr> - virtual address for this vhost
# <out>   - outbound bwidth cap in bps
# <in>    - inbound bwidth cap in bps
# <delay> - RTT between <id1> and <id2> in msec
#
###############################################################################

use strict;
use Getopt::Std;

use vars qw($opt_d $opt_q $opt_i $opt_y $opt_n $opt_v 
	    $opt_F $opt_O $opt_I $opt_L);

getopts("d:q:i:y:n:v:FO:I:L:");

our $MAX_NODES = $opt_n;
our $MAX_VSERVERS = $opt_v;
our $DIR = $opt_d || ".";
our $FAIL_FAST = !$opt_F;
our $SLOW_DOWN = $opt_y || 1;

our $QUEUELEN  = $opt_q || 5000;
our $INBOUND   = defined $opt_I ? 1000*$opt_I : undef;
our $OUTBOUND  = defined $opt_O ? 1000*$opt_O : undef;
our $LATENCY   = defined $opt_L ? $opt_L : undef;

our $TOPO_FILE = $ARGV[0];
our $LOCALNAME = $ARGV[1];

our $USAGE =<<EOT;
usage: Topology2Waspnet.pl [options] <topology_file> <iface_ip>

    <topology_file> - topology file generated by P2PSimGraphToVHostLats.pl
                      file must contain IP addresses, not dns addresses.
    <iface_ip>      - interface on which to bind virtual hosts

options:
    -d <dir>        - dir containing waspnet.o and waspnetctl
    -i <iface>      - force use of this interface (else lookup by <iface_ip>)
    -n <count>      - only use first <count> nodes in topology
    -v <count>      - only use first <count> vservers for each node in topology
    -y <slowdown>   - dilate latencies/bwidths by <slowdown>
    -F              - do not insert fail fast escapes in script
    -q <msec>       - max queuing delay (defines queue length)
    -O <kbps>       - override outbound bwidth of all vhosts (kbps)
    -I <kbps>       - override inbound bwidth of all vhosts (kbps)
    -L <rtt_msec>   - override rtt latency between all vhost pairs (msec)
EOT

die $USAGE if !$TOPO_FILE || !$LOCALNAME;

our $LOCALADDR = ResolveIP($LOCALNAME);

our $DEV = $opt_i || get_dev($LOCALADDR);
open(I, "<$TOPO_FILE") or die "can't open $TOPO_FILE: $!";

# map from $id => [$vaddr,$in,$out]
our %ID_MAP;
# each of the vservers on this host
our %MY_IDS;
our @MY_IDS;

do_init_filters();

my %VHOST_COUNT = ();
my @OCTETS = (undef, undef, undef, undef);

my $node_phase = 1;
while (<I>) {
    chomp;
    
    if (/^node\s+(\d+)\s+([^\t ]+)\s+([^\t ]+)\s+([^\t ]+)\s+([^\t ]+)/) {
	my ($id,$addr,$vaddr,$out,$in) = ($1,$2,$3,$4,$5);

	if ($MAX_NODES && !$VHOST_COUNT{$addr} && keys %VHOST_COUNT >= $MAX_NODES) {
	    next;
	}
	if ($MAX_VSERVERS && $VHOST_COUNT{$addr} >= $MAX_VSERVERS) {
	    next;
	}
	$VHOST_COUNT{$addr}++;

	$out = $OUTBOUND if (defined $OUTBOUND);
	$in = $INBOUND if (defined $INBOUND);

	$addr = ResolveIP($addr);
	
	die "dup id $id? $_" if $ID_MAP{$id};
	$ID_MAP{$id} = [$vaddr,$in,$out];

	if ($addr eq $LOCALADDR) {
	    push @MY_IDS, $id if !$MY_IDS{$id};
	    $MY_IDS{$id} = 1;
	}

	# figure out range of virtual network
	my @octets = split(/\./, $vaddr);
	for (my $i=0; $i<4; $i++) {
	    if (!defined $OCTETS[$i]) {
		$OCTETS[$i] = $octets[$i];
	    }
	    $OCTETS[$i] = -1 if $OCTETS[$i] != $octets[$i];
	}

    } elsif (/^(\d+),(\d+)\s+([\d\.]+)/) {
	if ($node_phase) {
	    do_host_filters();
	    $node_phase = 0;
	}

	my ($a, $b, $lat) = ($1, $2, $3);

	my $real_lat = int($SLOW_DOWN*$lat/2);
	$real_lat = $LATENCY if (defined $LATENCY);

	next if (!$ID_MAP{$a} || !$ID_MAP{$b});
	
	if ($MY_IDS{$a}) {
	    do_lat_filter($a, $b, $real_lat);
	}
	if ($MY_IDS{$b}) {
	    do_lat_filter($b, $a, $real_lat);
	}
    }
}

###############################################################################

sub do_init_filters()
{
    comment("restarting...");
    reset_all();
}

sub do_host_filters()
{
    # sanity check
    warn "a lot of vservers (" . (scalar keys %MY_IDS) . ")"
	if scalar keys %MY_IDS > 32;

    # add route for virtual network
    my $mask = 0;
    my @net = ();
    my $done = 0;
    for (my $i=0; $i<4; $i++) {
	if ($OCTETS[$i] < 0) {
	    $done = 1;
	}
	push @net, ($done ? "0" : $OCTETS[$i]);
	$mask += 8 if (!$done);
    }
    my $net = join(".", @net);
    ex("/sbin/route del -net $net/$mask >/dev/null 2>&1");
    ex("/sbin/route add -net $net/$mask dev $DEV || exit -1");

    my $index = 0;
    foreach my $v (@MY_IDS) {
	die "missing id $v" if !$ID_MAP{$v};
	my ($vaddr,$in,$out) = @{$ID_MAP{$v}};

	my $real_in  = int($in/$SLOW_DOWN);
	my $real_out = int($out/$SLOW_DOWN);

	comment("local rules for $v ($vaddr)");

	ex("/sbin/ifconfig $DEV:$index down >/dev/null 2>&1");
	ex("/sbin/ifconfig $DEV:$index $vaddr || exit -1");
	ex("/sbin/route del -host $vaddr >/dev/null 2>&1");
	ex("/sbin/route add -host $vaddr dev $DEV:$index || exit -1");
	
	if ($out > 0) {
	    add("send", $vaddr, undef, "-rate $real_out -qlen $QUEUELEN");
	}
	if ($in > 0) {
	    add("recv", undef, $vaddr, "-rate $real_in -qlen $QUEUELEN");
	}

	$index++;
    }
}

sub do_lat_filter($$$)
{
    my ($from, $to, $delay) = @_;

    my $saddr = $ID_MAP{$from}->[0];
    my $daddr = $ID_MAP{$to}->[0];

    comment("latency from $from to $to");
    add("drule", $saddr, $daddr, "-delay $delay");
}

###############################################################################

sub reset_all()
{
    ex("rmmod waspnet >/dev/null 2>&1");
    ex("insmod $DIR/waspnet.o || exit -1");
}

sub comment($)
{
    #print("# $_[0]\n");
    print("echo \"# $_[0]\" 1>&2\n");
}

sub ex($)
{
    print("echo \"$_[0]\" 1>&2\n");
    print("$_[0]\n");
}

sub add($$$$@)
{
    my $type = shift;
    my $from = shift;
    my $to   = shift;
    my $cmd  = shift;
    my $no_fail_fast = shift;

    $from = "0.0.0.0/0" if !$from;
    $to   = "0.0.0.0/0" if !$to;

    die "bad type $type" if $type !~ /^(send|recv|drule)$/;

    my $cmdline = "$DIR/waspnetctl -s -$type -src $from -dst $to $cmd";
    if ($FAIL_FAST && !$no_fail_fast) {
	$cmdline .= " || exit -1";
    }
    print("echo \"$cmdline\" 1>&2\n");
    print("$cmdline\n");
}

sub get_dev($)
{
    my $addr = shift;
    my $curr_dev;

    foreach my $line (split /\n/, `/sbin/ifconfig`) {
	if ($line =~ /^(\w+)/) {
	    $curr_dev = $1;
	} elsif ($line =~ /^\s+inet addr:(\d+\.\d+\.\d+\.\d+)/) {
	    my ($ip) = ($1);
	    if ($ip eq $addr) {
		die "don't have curr_dev?" if !$curr_dev;
		return $curr_dev;
	    }
	}
    }
    
    die "couldn't find dev for $addr";
}

sub ResolveIP($$) {
    my ($host) = @_;
    my $ip = undef;

    if ($host =~ /\d+\.\d+\.\d+\.\d+/) {
	return $host;
    }

    my (undef,undef,undef,undef,@addrs) = gethostbyname($host);
    if ($addrs[0]) {
	$ip = join(".",unpack('C4',$addrs[0]));
    }
    else {
	# xxx hacky special case
	if ($host eq 'localhost' || $host eq 'localhost.localdomain') {
	    return "127.0.0.1";
	}

	print STDERR "lookup failed for ($host)\n";
    }
    return $ip ? $ip : $host;
}
