# Drives the processing of a number of scripts for Colyseus
package ScriptDriver;
use strict;
use lib "$ENV{HOME}/Colyseus/run";
use lib "$ENV{HOME}/Colyseus/scripts";
use Travertine;
use ResultsConf;

###############################################################################

sub RedirectOP { 
    my $cmd = shift;
    my $fh = shift;

    tinfo ("RedirectOP: $cmd");
    open F, "$cmd | " or die $!;
    while (<F>) {
	print $fh $_;
    }
    close F;
}

my @DEFAULT_SCRIPTS = (
    "CSBC",
    "CSBCTimeSeries",
    "CountFizzableUpdates",
    "ExpMsgLog",
    "ExpMsgLogDetail",
    "ExpMsgLogIndividual",
    "FrameRates",
    "HopStats",
    "LoadPrediction",
    "LoadPredictionTimeSeries",
    "MercBwidthBreakdown",
    "ReplicaConsistency",
    "ReplicaConsistencyCombine",
    "ReplicaConsistencyDetailed",
    "ReplicaConsistencyDetailedSummary",
    "ReplicaConsistencyDistance",
    "SubDiscoveryTime",
    "SubDiscoveryTimeDetailed",
    "TimeSeries",
);

sub GetScripts {
    my $opt_scripts = shift;
    my $opt_exactmatch = shift;

    if (!$opt_scripts or $opt_scripts eq "") {
	return @DEFAULT_SCRIPTS;
    }
    my %pats = map { $_ => 1 } split (/,/, $opt_scripts);
    if ($opt_exactmatch) {
	return grep { exists $pats {$_} } @DEFAULT_SCRIPTS;
    }

    my @scripts = ();
    foreach my $pat (keys %pats) { 
	foreach my $sc (@DEFAULT_SCRIPTS)  {
	    push @scripts, $sc if ($sc =~ /$pat/);	    
	}
    }
    return @scripts;
}

sub GetScriptInstance {
    my $sc = shift;

    my $class = "Colyseus::Script::$sc";
    my $object = eval { 
	no strict;
	my $sobj  = $class->new ();
	$sobj;
    };

    if ($@) {
	Travertine::twarn "Script $sc not implemented? $@";
	return undef;
    }

    return $object;
}
package Colyseus::Expt;  # encapsulate state about an experiment
use Travertine;

sub new {
    my $class = shift;
    my $name   = shift;
    my $logdir = shift;
    my $outdir = shift;
    
    my %opts = @_;
    
    my $self = { 
	name => $name,
	logdir => $logdir,
	outdir => $outdir,
    };
    
    $self->{opts} = {};
    my %defaults = (
	-exactmatch => 1,
	-scripts => undef,
	-exptset => undef,
    );
    foreach (keys %defaults) { 
	$self->{$_} = exists $opts{$_} ? $opts{$_} : $defaults{$_};
    }

    bless ($self, $class);
    return $self;
}

sub RunScripts {
    my $self = shift;
    my @scripts = ScriptDriver::GetScripts ($self->{-scripts}, $self->{-exactmatch});
    
    tinfo "Running scripts [@scripts]";
    foreach my $sc (@scripts) {
	my $sobj = ScriptDriver::GetScriptInstance ($sc);
	next if (!defined $sobj);

	Travertine::tinfo "Running script $sc";
	$sobj->Process ($self);
    }
}

package Colyseus::ExptSet;
use Travertine;

sub new {
    my $class  = shift;
    my $name   = shift;
    my $explst = shift;    # ref to array containing tuples of <name, dir>
    my $outdir = shift;
    my %opts   = @_;

    my $self = {
	name => $name,
	outdir => $outdir,
	opts => {},
    };
    
    my %defaults = (
	-exactmatch => 1,
	-scripts => undef,
	-append => 0,
	-order => "scripts_first",     # run all scripts for each expt  OR  run one script on all expts, first
    );

    foreach (keys %defaults) {
	$self->{$_} = exists $opts{$_} ? $opts{$_} : $defaults{$_};
    }
	
    bless ($self, $class);

    $self->MakeExptList ($explst);
    return $self;
}

sub MakeExptList {
    my $self = shift;
    my $explst = shift;
    
    $self->{explst} = [];
    
    foreach my $pair (@{$explst}) {
	my $name = $pair->[0];
	my $dir  = $pair->[1];

	push @{$self->{explst}}, 
	new Colyseus::Expt (
	    $name, $dir, $self->{outdir},
	    -exactmatch => $self->{-exactmatch}, 
	    -scripts => $self->{-scripts},
	    -exptset => $self
	);

	
    }
}

sub RunScripts {
    my $self = shift;
    my @scripts = ScriptDriver::GetScripts ($self->{-scripts}, $self->{-exactmatch});
    
    tinfo "Running scripts [@scripts]";
    # for every script, run a InitExptSet method, before running on all the experiments
    foreach my $sc (@scripts) {
	$sc = ScriptDriver::GetScriptInstance ($sc);
	$sc->InitExptSet ($self) if ($sc);
    }

    if ($self->{-order} eq 'scripts_first') {   # all scripts get to go first
	foreach my $expt (@{$self->{explst}}) {
	    foreach my $sc (@scripts) { 
		# check for signals here??? 
		if ($sc) {
		    tinfo "* Running script $sc->{name} [all_scripts_at_once]";
		    $sc->Process ($expt);
		}
	    }
	}
    }
    else {
	foreach my $sc (@scripts) { 
	    foreach my $expt (@{$self->{explst}}) {
		# check for signals here???
		if ($sc) {
		    tinfo "* Running script $sc->{name} [all_expts_at_once]";
		    $sc->Process ($expt);
		}
	    }
	}
    }
}

package Colyseus::Script;
use Travertine;

sub new {
    my $class = shift;
    my $self = {};
    bless ($self, $class);
    
    $self->{name} = $self->name ();    
    return $self;
}

sub name { "abstract_script"; }

# Do some common processing or initialization here
sub Process { 
    my $self = shift;
    my $expt = shift;

    $self->{start} = ResultsConf::GetStartTime ($expt->{logdir});
    $self->{slowdown} = ResultsConf::GetSlowDown ($expt->{logdir});
    
    $self->{common_args} = "-S $self->{start} -s $ScriptDriver::SKIPTIME -l $ScriptDriver::LENGTH";
    $self->{common_args} .= " -w $self->{slowdown}" if ($self->{slowdown} > 1.0);
    
    ############# begin stuff that might be better moved into the Expt class 
    #
    # resdir = topdir[/exptset_name]/script_name
    $self->{resdir} = $expt->{outdir};
    $self->{exptset_name} = (defined $expt->{-exptset} ? $expt->{-exptset}->{name} : undef);
    
    if (defined $expt->{-exptset}) {
	$self->{resdir} .= "/$self->{name}/$expt->{-exptset}->{name}";
    }
    else {
	$self->{resdir} .= "/$self->{name}";
    }

    ppsystem ("mkdir -p $self->{resdir}");
    
    $self->{resfile} = "$self->{resdir}/$self->{name}.out";
    if (!defined $expt->{-exptset}) { 
	unlink $self->{resfile};
    }
    #
    ############## end
    
    Travertine::tinfo "### Expt=$expt->{name} ExptSet=$self->{exptset_name}";
    $self->Execute ($expt);
}

sub Execute { 
    Travertine::tdie "Colyseus::Script::Execute should not be called";
}

sub InitExptSet { 
    my $self = shift;
    my $exptset = shift;

    unless ($exptset->{-append}) { 
	my $resdir = "$exptset->{outdir}/$self->{name}/$exptset->{name}";
	ppsystem ("rm -rf $resdir");
    }    
}

package Colyseus::Script::CSBC;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "csbc" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    chdir $expt->{logdir};

    $self->{resdir} .= "/$expt->{name}";

    psystem("rm -rf $self->{resdir}/client-server/");
    psystem("mkdir -p $self->{resdir}/client-server/");
    
    # my $cmd = "$ENV{HOME}/Colyseus/scripts/ClientServerSim.pl -S $start -s $ScriptDriver::SKIPTIME -l $ScriptDriver::LENGTH ";
    my $cmd = "$ENV{HOME}/Colyseus/scripts/clisersim $self->{common_args}";

    psystem "$cmd > $self->{resdir}/client-server/client-server.out";
    psystem "rm -rf $self->{resdir}/bcast";
    psystem "mkdir -p $self->{resdir}/bcast";

    $cmd = "$ENV{HOME}/Colyseus/scripts/clisersim $self->{common_args} -b -o $self->{resdir}/bcast/";
    psystem "$cmd";
}

package Colyseus::Script::ExpMsgLog;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "bwidth"; }

sub Execute {
    my $self = shift;
    my $expt = shift;
    my $exptset = $expt->{-exptset};
    my $fh = new IO::File (">> $self->{resfile}");
    print $fh "Experiment = $expt->{name}\n", "="x80, "\n";
	
    my @files;
    if ($ScriptDriver::AGGLOGS) {
	@files = glob "$expt->{logdir}/AggregateMessage*";
    }
    else {
	@files = glob "$expt->{logdir}/Message*";
    }

    my $cmd = "$ENV{HOME}/Colyseus/scripts/ExpMsgLogTmSeries.pl $ScriptDriver::LOGARGS $self->{common_args} -e 1 -t proto -T ";
    ScriptDriver::RedirectOP ("$cmd @files", $fh);
    close $fh;
}

package Colyseus::Script::FrameRates;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "framerates" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my $fh = new IO::File (">> $self->{resfile}");
    print $fh "Experiment = $expt->{name}\n", "="x80, "\n";

    chdir ($expt->{logdir});
    my $cmd = "$ENV{HOME}/Colyseus/scripts/FrameRates.pl $self->{common_args}";

    ScriptDriver::RedirectOP ($cmd, $fh);
    close $fh;
}

package Colyseus::Script::TimeSeries;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "tseries" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my @files;
    my $bwscript = "$ENV{HOME}/Colyseus/scripts/MessageLogTimeSeries.pl";
    if ($ScriptDriver::AGGLOGS) {
	$bwscript = "$ENV{HOME}/Colyseus/scripts/AggMsgLogTimeSeries.pl";
	@files = glob("$expt->{logdir}/AggregateMessage*");
    } else {
	@files = glob "$expt->{logdir}/Message*";
    }
    my $pipe = "";

    my @dirs = ( 
	"om-bwidth",
	"merc-bwidth",
	"routing-bwidth",
	"matching-bwidth",
	"total-bwidth",
	"frame-time",
	"numreplicas"
    );

    $self->{resdir} .= "/$expt->{name}";

    foreach my $d (@dirs) {
	psystem "rm -rf $self->{resdir}/$d";
	psystem "mkdir -p $self->{resdir}/$d"
    }

    chdir $expt->{logdir};
    
    foreach my $f (@files) {
	$f =~ /(\d+\.\d+\.\d+\.\d+\:\d+)/;
	my $id = $1;
	die "bad filename $f" if !$id;

	my $shopts = $self->{common_args};
	my $bwopts = "$shopts -e 0.1 -t proto";
	my $field = 0;

	$field = 1 if (!$ScriptDriver::AGGLOGS);

	if ($self->{slowdown} != 1) {
	    $pipe = "$ENV{HOME}/Colyseus/scripts/TimeFilter.pl -S $self->{start} -w $self->{slowdown} -f $field |";		
	}

	my $cat = $f =~ /\.gz$/ ? "zcat" : "cat";

	psystem ("$cat $f | $pipe $bwscript $bwopts -o /dev/stdin > $self->{resdir}/om-bwidth/$id.log");
	psystem ("$cat $f | $pipe $bwscript $bwopts -m /dev/stdin > $self->{resdir}/merc-bwidth/$id.log");
	psystem ("$cat $f | $pipe $bwscript $bwopts -r /dev/stdin > $self->{resdir}/routing-bwidth/$id.log");
	psystem ("$cat $f | $pipe $bwscript $bwopts -c /dev/stdin > $self->{resdir}/matching-bwidth/$id.log");
	psystem ("$cat $f | $pipe $bwscript $bwopts /dev/stdin > $self->{resdir}/total-bwidth/$id.log");

	#$f = (glob("$dir/QuakeStatsLog.$id.log*"))[0];
	#die if !$f;
	#my $cat = $f =~ /\.gz$/ ? "zcat" : "cat";
	#psystem("$cat $f | $pipe cut -f1,4 | ~/Colyseus/scripts/TimeFilter.pl $shopts > $outdir/$out/frame-time/$exp/$id.log");

	my $of = (glob("$expt->{logdir}/ReplicaLifetimeLog.$id.log*"))[0];
	die if !$of;
	psystem("$cat $of | $pipe $ENV{HOME}/Colyseus/scripts/ReplicaCountTimeSeries.pl $shopts /dev/stdin > $self->{resdir}/numreplicas/$id.log");
	#psystem("cat ReplicaLifetimeLog.$id.log | $pipe ~/Colyseus/scripts/ReplicaCountTimeSeries.pl $shopts -d /dev/stdin > $outdir/$out/numreplicas-detailed/$exp/$id.log");
    }

}

package Colyseus::Script::ExpMsgLogDetail;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "bwidth-detail" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my $fh = new IO::File (">> $self->{resfile}");
    print $fh "Experiment = $expt->{name}\n", "="x80, "\n";

    my @files;
    if (!$ScriptDriver::AGGLOGS) {
	@files = glob "$expt->{logdir}/Message*";
    } else {
	@files = glob "$expt->{logdir}/AggregateMessage*";
    }
	
    my $cmd;
    if (!$ScriptDriver::AGGLOGS) {
	$cmd = "$ENV{HOME}/Colyseus/scripts/MessageLogAggregate.pl -b $self->{common_args}";
    }
    else {
	$cmd = "$ENV{HOME}/Colyseus/scripts/AggMsgLogAggregate.pl $self->{common_args}";
    }

    tinfo ("$cmd");
    ScriptDriver::RedirectOP ("$cmd @files", $fh);
    close $fh;
}

package Colyseus::Script::ExpMsgLogIndividual;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "bwidth-individual" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my @files;
    if ($ScriptDriver::AGGLOGS) {
	@files = glob "$expt->{logdir}/AggregateMessage*";
    }
    else {
	my @files = glob "$expt->{logdir}/Message*";
    }
    
    $self->{resdir} .= "/$expt->{name}";
    
    psystem ("rm -rf $self->{resdir}");
    psystem ("mkdir -p $self->{resdir}");
    
    foreach my $f (@files) {
	$f =~ /(\d+\.\d+\.\d+\.\d+\:\d+)/;
	my $id = $1;
	    
	psystem ("$ENV{HOME}/Colyseus/scripts/ExpMsgLogTmSeries.pl $self->{common_args} $ScriptDriver::LOGARGS -e 1 -t proto -T $f > $self->{resdir}/$id.log");
    }
}

package Colyseus::Script::HopStats;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "hops" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my $fh = new IO::File (">> $self->{resfile}");
    print $fh "Experiment = $expt->{name}\n", "="x80, "\n";

    my @files = glob "$expt->{logdir}/DiscoveryLat*";
	
    my $cmd = "$ENV{HOME}/Colyseus/scripts/HopStats.pl $self->{common_args}";
	
    tinfo ("$cmd");
    ScriptDriver::RedirectOP ("$cmd @files", $fh);
    close $fh;
}

package Colyseus::Script::CSBCTimeSeries;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "csbc" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my @files;
    if ($ScriptDriver::AGGLOGS) {
	@files = glob "$expt->{logdir}/AggregateMessage*";
    }
    else {
	my @files = glob "$expt->{logdir}/Message*";
    }

    $self->{resdir} .= "/$expt->{name}";
    psystem "mkdir -p $self->{resdir}";

    my @tp = ('client-server', 'bcast');
    foreach my $f (@tp) { 
	if (!-d "$self->{resdir}/$f") { 
	    twarn "Script::CSBC should be run before Script::CSBCTimeSeries";
	    return;
	}
	my $nl = `ls $self->{resdir}/$f | wc -l`;
	chomp $nl;
	if ($nl == 0) { 
	    twarn "No result files in $self->{resdir}/$f? Script::CSBC should be run before Script::CSBCTimeSeries";
	    return;
	}
    }

    foreach my $f (@tp) {
	open (F, ">> $self->{resdir}/$f.out") || die $!;
	print F "Experiment = $expt->{name}\n", "="x80, "\n";
	close (F);

	my ($type) = ($f =~ /(.)/);
	my @files = glob "$self->{resdir}/$f/*";	    

	my $cmd = "$ENV{HOME}/Colyseus/scripts/CSBMsgLogTmSeries.pl -t $type -s 0 -e 1 -T -w $self->{slowdown}";
	psystem "$cmd @files >> $self->{resdir}/$f.out";
    }
}

package Colyseus::Script::CountFizzableUpdates;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "fizzable" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    chdir $expt->{logdir};

    # my $cmd = "$ENV{HOME}/Colyseus/scripts/ClientServerSim.pl -S $start -s $SKIPTIME -l $LENGTH ";
    my $cmd = "$ENV{HOME}/Colyseus/scripts/clisersim -N $self->{common_args}";
    psystem "$cmd > $self->{resdir}/fizzable.log";
}

package Colyseus::Script::SubDiscoveryTime;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "latency" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my $fh = new IO::File (">> $self->{resfile}");
    print $fh "Experiment = $expt->{name}\n", "="x80, "\n";

    my @files = glob "$expt->{logdir}/DiscoveryLat*";
	
    my $cmd = "$ENV{HOME}/Colyseus/scripts/nsd1 $self->{common_args} -o $expt->{logdir} ";
    psystem ("$cmd @files");
	
    $cmd = "$ENV{HOME}/Colyseus/scripts/NewSubDiscoveryTime2.pl $ScriptDriver::LOGARGS $self->{common_args} -n ";
    ScriptDriver::RedirectOP ("$cmd $expt->{logdir}/aliases.map $expt->{logdir}/discov.log.gz", $fh);
    close $fh;
}

package Colyseus::Script::SubDiscoveryTimeDetailed;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "latency-detailed" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my $fh = new IO::File (">> $self->{resfile}");
    print $fh "Experiment = $expt->{name}\n", "="x80, "\n";

    my @files = glob "$expt->{logdir}/DiscoveryLat*";
	
    my $cmd = "$ENV{HOME}/Colyseus/scripts/nsd1 $self->{common_args} -o $expt->{logdir} ";
    psystem ("$cmd @files");
	
    $cmd = "$ENV{HOME}/Colyseus/scripts/NewSubDiscoveryTime2.pl -a $ScriptDriver::LOGARGS $self->{common_args} -n ";
    ScriptDriver::RedirectOP ("$cmd $expt->{logdir}/aliases.map $expt->{logdir}/discov.log.gz", $fh);
    close $fh;
}

package Colyseus::Script::MercBwidthBreakdown;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "merc-bwidth" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    my @files;
    my $bwscript = "$ENV{HOME}/Colyseus/scripts/MessageLogTimeSeries.pl";
    if ($ScriptDriver::AGGLOGS) {
	$bwscript = "$ENV{HOME}/Colyseus/scripts/AggMsgLogTimeSeries.pl";
	@files = glob("$expt->{logdir}/AggregateMessage*");
    } else {
	@files = glob "$expt->{logdir}/Message*";
    }
    my $pipe = "";

    my @dirs = ( 
	"pub-send",
	"sub-send",
	"pub-route",
	"sub-route",
	"pub-linear",
	"sub-linear",
	"match",
	#"merc-bwidth",
	#"routing-bwidth",
	#"matching-bwidth",
	#"total-bwidth",
	#"frame-time",
	#"numreplicas",
	#"numreplicas-detailed",
    );

    $self->{resdir} .= "/$expt->{name}";

    foreach my $d (@dirs) {
	psystem "rm -rf $self->{resdir}/$d";
	psystem "mkdir -p $self->{resdir}/$d"
    }

    chdir $expt->{logdir};
    
    foreach my $f (@files) {
	$f =~ /(\d+\.\d+\.\d+\.\d+\:\d+)/;
	my $id = $1;
	die "bad filename $f" if !$id;

	my $shopts = $self->{common_args};
	my $bwopts = "$shopts -e 0.1 -t proto";
	my $field = 0;

	$field = 1 if (!$ScriptDriver::AGGLOGS);

	if ($self->{slowdown} != 1) {
	    $pipe = "$ENV{HOME}/Colyseus/scripts/TimeFilter.pl -S $self->{start} -w $self->{slowdown} -f $field |";		
	}

	my $cat = $f =~ /\.gz$/ ? "zcat" : "cat";

	psystem ("$cat $f | $pipe $bwscript $bwopts " .
	"-M 'MSG_RANGE_PUB_NOTROUTING' /dev/stdin > " .
	"$self->{resdir}/pub-send/$id.log");

	psystem ("$cat $f | $pipe $bwscript $bwopts " .
	"-M 'MSG_SUB_NOTROUTING' /dev/stdin > " .
	"$self->{resdir}/sub-send/$id.log");

	psystem ("$cat $f | $pipe $bwscript $bwopts " .
	"-M 'MSG_RANGE_PUB,MSG_PUB' /dev/stdin > " .
	"$self->{resdir}/pub-route/$id.log");

	psystem ("$cat $f | $pipe $bwscript $bwopts " .
	"-M 'MSG_SUB' /dev/stdin > " .
	"$self->{resdir}/sub-route/$id.log");

	psystem ("$cat $f | $pipe $bwscript $bwopts " .
	"-M 'MSG_LINEAR_PUB' /dev/stdin > " .
	"$self->{resdir}/pub-linear/$id.log");

	psystem ("$cat $f | $pipe $bwscript $bwopts " .
	"-M 'MSG_LINEAR_SUB' /dev/stdin > " .
	"$self->{resdir}/sub-linear/$id.log");

	psystem ("$cat $f | $pipe $bwscript $bwopts " .
	"-M 'MSG_MATCHED_PUB,MSG_RANGE_MATCHED_PUB' /dev/stdin >" .
	"$self->{resdir}/match/$id.log");

    }

}

package Colyseus::Script::ReplicaConsistency;
use Travertine;
our @ISA = qw(Colyseus::Script);
sub name { "rep-consistency" }

sub GetFrameTime {
    my $self = shift;
    my $expt = shift;
    
    if (exists $ENV{COL_RESULTS_FRAMETIME}) {
	return $ENV{COL_RESULTS_FRAMETIME};
    }
    
    my $olog = `ls $expt->{logdir}/Output* | grep -v bootstrap | head -1`;
    chomp $olog;
    tdie ("No outputlog in directory $expt->{logdir}") if ($olog eq "");
    $olog = `grep colyquake3 $olog`;
    chomp $olog;
    if ($olog ne "") { 
	return 0.05;
    }
    else {
	return 0.1;
    }
}

sub Execute { 
    my $self = shift;
    my $expt = shift;

    chdir ($expt->{logdir});
    $self->{resdir} .= "/$expt->{name}";
    $self->{common_args} .= " --frametime " . $self->GetFrameTime ($expt);

    foreach my $f (0, 1, 4) { 
	psystem ("rm -rf $self->{resdir}/$f");
	psystem ("mkdir -p $self->{resdir}/$f");

	my $cmd;

	$cmd = "$ENV{HOME}/Colyseus/scripts/repcon $self->{common_args} -o $self->{resdir}/$f";
	if ($f != 0) {
	    $cmd .= " -f $f";
	}

	ppsystem "$cmd";
    }
}

package Colyseus::Script::ReplicaConsistencyCombine;
use Travertine;
use Statistics::Descriptive;
our @ISA = qw(Colyseus::Script);
sub name { "rep-consistency" } 

sub InitExptSet { 
    my $self = shift;
    my $exptset = shift;
    
    # dont get rid of the directory, please :)
    unless ($exptset->{-append}) {
	my $resdir = "$exptset->{outdir}/$self->{name}/$exptset->{name}";
	psystem ("rm -f $resdir/fudge.?.stats");
    }
}

our $ABSOLUTE = 0;
our $NEEDED = 0;        # what the fuck does this mean?? 

sub Execute {
    my $self = shift;
    my $expt = shift;
    
    foreach my $fudge (0, 1, 4) {
	my $new = 0;
	if (!-f "$self->{resdir}/fudge.$fudge.stats") { 
	    $new = 1;
	}
	my $fh = new IO::File (">> $self->{resdir}/fudge.$fudge.stats");
	if ($new) { 
	    print $fh "# stat\tmissing_fraction\n";
	}

	print $fh "$expt->{name}\n";
	print $fh (("=" x 79) . "\n");

	my $f = "$self->{resdir}/$expt->{name}/$fudge";
	die "no such dir $f" if ! -d $f;

	my $stat = Statistics::Descriptive::Full->new();

	foreach my $log (glob("$f/*")) {
	    my $fields;
	    if ($ABSOLUTE) {
		$fields = "2,4";
	    } else {
		$fields = "4,5";
	    }

	    open (D, "cut -f $fields $log |") || die "can't open pipe to $log";
	    while (<D>) {
		chomp $_;
		die "bad line: $_" if $_ !~ /\d/;
		my ($needed, $point);

		if ($ABSOLUTE) {
		    ($point, $needed) = split(/\t/, $_);
		} else {
		    ($needed, $point) = split(/\t/, $_);
		}
		next if $needed == 0;

		if ($NEEDED) {
		    $point = $needed;
		}

		$stat->add_data($point);
	    }
	    close(D);
	}

	if ($stat->count() == 0) {
	    $stat->add_data( 0 );
	}

	my $mean = sprintf "%.5f", $stat->mean();
	my $std  = sprintf "%.5f", $stat->standard_deviation();
	my $med = sprintf "%.5f", $stat->median();
	my $ninefive = sprintf "%.5f", $stat->percentile(95);
	my $ninenine = sprintf "%.5f", $stat->percentile(99);
	my $nineninenine = sprintf "%.5f", $stat->percentile(99.9);

	print $fh "MEAN\t$mean\n";
	print $fh "STDDEV\t$std\n";
	print $fh "MEDIAN\t$med\n";
	print $fh "95%\t$ninefive\n";
	print $fh "99%\t$ninenine\n";
	print $fh "99.9%\t$nineninenine\n";

	close ($fh);
    }
}

package Colyseus::Script::ReplicaConsistencyDetailed;
use Travertine;
our @ISA = qw(Colyseus::Script Colyseus::Script::ReplicaConsistency);
sub name { "rep-consistency-detail" }

sub Execute { 
    my $self = shift;
    my $expt = shift;

    chdir ($expt->{logdir});
    $self->{resdir} .= "/$expt->{name}";

    $self->{common_args} .= " --frametime " . $self->GetFrameTime ($expt);
    foreach my $f (0, 1, 4) {
	psystem ("rm -rf $self->{resdir}/$f");
	psystem ("mkdir -p $self->{resdir}/$f");

	my $cmd;

	$cmd = "$ENV{HOME}/Colyseus/scripts/repcon -d $self->{common_args} -o $self->{resdir}/$f";
	if ($f != 0) {
	    $cmd .= " -f $f";
	}

	ppsystem "$cmd";
    }
}

package Colyseus::Script::ReplicaConsistencyDetailedSummary;
use Travertine;
use Statistics::Descriptive;
our @ISA = qw(Colyseus::Script);
sub name { "rep-consistency-detail" } 

sub InitExptSet { 
    my $self = shift;
    my $exptset = shift;
    
    # dont get rid of the directory, please :)
    unless ($exptset->{-append}) {
	my $resdir = "$exptset->{outdir}/$self->{name}/$exptset->{name}";
	psystem ("rm -f $resdir/fudge.?.stats");
    }
}

our $ABSOLUTE = 0;
our $NEEDED = 0;        # what the fuck does this mean?? 

sub Execute {
    my $self = shift;
    my $expt = shift;
    
    my @fields = (0, 1, 2, 3, 4);
    my @types  = ('P', 'M', 'I', 'O', 'U');

    foreach my $fudge (0, 1, 4) {
	my $new = 0;
	if (!-f "$self->{resdir}/fudge.$fudge.stats") { 
	    $new = 1;
	}
	my $fh = new IO::File (">> $self->{resdir}/fudge.$fudge.stats");
	if ($new) { 
	    print $fh "# type stat\tmissing_fraction\n";
	}

	print $fh "$expt->{name}\n";
	print $fh (("=" x 79) . "\n");

	my $f = "$self->{resdir}/$expt->{name}/$fudge";
	die "no such dir $f" if ! -d $f;

	my @stats;
	for (my $i=0; $i < @fields; $i++) {
	    $stats[$i] = new Statistics::Descriptive::Full();
	}

	tinfo ("Reading from | $ENV{HOME}/Colyseus/scripts/MapGUIDListToType.pl -f2,3 -g $expt->{logdir} $f/*.out |");
	open(P, "$ENV{HOME}/Colyseus/scripts/MapGUIDListToType.pl -f2,3 -g $expt->{logdir} $f/*.out |") or die "$!";
	while (<P>) {
	    chomp;
	    if (/[^ \t]+\t([\d,]*)\t([\d,]*)/) {
		my ($miss, $req) = ($1, $2);
		my @miss = split /,/, $miss;
		my @req  = split /,/, $req;

		for (my $i=0; $i<@fields; $i++) {
		    if (!defined $miss[$fields[$i]] ||
		    !defined $req[$fields[$i]]) {
			warn "missing field $fields[$i]: $_";
			next;
		    }
		    my $s = $req[$i] > 0 ? $miss[$i]/$req[$i] : 0;
		    $stats[$i]->add_data($s);
		}
	    } else {
		warn "bad line: $_ ($!)";
	    }
	}
	close(P);

	for (my $i=0; $i<@fields; $i++) {
	    my $t = $types[$i];
	    my $stat = $stats[$i];

	    my $mean = sprintf "%.5f", $stat->mean();
	    my $std  = sprintf "%.5f", $stat->standard_deviation();
	    my $med = sprintf "%.5f", $stat->median();
	    my $ninefive = sprintf "%.5f", $stat->percentile(95);
	    my $ninenine = sprintf "%.5f", $stat->percentile(99);
	    my $nineninenine = sprintf "%.5f", $stat->percentile(99.9);

	    print $fh "$t MEAN    $mean\n";
	    print $fh "$t STDDEV  $std\n";
	    print $fh "$t MEDIAN  $med\n";
	    print $fh "$t 95%     $ninefive\n";
	    print $fh "$t 99%     $ninenine\n";
	    print $fh "$t 99.9%   $nineninenine\n";
	    print $fh "#======================================\n";
	}
	close ($fh);
    }
}

1;
