#!/usr/bin/perl
#
# Calculate the time between creations of the same replica.
#
# -a  - show all (instead of mean/stddev/95%)
# -i  - between info instead of construct

use strict;
use Statistics::Descriptive;
use Getopt::Std;
use vars qw($opt_a $opt_S $opt_s $opt_l $opt_i $opt_d $opt_g);

our %InfoTypes    =  ( 'i' => 'OBJINFO' );
our %CreateTypes  =  ( 'm' => 'MCREATE',
		       'c' => 'DCREATE',
		       'p' => 'PCREATE',
		       'f' => 'FCREATE', );
our %StoreTypes   =  ( 's' => 'STORE' );
our %DestroyTypes =  ( 'y' => 'MDESTROY',
		       'd' => 'TDESTROY',
		       'e' => 'EDESTROY' );

getopts("aS:s:l:id:g:");

our $SHOWALL  = defined $opt_a;
our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 0;
our $LENGTH   = defined $opt_l ? $opt_l : undef;
our $INFOS    = defined $opt_i;
our $GREATER  = defined $opt_g ? $opt_g : 0; 
our $DIR      = defined $opt_d ? $opt_d : ".";

our %SKIP_TYPES = ( "bolt" => 1,
		    "grenade" => 1,
		    "hgrenade" => 1,
		    "rocket" => 1,
		    "bfg blast" => 1 );
our %SKIP_GUIDS;

###############################################################################

our %GUID_MAP;

my @guidlogs = glob("$DIR/GUIDLog.*");
die "no guid logs" if @guidlogs == 0;

print STDERR "processing guid logs: ";

foreach my $log (@guidlogs) {
    open(L, "<$log") || die "can't open $log: $!";
    while (<L>) {
	if ($_ =~ /\d+\.\d+\t([^\t]+)\t([^\t]+)\t(\w)/) {
	    my ($guid, $type, $act) = ($1, $2, $3);
	    if ($act eq 'C' && $SKIP_TYPES{$type}) {
		$SKIP_GUIDS{$guid} = 1;
	    }
	}
    }
    close(L);
    print STDERR ".";
}

print STDERR "\n";

###############################################################################

my @results = ProcessLogs(@ARGV);
my $stats   = new Statistics::Descriptive::Full();

foreach my $t (@results) {
    next if $t->[1] < $GREATER;
    $stats->add_data($t->[1]);
    print "$t->[0]\t$t->[1]\n" if $SHOWALL;
}

if (!$SHOWALL) {
foreach my $f ( [ "MEAN",   sub { $_[0]->mean() } ],
		[ "STDDEV", sub { $_[0]->standard_deviation() } ],
		[ "MEDIAN", sub { $_[0]->median() } ],
		[ "95 %",   sub { $_[0]->percentile(95) } ] ) {
    print sprintf("%10s\t%10.3f\n", $f->[0], &{$f->[1]}($stats));
}
}

###############################################################################

sub ProcessLogs(@)
{
    my @files = @_;

    my %created  = ();
    my %deleted  = ();
    my @results  = ();

    print STDERR "processing logs: ";

    # first-pass: find aliases and messages to track
    foreach my $f (@files) {
	open(F, "<$f") || die "can't open $f";
	my($start, $end);
	$start = $START;

	%created = ();
	%deleted = ();

	while (<F>) {
	    chomp $_;

	    my ($time, $guid, $type) = ($_ =~ /^([^\t]+)\t([^\t]+)\t([^\t]+)/);
	    next if $SKIP_GUIDS{$guid};
	    
	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    if ($INFOS && $InfoTypes{$type}) {
		if (exists $deleted{$guid}) {
		    push @results, [$guid, $time - $deleted{$guid}];
		    delete $deleted{$guid};
		}
		
		$created{$guid} = $time;
	    } elsif ($CreateTypes{$type}) {
		if (exists $deleted{$guid}) {
		    push @results, [$guid, $time - $deleted{$guid}];
		    delete $deleted{$guid};
		}
		
		$created{$guid} = $time;
	    } elsif ($DestroyTypes{$type}) {
		delete $created{$guid};
		$deleted{$guid} = $time;
	    }
	}

	# time the ones still dead at the end
	foreach my $g (keys %deleted) {
	    push @results, [ $g, $end - $deleted{$g} ];
	}

	print STDERR ".";
    }

    print STDERR "\n";

    return @results;
}
