#!/usr/bin/perl
#
# Calculate the lifetime of publications.
#
# -a  - show all (instead of mean/stddev/95%)
#

use strict;
use Statistics::Descriptive;
use Getopt::Std;
use vars qw($opt_S $opt_a $opt_s $opt_l $opt_d);

our %InfoTypes    =  ( 'i' => 'OBJINFO' );
our %CreateTypes  =  ( 'm' => 'MCREATE',
		       'c' => 'DCREATE',
		       'p' => 'PCREATE',
		       'f' => 'FCREATE', );
our %StoreTypes   =  ( 's' => 'STORE' );
our %DestroyTypes =  ( 'y' => 'MDESTROY',
		       'd' => 'TDESTROY',
		       'e' => 'EDESTROY' );

getopts("aS:s:l:d:");

our $SHOWALL  = defined $opt_a;
our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 0;
our $LENGTH   = defined $opt_l ? $opt_l : undef;
our $DIR      = defined $opt_d ? $opt_d : ".";

our %SKIP_TYPES = ( "missile" => 1,
		    "bolt" => 1,
		    "grenade" => 1,
		    "hgrenade" => 1,
		    "rocket" => 1,
		    "bfg blast" => 1 );
our %SKIP_GUIDS;

###############################################################################

our %GUID_MAP;

my @guidlogs = glob("$DIR/GUIDLog.*");
die "no guid logs" if @guidlogs == 0;

print STDERR "processing guid logs: ";

foreach my $log (@guidlogs) {
    open(L, "<$log") || die "can't open $log: $!";
    while (<L>) {
	if ($_ =~ /\d+\.\d+\t([^\t]+)\t([^\t]+)\t(\w)/) {
	    my ($guid, $type, $act) = ($1, $2, $3);
	    if ($act eq 'C' && $SKIP_TYPES{$type}) {
		$SKIP_GUIDS{$guid} = 1;
	    }
	}
    }
    close(L);
    print STDERR ".";
}

print STDERR "\n";

###############################################################################

my @results = ProcessLogs(@ARGV);
my %stats   = ();

my @fields = ("ID", "INFO", "CREATE", "STORE", "DESTROY", "TOTAL");

print sprintf("%22s %10s %10s %10s %10s %10s\n", 
	      @fields);

for (my $i=1; $i<@fields; $i++) {
    $stats{$fields[$i]} = new Statistics::Descriptive::Full();
}

foreach my $t (@results) {
    my $guid = $t->[0];
    my @vals = ();
    my $last;
    my $total = 0;

    for (my $i=1; $i<@fields-1; $i++) {
	if ($t->[$i]) {
	    if (!$last) {
		$last = $t->[$i]->[0];
	    }
	    my $time = ($t->[$i]->[0] - $last);
	    push @vals, $time;
	    $total += $time;
	    $stats{$fields[$i]}->add_data($time);
	    $last = $t->[$i]->[0];
	} else {
	    push @vals, -1;
	}
    }

    $stats{TOTAL}->add_data($total);

    print sprintf("%22s %10.3f %10.3f %10.3f %10.3f %10.3f\n", 
		  $guid, @vals, $total) if $SHOWALL;
}

foreach my $f ( [ "MEAN",   sub { $_[0]->mean() } ],
		[ "STDDEV", sub { $_[0]->standard_deviation() } ],
		[ "MEDIAN", sub { $_[0]->median() } ],
		[ "95 %",   sub { $_[0]->percentile(95) } ] ) {
    print sprintf("%22s ", $f->[0]) if !$SHOWALL;
    for (my $i=1; $i<@fields; $i++) {
	print sprintf("%10.3f ", &{$f->[1]}($stats{$fields[$i]}) ) 
	    if !$SHOWALL;
    }
    print "\n" if !$SHOWALL;
}

###############################################################################

sub ProcessLogs(@)
{
    my @files = @_;

    my %tracked  = ();
    my @results  = ();

    print STDERR "processing logs: ";

    # first-pass: find aliases and messages to track
    foreach my $f (@files) {
	open(F, "<$f") || die "can't open $f";
	my($start, $end);
	$start = $START;

	%tracked = ();

	while (<F>) {
	    chomp $_;

	    my ($time, $guid, $type) = ($_ =~ /^([^\t]+)\t([^\t]+)\t([^\t]+)/);
	    next if $SKIP_GUIDS{$guid};
	    
	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    if ($InfoTypes{$type}) {
		$tracked{$guid} = {};
		$tracked{$guid}->{INFO} = [$time, $type];
	    } elsif (defined $tracked{$guid}) {
		if ($CreateTypes{$type}) {
		    if (!defined $tracked{$guid}->{INFO}) {
			warn "no info but got create! $guid";
		    }
		    $tracked{$guid}->{CREATE} = [$time, $type];
		} elsif ($StoreTypes{$type}) {
		    #if (!defined $tracked{$guid}->{CREATE}) {
		#	warn "no create but got store! $guid";
		#    }
		    $tracked{$guid}->{STORE} = [$time, $type];
		} elsif ($DestroyTypes{$type}) {
		    if (!defined $tracked{$guid}->{INFO}) {
			warn "no info but got delete! $guid";
		    }
		    # if we didn't create it, ignore the interest since
		    # we probably decided that we weren't really interested
		    # in it.
		    if (defined $tracked{$guid}->{CREATE}) {
			$tracked{$guid}->{DELETE} = [$time, $type];
			
			push @results, Output($guid, $tracked{$guid});
		    }
		    delete $tracked{$guid};
		} else {
		    warn "unknown type: $type\n";
		}
	    } else {
		#warn "here: $guid '$type'\n";
	    }
	}

	#foreach my $g (keys %tracked) {
	#    my $t = $tracked{$g};
	#    $t->{DELETE} = [$end, 'E'];
	#    push @results, Output($g, $t);
	#}

	print STDERR ".";
    }

    print STDERR "\n";

    return @results;
}

sub Output($$)
{
    my $guid = shift;
    my $ref  = shift;

    return [ $guid, $ref->{INFO}, $ref->{CREATE}, $ref->{STORE}, $ref->{DELETE} ];
}
