#!/usr/bin/perl
#
# Creates a time series of how many replicas were on a node(s).
#
# -s num   - skip the first x seconds of the trace
# -l num   - the length of time to aggregate over (ignore remainder)
# -e epoch - size of each aggregation epoch (sec -- floating point)
# -d       - detailed output

use strict;
use Getopt::Std;
use IO::File;
use vars qw($opt_S $opt_s $opt_l $opt_e $opt_d); 

getopts("S:s:l:e:d");

our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 120;
our $LENGTH   = defined $opt_l ? $opt_l : 600;
our $EPOCH    = defined $opt_e ? $opt_e : 1;
our $DETAILED = $opt_d;

###############################################################################

our %InfoTypes    =  ( 'i' => 'OBJINFO' );
our %CreateTypes  =  ( 'm' => 'MCREATE',
		       'c' => 'DCREATE',
		       'p' => 'PCREATE',
		       'f' => 'FCREATE', );
our %StoreTypes   =  ( 's' => 'STORE' );
our %DestroyTypes =  ( 'y' => 'MDESTROY',
		       'd' => 'TDESTROY',
		       'e' => 'EDESTROY' );

our @LOGS = OpenFiles(@ARGV);
our $start = (defined $START ? $START : min(GetHeadTimes(@LOGS))) + $SKIPTIME;
our $end = $start+$LENGTH;
ScrollUntilTime($start, @LOGS);

ProcessInput(@LOGS);

###############################################################################

our @Objects;
foreach my $l (@LOGS) {
    push @Objects, {};
}

sub ProcessInput(@)
{
    my @files = @_;

    my @buckets;

    my $line;
    my $currtime = $start;

    my $done = 0;

    while (!$done) {

	$done = 1;
	my $count = 0;
	my @all;

	for (my $i=0; $i<@files; $i++) {
	    my $f = $files[$i];
	    my $s = $Objects[$i];
	    
	    my $stoptime = $currtime+$EPOCH;
	    
	    while ($line = GetNext($f)) {
		chomp $line;
		if ($line =~ /^(\d+\.\d+)\t([^\t]+)\t([^\t]+)/) {
		    my ($time, $guid, $type) = ($1, $2, $3);
		    
		    if ($time > $end) {
			last;
		    } else {
			$done = 0;
		    }
		    
		    if ($time >= $stoptime) {
			AddResid($f, $line);
			last;
		    }
		    
		    if ($CreateTypes{$type}) {
			$s->{$guid} = 1;
		    } elsif ($DestroyTypes{$type}) {
			delete $s->{$guid};
		    }
		}
	    }

	    if (!$DETAILED) {
		$count += scalar keys %$s;
	    } else {
		push @all, keys %$s;
	    }
	}

	if (!$DETAILED) {
	    print "$currtime\t$count\n";
	} else {
	    print "$currtime\t" . join(",", @all) . "\n";
	}
	$currtime += $EPOCH;
    }

}

###############################################################################

sub OpenFiles {
    my @logs = @_;
    my @ret;

    foreach my $file (@logs) {
	my $fh = new IO::File("<$file");
	die "can't open $file" if !$fh;
	push @ret, [ [], $fh, 0 ];
    }

    return @ret;
}

sub GetResid($) { 
    my $ret = shift @{$_[0]->[0]}; 
    $_[0]->[2] = 0 if @{$_[0]->[0]} == 0;
    return $ret;
}

sub AddResid($@) { 
    my $log = shift;
    push @{$log->[0]}, @_; 
    $log->[2] = 1;
}

sub GetFile($) { return $_[0]->[1] }

sub GetNext($) {
    my $log = shift;
    if ($log->[2]) {
	#print STDERR "here\n";
	return GetResid($log);
    } else {
	#print STDERR "there\n";
	my $f = $log->[1];
	return <$f>;
    }
}

sub GetHeadTimes {
    my @logs = @_;
    my @ret;

    foreach my $log (@logs) {
	my $line = GetNext($log);
	die "no lines in log!" if !$line;
	chomp $line;
	$line =~ /^(\d+\.\d+)/;
	push @ret, $1;
	AddResid($log, $line);
    }

    return @ret;
}

sub ScrollUntilTime {
    my $time = shift;
    my @logs = @_;
    my @ret;

    print STDERR "scrolling to start: ";

    foreach my $log (@logs) {
	my $line;

	print STDERR ".";

	while ($line = GetNext($log)) {
	    chomp $line;
	    $line =~ /^(\d+\.\d+)/;
	    my $t = $1;
	    #push @ret, $line;
	    if ($t < $time) {
		next;
	    } else {
		last;
	    }
	}
	AddResid($log, $line) if $line;
    }

    print STDERR "done!\n";
}

sub min {
    my @vals = @_;
    my $min = shift @vals;

    foreach my $v (@vals) {
	if ($min > $v) {
	    $min = $v;
	}
    }

    return $min;
}
