#!/usr/bin/perl
#
# Usage: ./ReplicaConsistencyDebug.pl [options]
#
# (in directory where logs are)
#
# Options:
#
# -s time   number of second to skip from beginning of trace
# -l time   number of seconds of trace to examine
# -o dir    output directory where to create logs 
#           (output as consistency.$ID.out)
# -f frames allow fudge frame fudge (i.e., only missing if required fudge
#           frames ago and still missing this frame).
# -d        detailed output
# -w        slowdown
#
# Output format:
#
# time num_missing num_required UNUSED frac_missing UNUSED
#
# where we only consider mobile replicas (players, monsters, missiles), not
# primary objects or items.

use strict;
use IO::File;
use Getopt::Std;
use Carp;
use vars qw($opt_S $opt_s $opt_l $opt_o $opt_f $opt_d $opt_D $opt_w);

getopts("S:s:l:o:f:dDw:");

my $START    = defined $opt_S ? $opt_S : undef;
my $SKIPTIME = defined $opt_s ? $opt_s : 120;
my $LENGTH   = defined $opt_l ? $opt_l : 10*60;
my $OUT_FILE = defined $opt_o ? $opt_o : ".";
my $FUDGE    = defined $opt_f ? $opt_f : 0;
my $DETAILED = defined $opt_d;
my $DISTANCE = defined $opt_D;
if ($DISTANCE) {
    $DETAILED = 1;
}
our $SLOWDOWN = defined $opt_w ? $opt_w : 1.0;

our %InfoTypes    =  ( 'i' => 'OBJINFO' );
our %CreateTypes  =  ( 'm' => 'MCREATE',
		       'c' => 'DCREATE',
		       'p' => 'PCREATE',
		       'f' => 'FCREATE', );
our %StoreTypes   =  ( 's' => 'STORE' );
our %DestroyTypes =  ( 'y' => 'MDESTROY',
		       'd' => 'TDESTROY',
		       'e' => 'EDESTROY' );

# define this to spit out debug info about this guid
our $DEBUG_EXAMINE_GUID = '';


my $FRAME_TIME = 0.1; # sec
if ($SLOWDOWN > 1.0) { 
    $SKIPTIME *= $SLOWDOWN;
    $LENGTH *= $SLOWDOWN;
    $FRAME_TIME *= $SLOWDOWN;
}

# this log records what areas each object is interested in
my @InterestLogs = OpenFiles( sort { $a cmp $b } glob("ObjectInterestLog.*") );
# this log records when objects are created and deleted
my @DeltaLogs    = OpenFiles( sort { $a cmp $b } glob("ObjectDeltaLog.*") );
# this log records when replicas are created or deleted 
my @ReplicaLogs  = OpenFiles( sort { $a cmp $b } glob("ReplicaLifetimeLog.*") ); 

die "no logs: int(@InterestLogs) del(@DeltaLogs)!" 
    if @InterestLogs==0 || @DeltaLogs==0 || @ReplicaLogs==0;

###############################################################################

# $guid => cost_so_far
my %Clients = ();
# $guid => [ lastvframe_up, type, [pos], [[min],[max]], stat, create_time ]
my %Objects = ();

my ($VF, $TYPE, $POS, $BBOX, $STAT, $CREATE_TIME) = (0,1,2,3,4,5);

my @STORE;
my @TO_DELETE;
my $index = 0;
foreach my $file (sort { $a cmp $b } glob("ObjectDeltaLog.*")) {
    $STORE[$index++] = {};
    $TO_DELETE[$index++] = {};
}

my $CURRENT_SERVER = undef;
my %GUID_MAP;

###############################################################################

# get the max start-time
my $_start = defined $START ? $START : min( GetHeadTimes(@InterestLogs, @DeltaLogs, @ReplicaLogs) );

my $realstart = $_start + $SKIPTIME;

# scroll logs until that time to synchronize
#print STDERR "** scrolling lines to starttime: $realstart: ";
#ScrollUntilTime($realstart, @InterestLogs);
#ScrollUntilTime($realstart, @DeltaLogs);
#ScrollUntilTime($realstart, @ReplicaLogs);
#print STDERR "\n";

print STDERR "** scrolling until start of sim: ";

# process all the replica info before our simulation starts
my $started = 0;
our $pframe = 0;
our $pre_frame = $_start;
while (1) {
    $pre_frame = $pre_frame+$FRAME_TIME;

    BeginFrame();

    my $done = ProcessInterests($pre_frame);
    ProcessDeltas($pre_frame);
    ProcessReplicas($pre_frame);

    if ($pre_frame >= $realstart) {
	last;
    }

    $pframe++;
    if ($pframe % 100 == 0) {
	print STDERR ",";
	if ($pframe % 600 == 0) {
	    print STDERR $pre_frame;
	}
    }
}

$started = 1;

###############################################################################

my @times = GetHeadTimes(@InterestLogs, @DeltaLogs );
my $realstart = min( @times );
print "\nSTART: @times\n";

my $frame_end = $realstart;

my $vframe = 0;

sub vtime {
    if (!$started) {
	return $_start+$pframe*$FRAME_TIME;
    } else {
	return $realstart+$vframe*$FRAME_TIME;
    }
}

###############################################################################

my @OUT;

foreach my $file (sort { $a cmp $b } glob("ObjectDeltaLog.*")) {
    #next if $file =~ /10\.1\.1\.2\:/; # skip master!
    $file =~ /(\d+\.\d+\.\d+\.\d+\:\d+)/;
    #print "$1\n";
    die "bad filename: $file" if ! $1;
    my $fh = new IO::File(">$OUT_FILE/consistency.$1.out");
    die "can't create consistency.$1.out" if !$fh;

    push @OUT, $fh;
}

# now start the simulation
while (1) {
    $frame_end = $frame_end+$FRAME_TIME;

    BeginFrame();

    my $done = ProcessInterests($frame_end);

    ProcessDeltas($frame_end);

    ProcessReplicas($frame_end);

    ComputeCosts();

    $vframe++;

    {
	use integer;
	if ($vframe % 100 == 0) {
	    print STDERR ".";
	    if ($vframe % 1200 == 0) {
		print STDERR $frame_end;
	    }
	}
    }

    if ( $done ) {
	print STDERR "\n ===== DONE! =====\n";
	last;
    }

    last if ($frame_end > $realstart + $LENGTH);
}

###############################################################################

sub GetObj
{
    my $guid = shift;

    if (!defined $Objects{$guid}) {
	$Objects{$guid} = [undef,undef,undef,[],undef,undef];
    }

    return $Objects{$guid};
}

sub BBoxContains($$)
{
    my $box = shift;
    my $pt  = shift;

    for (my $i=0; $i<3; $i++) {
	if ($box->[0]->[$i] > $pt->[$i] || 
	    $box->[1]->[$i] < $pt->[$i]) {
	    return 0;
	}
    }

    return 1;
}

sub Distance($$)
{
    my ($p1,$p2) = @_;
    my $x = ($p1->[0] - $p2->[0]);
    my $y = ($p1->[1] - $p2->[1]);
    my $z = ($p1->[2] - $p2->[2]);
    return sqrt($x*$x+$y*$y+$z*$z);
}

###############################################################################

sub BeginFrame
{
    foreach my $g (keys %Objects) {
	my $ref = $Objects{$g};

	# delete it if it was deleted in the last frame.
	if ($ref->[$STAT] eq 'd') {
	    delete $Objects{$g};
	}
    }

    for (my $i=0; $i<scalar @TO_DELETE; $i++) {
	# delete replicas deleted in the last frame
	foreach my $guid (keys %{$TO_DELETE[$i]}) {
	    delete $STORE[$i]->{$guid};
	}
	$TO_DELETE[$i] = {};
    }
}

our @PREV_REQUIRED;
#our @PREV_KNOWABOUT;
our @PREV_MISSING;

sub ComputeCosts
{
    my $time = $realstart + $vframe*$FRAME_TIME;
    #my $framecost = 0;

    my @required;
    #my @knowabout;
    my @missing;
    my @dist;

    for (my $i=0; $i<@OUT; $i++) {
	$required[$i] = {};
	#$knowabout[$i] = {};
	$missing[$i] = {};
	$dist[$i] = {};
    }

=start
    # debug only...
    my $nobjs = 0;
    foreach my $o (keys %Objects) { 
	my $oref = $Objects{$o};
	next if !$oref->[$POS];
	next if $oref->[$TYPE] eq 'o';
	next if !defined $oref->[$STAT];

	$nobjs++;
    }

    print STDERR "perl; frame $vframe inserted $nobjs objects\n";
=cut

    #print "clients: " . scalar(keys %Clients) . "\n";
    #print "objects: " . scalar(keys %Objects) . "\n";

    #for (my $i=0; $i<@OUT; $i++) {
    #	print "server $i: " . scalar(keys %{$STORE[$i]}) . "\n";
    #}

    # print STDERR "perl; frame $vframe nclients=", scalar keys %Clients, "\n";
    foreach my $c (keys %Clients) {
	my $cref = $Objects{$c};
	next if !$cref || !$cref->[$POS];
	next if !defined $GUID_MAP{$c};
	my $bbox = $cref->[$BBOX];
	my $cpos = $cref->[$POS];
	#my $client_framecost = 0;

	my $serverid = $GUID_MAP{$c};

	foreach my $o (keys %Objects) {
	    my $oref = $Objects{$o};
	    my $pos = $oref->[$POS];
	    next if !$pos;
	    #print "here: '$GUID_MAP{$c}' $serverid\n";
	    next if !defined $GUID_MAP{$o} || $serverid == $GUID_MAP{$o};
	    # XXX - don't have enough info to accurately determine which
	    # items we should have because we dont replicate it unless the
	    # point is in our bounding box AND in our PVS
	    next if $oref->[$TYPE] eq 'i';
	    next if !defined $oref->[$STAT];

	    my $bbox_contains = BBoxContains($bbox, $pos);

	    if ($bbox_contains) {
		# print STDERR "\tperl; $o\n";
		
		#print STDERR "ITEM: $o\n" if $oref->[$TYPE] eq 'i';
		# see if my server has it as a replica!
		$required[$serverid]->{$o} = 1;

		my $dist = Distance($cpos, $pos);
		$dist[$serverid]->{$o} = $dist;

		if (!exists $STORE[$serverid]->{$o}) {
		    # missing object we're interested in!
		    $missing[$serverid]->{$o} = 1;

		    #my $min = join(",", @{$bbox->[0]});
 		    #my $max = join(",", @{$bbox->[1]});
		    #print STDERR "\n$time  $c $cref->[$TYPE] ($GUID_MAP{$c}) [$min|$max] missing $o $oref->[$TYPE] ($GUID_MAP{$o}) [@$pos]\n";# if $c eq $DEBUG_EXAMINE_GUID || $o eq $DEBUG_EXAMINE_GUID;
		    #print STDERR "STORE: " . join(",", keys %{$STORE[$serverid]}) . "\n" if $c eq $DEBUG_EXAMINE_GUID || $o eq $DEBUG_EXAMINE_GUID;
		} elsif ($STORE[$serverid]->{$o} ne 'c') {
		    # know about it, but haven't been able to construct it yet
		    $missing[$serverid]->{$o} = 1;
		    #$knowabout[$serverid]->{$o} = 1;
		} else {
		    # else i must know about it and have it
		    #$knowabout[$serverid]->{$o} = 1;
		}
	    }
	}
    }

    if ($FUDGE > 0) {
	push @PREV_REQUIRED, [];
	push @PREV_MISSING, [];
    }

    for (my $i=0; $i<@OUT; $i++) {
	my $ftime;

	my $required;
	my $missing;
	my @_required;
	my @_missing;
	#my $knowabout;

	if ($FUDGE > 0) {
	    my $skip = 0;
	    $missing = 0;

	    # if we are fudging then an object is only missing if it was
	    # required $FUDGE frames ago and we still have not gotten it yet

	    if (scalar @PREV_REQUIRED > $FUDGE) {
		foreach my $guid (keys %{$PREV_REQUIRED[0]->[$i]}) {
		    #if ($PREV_MISSING[$i]->{$guid}) {
		    #	print STDERR "$i prev: $guid ($missing{$guid})\n";
		    #}

		    if ($missing[$i]->{$guid}) {
			my $ismissing = 1;
			for (my $j=0; $j<$FUDGE; $j++) {
			    if (!$PREV_MISSING[$j]->[$i]->{$guid}) {
				$ismissing = 0;
				last;
			    }
			}
			$missing += $ismissing;
			if ($ismissing && $DETAILED) {
			    push @_missing, $guid
			}
		    }
		}
		$ftime     = $time - $FRAME_TIME;
		$required  = scalar keys %{$PREV_REQUIRED[0]->[$i]};
		if ($DETAILED) {
		    @_required = keys %{$PREV_REQUIRED[0]->[$i]};
		}
		#$knowabout = scalar $PREV_KNOWABOUT[$i];
	    } else {
		$skip = 1;
	    }

	    $PREV_REQUIRED[$#PREV_REQUIRED]->[$i] = $required[$i];
	    #$PREV_KNOWABOUT[$i] = $knowabout[$i];
	    $PREV_MISSING[$#PREV_MISSING]->[$i]   = $missing[$i];

	    next if $skip;
	} else {
	    $ftime     = $time;
	    $required  = scalar keys %{$required[$i]};
	    #$knowabout = scalar keys %{$knowabout[$i]};
	    $missing   = scalar keys %{$missing[$i]};

	    if ($DETAILED) {
		@_required = keys %{$required[$i]};
		@_missing  = keys %{$missing[$i]};
	    }

	    # print STDERR "perl; i=$i required=$required missing=$missing\n";
	}

	my $fh = $OUT[$i];
	#my $required = scalar keys %required;
	#my $missing = scalar keys %missing;
	#my $knowabout = scalar keys %knowabout;
	if ($DISTANCE) {
	    foreach my $m (@_missing) {
		my $dist = $dist[$i]->{$m};
		print $fh "1\t$ftime\t$dist\n";
	    }
	    foreach my $o (@_required) {
		my $dist = $dist[$i]->{$o};
		print $fh "0\t$ftime\t$dist\n";
	    }
	} elsif (!$DETAILED) {
	    my $ratio1 = sprintf("%.5f", $required ? ($missing/$required) : 0);
	    print $fh "$ftime\t$missing\tNA\t$required\t$ratio1\tNA\n";
	} else {
	    print $fh "$ftime\t" . join(",", @_missing) . "\t" .
		join(",", @_required) . "\n";
	}
    }

    if ($FUDGE > 0 && scalar @PREV_REQUIRED > $FUDGE) {
	shift @PREV_REQUIRED;
	shift @PREV_MISSING;
    }
}

###############################################################################

sub ProcessInterestLine
{
    my $line = shift;

    if ($line =~ /^\d+\.\d+\t\d+\t([\d\.\:]+)\t(\w)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)/) {
	my $guid = $1;
	my $type = $2;
	my @pos  = ($3,$4,$5);
	my @min  = ($6,$7,$8);
	my @max  = ($9,$10,$11);

	# XXX - don't have enough info to accurately determine which
	# items we should have because we dont replicate it unless the
	# point is in our bounding box AND in our PVS
	#
	# therefore, just skip them to minimize the space used
	if ($type eq 'i') {
	    delete $Objects{$guid};
	    delete $GUID_MAP{$guid};
	    next;
	}

	if ($type eq 'p'|| $type eq 'o' || $type eq 'm') {
	    if (!exists $Clients{$guid}) {
		$Clients{$guid} = 1;
	    }
	}

	if (defined $CURRENT_SERVER) {
	    $GUID_MAP{$guid} = $CURRENT_SERVER;
	    #print STDERR ">> $type\t$guid\t$CURRENT_SERVER\n" if $type ne 'i';
	}

	my $ref = GetObj($guid);
	$ref->[$POS] = \@pos;
	$ref->[$TYPE] = $type;
	$ref->[$BBOX]->[0] = \@min;
	$ref->[$BBOX]->[1] = \@max;

#	if ($guid eq $DEBUG_EXAMINE_GUID) {
#	    print STDERR "\n" . vtime() . " [DD] INTEREST $type (" . join(",", @pos) . ") {" .
#		join(",", @min) . "}{" . join(",", @max) . "\n";
#	}
    } else {
	warn "bad interest line: $line\n";
    }
}

sub ProcessInterestLines
{
    my @lines = @_;

    $CURRENT_SERVER = undef; # dunno

    foreach my $line (@lines) {
	ProcessInterestLine($line);
    }
}

sub ProcessInterests
{
    my $endtime = shift;
    #my @residual;
    my $line;
    my $done = 1;

    my $index = 0;
    foreach my $f (@InterestLogs) {
	$CURRENT_SERVER = $index++;
	
	while ($line = GetNext($f)) {
	    chomp $line;
	    $line =~ /^(\d+\.\d+)/;
	    my $time = $1;
	    if ($time >= $endtime) {
		#push @residual, $_;
		AddResid($f, $line);
		$done = 0;
		last;
	    }
	    ProcessInterestLine($line);
	}
    }

    return $done;
    #return @residual;
}

###############################################################################

sub ProcessDeltaLine
{
    my $line  = shift;
    my $force = shift;

    if ($line =~ /^(\d+\.\d+)\t\d+\t([\d\.\:]+)\t(\w)\t(\d+)\t(\d+)/) {
	my $time = $1;
	my $guid = $2;
	my $status = $3;
	my $delta = $4;
	my $full = $5;

	return undef if $status eq 'u'; # skip updates (only care about life/death)

	if (defined $CURRENT_SERVER) {
	    $GUID_MAP{$guid} = $CURRENT_SERVER;
	}

	my $ref = GetObj($guid);

	#if (!$force && $ref->[$VF] >= $vframe) {
	    # already did an update for this obj during this vframe,
	    # save this update and apply it the next vframe
	    #return $line;
	#}

	$ref->[$VF]    = $vframe;
	$ref->[$STAT]  = $status;
	if ($status eq 'n') {
	    $ref->[$CREATE_TIME] = $time;
	}

	if ($guid eq $DEBUG_EXAMINE_GUID) {
	    print STDERR "\n" . vtime() . " [DD] DELTA $status\n";
	}
    } else {
	warn "bad interest line: $line\n";
    }

    return undef;
}

sub ProcessDeltaLines
{
    my @lines = @_;

    foreach my $line (@lines) {
	ProcessDeltaLine($line, 1);
    }
}

sub ProcessDeltas
{
    my $endtime = shift;
    #my @residual;
    my $line;

    #print STDERR "** IN\n";

    my $index = 0;
    for (my $i=0; $i<@DeltaLogs; $i++) {
	$index = $i;
	my $f = $DeltaLogs[$i];
	$CURRENT_SERVER = $i;
    #foreach my $f (@DeltaLogs) {
	#$CURRENT_SERVER = $index++;
	while ($line = GetNext($f)) {
	    chomp $line;
	    $line =~ /^(\d+\.\d+)/;
	    my $time = $1;
	    if ($time >= $endtime) {
		#push @residual, $_;
		AddResid($f, $line);
		last;
	    }
	    # might get 2 updates for same object in 1 "approx" frame
	    # if so, just delay one of them until next frame
	    #print STDERR "## $i\n";
	    my $resid = ProcessDeltaLine($line);
	    #push @residual, $resid if $resid;
	    AddResid($f, $resid) if $resid;
	}
    }

    #print STDERR "** OUT\n";

    #return @residual;
}

###############################################################################

sub ProcessReplicaLine
{
    my $line  = shift;
    my $force = shift;

    if ($line =~ /^\d+\.\d+\t([\d\.\:]+)\t(\w)/) {
	my $guid = $1;
	my $status = $2;

	die "no current server!" if !defined $CURRENT_SERVER;

	if ($DestroyTypes{$status}) {
	    $TO_DELETE[$CURRENT_SERVER]->{$guid} = 1;
	    #delete $STORE[$CURRENT_SERVER]->{$guid};
	} else {
	    if ($InfoTypes{$status}) {
		$STORE[$CURRENT_SERVER]->{$guid} = 'i';
	    } elsif ($StoreTypes{$status}) {
		$STORE[$CURRENT_SERVER]->{$guid} = 'c';
	    }
	}

	if ($guid eq $DEBUG_EXAMINE_GUID) {
	    print STDERR "\n" . vtime() . " [DD] REPLICA $CURRENT_SERVER $status\n";
	}
    } else {
	warn "bad interest line: $line\n";
    }

    return undef;
}

sub ProcessReplicas
{
    my $endtime = shift;
    #my @residual;
    my $line;

    my $index = 0;
    foreach my $f (@ReplicaLogs) {
	$CURRENT_SERVER = $index++;
	while ($line = GetNext($f)) {
	    chomp $line;
	    $line =~ /^(\d+\.\d+)/;
	    my $time = $1;

	    if ($time >= $endtime) {
		AddResid($f, $line);
		last;
	    }

	    ProcessReplicaLine($line);
	}
    }
}

###############################################################################

sub OpenFiles {
    my @logs = @_;
    my @ret;

    foreach my $file (@logs) {
	#next if $file =~ /10\.1\.1\.2\:/; # skip master!
	my $fh;
	
	if ($file =~ /.gz$/) {
	    $fh = new IO::File ("gunzip -c $file | ");
	}
	else {
	    $fh = new IO::File("<$file");
	}
	die "can't open $file: $!" if !$fh;
	push @ret, [ [], $fh, 0 ];
    }

    return @ret;
}

sub GetResid($) { 
    my $ret = shift @{$_[0]->[0]}; 
    $_[0]->[2] = 0 if @{$_[0]->[0]} == 0;
    return $ret;
}
sub AddResid($@) { 
    my $log = shift;
    push @{$log->[0]}, @_; 
    $log->[2] = 1;
}
sub GetFile($) { return $_[0]->[1] }
sub GetNext($) {
    my $log = shift;
    if ($log->[2]) {
	#print STDERR "here\n";
	return GetResid($log);
    } else {
	#print STDERR "there\n";
	my $f = $log->[1];
	return <$f>;
    }
}

sub GetHeadTimes {
    my @logs = @_;
    my @ret;

    foreach my $log (@logs) {
	my $line = GetNext($log);
	die "no lines in log!" if !$line;
	chomp $line;
	$line =~ /^(\d+\.\d+)/;
	push @ret, $1;
	AddResid($log, $line);
    }

    return @ret;
}

sub ScrollUntilTime {
    my $time = shift;
    my @logs = @_;
    my @ret;

    foreach my $log (@logs) {
	my $line;

	print STDERR ".";

	while ($line = GetNext($log)) {
	    chomp $line;
	    $line =~ /^(\d+\.\d+)/;
	    my $t = $1;
	    #push @ret, $line;
	    if ($t < $time) {
		next;
	    } else {
		last;
	    }
	}
	AddResid($log, $line) if $line;
	#push @ret, $line if $line;
    }

    print STDERR "done!";

    #return @ret;
}

sub min {
    my @vals = @_;
    my $min = shift @vals;

    foreach my $v (@vals) {
	if ($min > $v) {
	    $min = $v;
	}
    }

    return $min;
}
