////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <map>
#include <hash_map.h>
#include <vector>
#include <string>
#include <list>
#include <util/debug.h>
#include <util/Options.h>
#include <gameapi/common.h>
#include <util/GPL/callback.h>
#include <util/RTree.h>

#include "putils.cxx"

double START;
float FRAME_TIME;
int SKIPTIME;
int LENGTH;
char  OUTDIR[1024];
float SLOWDOWN;
bool DETAILED;
bool DISTANCE;
int FUDGE;

const int DIMENSION = 2;    // how many co-ordinates are used { 1 | 2 | 3 }
OptionType options[] = {
    { 'o', "outdir", OPT_STR, "", OUTDIR, ".", NULL },
    { 'S', "start", OPT_DBL, "", &START, "-1.0", NULL },
    { 'l', "length", OPT_INT, "", &LENGTH, "600", NULL  },
    { 's', "skiptime", OPT_INT, "", &SKIPTIME, "120", NULL  },
    { '/', "frametime", OPT_FLT, "frame-time in seconds", &FRAME_TIME, "0.1", NULL },
    { 'w', "slowdown", OPT_FLT, "", &SLOWDOWN, "1.0", NULL },
    { 'd', "detailed", OPT_BOOL | OPT_NOARG, "", &DETAILED, "0", (void *) "1" },
    { 'D', "distance", OPT_BOOL | OPT_NOARG, "", &DISTANCE, "0", (void *) "1" },
    { 'f', "fudge", OPT_INT, "", &FUDGE, "0", NULL  },
    { 0, 0, 0, 0, 0, 0, 0 },
};

// data type definitions
struct obj;
typedef hash_map<string, struct obj *, hash_string_1, eq_string> GUIDObjMap;
typedef GUIDObjMap::iterator GUIDObjMapIter;

typedef hash_map<string, int, hash_string_1, eq_string> GUIDCliMap;
typedef GUIDCliMap::iterator GUIDCliMapIter;

typedef GUIDCliMap ObjSvrMap;
typedef GUIDCliMapIter ObjSvrMapIter;

typedef hash_map<string, char, hash_string_1, eq_string> GUIDCharMap;
typedef hash_map<string, float, hash_string_1, eq_string> GUIDFloatMap;
typedef GUIDCharMap::iterator GUIDCharMapIter;
typedef GUIDFloatMap::iterator GUIDFloatMapIter;

typedef vector<GUIDCharMap> VecGUIDHash;
typedef VecGUIDHash::iterator VecGUIDHashIter;

typedef vector<GUIDFloatMap> VecDistHash;
typedef VecDistHash::iterator VecDistHashIter;

GUIDObjMap Objects;
GUIDCliMap Clients;
ObjSvrMap  GUID_MAP;

VecGUIDHash Store;
VecGUIDHash ToDelete;

class Log {
    FILE *fp;                 // underlying log
    string name;              // log name
    int  nres;                // number of residuals
    list<string> residuals;   // residual lines to read from
public:
    Log (string& name, FILE *fp) : fp (fp), name (name), nres (0) {}

    string GetNext () {
	if (nres > 0) {
	    nres--;
	    string ret = residuals.front ();
	    residuals.pop_front ();
	    return ret;
	}
	if (feof (fp))
	    return "";

	// XXX I have no idea why this is required, but otherwise for some 
	// very unknown reason, I get old stuff across different file pointers! 
	// (some line gets read both by ReplicaLifeTimeLog and ObjectInterestLog!)
	//
	// Spent lots of time but cant figure out ... :(
	memset (line, 0, sizeof (line));
	fgets (line, sizeof (line), fp);
	return line;
    }

    void AddResid (const string& res) {
	residuals.push_back (res);
	nres++;
    }

    const string& GetName () const { return name; }
};

typedef vector<Log *> LogVec;
typedef LogVec::iterator LogVecIter;

map<char, string> InfoTypes, CreateTypes, StoreTypes, DestroyTypes;
vector<Log *> InterestLogs;
vector<Log *> DeltaLogs;
vector<Log *> ReplicaLogs;
vector<FILE *> OUT;

struct obj {
    int VF;
    char TYPE;
    vec3_t POS;
    vec3_t BBOX_MIN;
    vec3_t BBOX_MAX;
    char STAT;
    double CREATE_TIME;

    string guid;

    void GetExtent (Rect<double> *extent) {
	for (int i = 0; i < DIMENSION; i++) 
	    extent->min[i] = extent->max[i] = POS[i];
    }
};

void InitializeTypes ()
{
    InfoTypes['i'] = "OBJINFO";

    CreateTypes['m'] = "MCREATE";
    CreateTypes['c'] = "DCREATE";
    CreateTypes['p'] = "PCREATE";
    CreateTypes['f'] = "FCREATE";

    StoreTypes['s'] = "STORE";

    DestroyTypes['y'] = "MDESTROY";
    DestroyTypes['d'] = "TDESTROY";
    DestroyTypes['e'] = "EDESTROY";
}

struct obj *GetObj (char *g)
{
    string s (g);
    GUIDObjMapIter it = Objects.find (s);

    if (it == Objects.end ()) { 
	struct obj *n = new struct obj ();
	n->VF = -1;
	n->TYPE = '\0';
	n->STAT = '\0';
	n->guid = g;
	n->CREATE_TIME = -1;

	for (int i = 0; i < 3; i++) 
	    n->POS[i] = n->BBOX_MIN[i] = n->BBOX_MAX[i] = sqrt (-1.0);    // create a NaN

	Objects.insert (GUIDObjMap::value_type (n->guid, n));    
	return n;
    }
    return it->second;
}

// forward declarations
void OpenLogs (LogVec& logs, const char *pat);
void GetHeadTimes (LogVec& logs, vector<double> *times);
void ScrollUntilTime (double start, LogVec& logs);
double VecMin (vector<double> *times);
void CreateOutFiles ();

void _insert_hash (GUIDCharMap& hs, const string& s, char c)
{
    GUIDCharMapIter it = hs.find (s);
    if (it == hs.end ())
	hs.insert (pair<string, char> (s, c));
    else
	it->second = c;
}

void _insert_hash (GUIDFloatMap& hs, const string& s, float f)
{
    GUIDFloatMapIter it = hs.find (s);
    if (it == hs.end ())
	hs.insert (pair<string, float> (s, f));
    else
	it->second = f;
}

void BeginFrame ()
{
    for (GUIDObjMapIter it = Objects.begin (); it != Objects.end (); /* ++it */) {
	struct obj *o = it->second;
	// cerr << "guid=" << it->first << endl;
	if (o->STAT == 'd') {
	    GUIDObjMapIter oit (it);
	    ++oit;
	    Objects.erase (it);

	    delete o;
	    it = oit;
	}
	else {
	    ++it;
	}
    }

    for (int i = 0, n = ToDelete.size (); i < n; i++) {
	GUIDCharMap& hs = ToDelete[i];

	for (GUIDCharMapIter it = hs.begin (); it != hs.end (); ++it)
	    Store[i].erase (it->first);
	hs.clear ();
    }
}

void ProcessInterestLine (const string& log, string& interest_line, int current_server)
{
    if (!match_capture (
			"^\\d+\\.\\d+\\t\\d+\\t([\\d\\.\\:]+)\\t(\\w)\\t([-\\d\\.]+),([-\\d\\.]+),([-\\d\\.]+)\\t([-\\d\\.]+),([-\\d\\.]+),([-\\d\\.]+)\\t([-\\d\\.]+),([-\\d\\.]+),([-\\d\\.]+)",
			(char *) interest_line.c_str (), 
			11, /* npats */
			(char **) patbufs)) {
	Debug::warn ("\n(%s) bad interest line: %s", log.c_str (), interest_line.c_str ());
	return;
    }

    char *guid = patbufs[0];
    char type = *(patbufs[1]);

    if (type == 'i') {
	Objects.erase (guid);
	GUID_MAP.erase (guid);
	return; 
    }

    struct obj *o = GetObj (guid);
    if (type == 'p' || type == 'o' || type == 'm') {
	if (Clients.find (o->guid) == Clients.end ()) {
	    Clients[o->guid] = 1;
	}
    }
    if (current_server >= 0) {
	GUID_MAP[o->guid] = current_server;
    }

    o->TYPE = type;
    o->POS[0] = atof (patbufs[2]);
    o->POS[1] = atof (patbufs[3]);
    o->POS[2] = atof (patbufs[4]);

    o->BBOX_MIN[0] = atof (patbufs[5]);
    o->BBOX_MIN[1] = atof (patbufs[6]);
    o->BBOX_MIN[2] = atof (patbufs[7]);

    o->BBOX_MAX[0] = atof (patbufs[8]);
    o->BBOX_MAX[1] = atof (patbufs[9]);
    o->BBOX_MAX[2] = atof (patbufs[10]);    
}

bool ProcessInterests (double endtime)
{
    bool done = true;
    int index = 0;

    for (LogVecIter it = InterestLogs.begin (); it != InterestLogs.end (); ++it) {
	string l;

	while ((l = (*it)->GetNext ()) != "") {
	    match_capture ("^(\\d+\\.\\d+)", l.c_str (), 1, (char **) patbufs);
	    double f;
	    sscanf (patbufs[0], "%lf", &f);

	    if (f >= endtime) {
		(*it)->AddResid (l);
		done = false;
		break;
	    }

	    ProcessInterestLine ((*it)->GetName (), l, index);
	}
	index++;
    }

    return done;
}

bool ProcessDeltaLine (string& delta_line, int vframe, int current_server, bool force)
{
    if (!match_capture (
			"^(\\d+\\.\\d+)\\t\\d+\\t([\\d\\.\\:]+)\\t(\\w)\\t(\\d+)\\t(\\d+)",
			(char *) delta_line.c_str (),
			5,
			(char **) patbufs)) {
	Debug::warn ("bad delta line: %s", delta_line.c_str ());
	return false;
    }

    double tim;
    sscanf (patbufs[0], "%lf", &tim);
    char *guid = patbufs[1];
    char status = *(patbufs[2]);
    int delta = atoi (patbufs[3]);
    int full = atoi (patbufs[4]);

    if (status == 'u')
	return false;

    if (current_server >= 0)
	GUID_MAP[guid] = current_server;

    struct obj *o = GetObj (guid);
    o->VF = vframe;
    o->STAT = status;
    if (status == 'n')
	o->CREATE_TIME = tim;

    return false;
}

void ProcessDeltas (int vframe, double endtime)
{
    int index = 0;
    for (LogVecIter it = DeltaLogs.begin (); it != DeltaLogs.end (); ++it) {
	string l;

	while ((l = (*it)->GetNext ()) != "") {
	    match_capture ("^(\\d+\\.\\d+)", l.c_str (), 1, (char **) patbufs);
	    double f;
	    sscanf (patbufs[0], "%lf", &f);

	    if (f >= endtime) {
		(*it)->AddResid (l);
		break;
	    }

	    if (ProcessDeltaLine (l, vframe, index, false))
		(*it)->AddResid (l);
	}
	index++;
    }

}

void ProcessReplicaLine (string& replica_line, int current_server)
{
    if (!match_capture (
			"^\\d+\\.\\d+\\t([\\d\\.\\:]+)\\t(\\w)",
			(char *) replica_line.c_str (),
			2,
			(char **) patbufs)) {
	Debug::warn ("bad replica line: %s", replica_line.c_str ());
	return;
    }

    char *guid = patbufs[0];
    char status = *(patbufs[1]);

    ASSERT (current_server >= 0);

    if (DestroyTypes.find (status) != DestroyTypes.end ()) {
	_insert_hash (ToDelete[current_server], guid, 1);
    }
    else {
	if (InfoTypes.find (status) != InfoTypes.end ()) {
	    _insert_hash (Store[current_server], guid, 'i');
	}
	else if (StoreTypes.find (status) != StoreTypes.end ()) {
	    _insert_hash (Store[current_server], guid, 'c');
	}	    
    }
}

void ProcessReplicas (double endtime)
{
    int index = 0;
    for (LogVecIter it = ReplicaLogs.begin (); it != ReplicaLogs.end (); ++it) {
	string l;

	while ((l = (*it)->GetNext ()) != "") {
	    match_capture ("^(\\d+\\.\\d+)", l.c_str (), 1, (char **) patbufs);
	    double f;
	    sscanf (patbufs[0], "%lf", &f);

	    if (f >= endtime) {
		(*it)->AddResid (l);
		break;
	    }

	    ProcessReplicaLine (l, index);
	}
	index++;
    }
}

GUIDCliMap bbox_contains;
void collect_matches (struct obj *o) {
    bbox_contains[o->guid] = 1;
}

float Distance (vec3_t& a, vec3_t& b)
{
    vec3_t d;
    for (int i = 0; i < 3; i++)
	d[i] = (b[i] - a[i]) * (b[i] - a[i]);
    return sqrt (d[0] + d[1] + d[2]);
}

typedef vector<VecGUIDHash *> ArrayVecGUIDHash;

// each index is for a frame;
ArrayVecGUIDHash PREV_REQUIRED;
ArrayVecGUIDHash PREV_MISSING;

struct less_const_char_t {
    bool operator () (const char *a, const char *b) { 
	return strcmp (a, b) < 0;
    }
};

// These names seem to be relics from the ClientServerSim.pl script...
void ComputeCosts (int frame, double realstart)
{
    double tim = realstart + frame * FRAME_TIME;

    VecGUIDHash required, missing;
    VecDistHash dist;
    GUIDCharMap t;
    GUIDFloatMap f;

    START(ComputeCosts::FRAME_INIT);
    for (int i = 0, n = OUT.size (); i < n; i++) {
	required.push_back (t);
	missing.push_back (t);
	dist.push_back (f);
    }

    RTree<struct obj, double> rtree (DIMENSION);
    Rect<double> trect (DIMENSION);

    // insert all objects into the rtree
    for (GUIDObjMapIter it = Objects.begin (); it != Objects.end (); ++it) {
	struct obj *o = it->second;
	if (isnan (o->POS[0]))
	    continue;
	if (o->STAT == '\0')
	    continue;
	if (o->TYPE == 'i')
	    continue;

	rtree.Insert (o);
    }
    // fprintf (stderr, "frame %d inserted %d objects\n", frame, n_objs);
    STOP(ComputeCosts::FRAME_INIT);

    START(ComputeCosts::MATCHING_ETC);

    // fprintf (stderr, "frame %d clients %d\n", frame, Clients.size ());
#undef SORT_CLIENTS

#ifdef SORT_CLIENTS
    list<const char *> tclist;
    for (GUIDCliMapIter it = Clients.begin (); it != Clients.end (); ++it) {
	tclist.push_back (it->first.c_str ());
    }
    tclist.sort (less_const_char_t ());
#endif

#ifdef SORT_CLIENTS
    for (list<const char *>::iterator it = tclist.begin (); it != tclist.end (); ++it) 
#else
	for (GUIDCliMapIter it = Clients.begin (); it != Clients.end (); ++it)
#endif
	    {
#ifdef SORT_CLIENTS
		GUIDObjMapIter oit = Objects.find (*it);
#else	
		GUIDObjMapIter oit = Objects.find (it->first);
#endif

		bool jump = (oit == Objects.end () || isnan (oit->second->POS[0]));
		if (jump)
		    continue;

		struct obj *c = oit->second;
		int serverid = GUID_MAP[c->guid];

		for (int i = 0; i < DIMENSION; i++)
		    trect.min[i] = c->BBOX_MIN[i], trect.max[i] = c->BBOX_MAX[i];

		START(ComputeCosts::GetOverlaps);
		// check for overlaps this frame
		bbox_contains.clear ();
		rtree.GetOverlaps (&trect, wrap (collect_matches));
		STOP(ComputeCosts::GetOverlaps);

		START(ComputeCosts::InnerLoop);
		for (GUIDCliMapIter lit = bbox_contains.begin (); lit != bbox_contains.end (); ++lit) {
		    struct obj *oref = Objects[lit->first];
		    ASSERT (oref != NULL);

		    if (GUID_MAP.find (oref->guid) == GUID_MAP.end () || serverid == GUID_MAP[oref->guid])
			continue;

		    _insert_hash (required[serverid], oref->guid, 1);
		    float distance = Distance (c->POS, oref->POS);
		    _insert_hash (dist[serverid], oref->guid, distance);

		    GUIDCharMapIter vgit = Store[serverid].find (oref->guid);
		    if (vgit == Store[serverid].end ()) {
			_insert_hash (missing[serverid], oref->guid, 1);		
		    }
		    else if (vgit->second != 'c') {
			_insert_hash (missing[serverid], oref->guid, 1);
		    }
		    else {
		    }
		}
		STOP(ComputeCosts::InnerLoop);
	    }
    STOP(ComputeCosts::MATCHING_ETC);

    /* ok, now analyze what we've got for this frame */
    if (FUDGE > 0) {
	PREV_REQUIRED.push_back (new VecGUIDHash (required));
	PREV_MISSING.push_back (new VecGUIDHash (missing));
    }

    for (int i = 0, nout = OUT.size (); i < nout; i++) {
	int n_missing = 0, n_required = 0;
	vector<const char *> _required, _missing;
	double ftime = 0.0;

	if (FUDGE > 0) {
	    bool skip = false;
	    n_missing = 0;

	    // if we are fudging then an object is only missing if it was
	    // required FUDGE frames ago and we still have not gotten it yet

	    if (PREV_REQUIRED.size () > (uint32) FUDGE) {
		VecGUIDHash *pr = PREV_REQUIRED[0];

		for (GUIDCharMapIter git = (*pr)[i].begin (); git != (*pr)[i].end (); ++git)
		    {
			const string& guid = git->first;
			if (missing[i].find (guid) != missing[i].end ()) {
			    bool ismissing = true;
			    for (int j = 0; j < FUDGE; j++) {
				VecGUIDHash *pm = PREV_MISSING[j];

				if ((*pm)[i].find (guid) == (*pm)[i].end ()) {
				    ismissing = false;
				    break;
				}
			    }
			    if (ismissing) {
				n_missing++;
				if (DETAILED)
				    _missing.push_back (guid.c_str ());
			    }
			}
			if (DETAILED)
			    _required.push_back (guid.c_str ());
		    }

		ftime = tim - FRAME_TIME;
		n_required  = (*pr)[i].size (); 
	    } else {
		skip = true;
	    }

	    if (skip)
		continue;
	}
	else {
	    ftime      = tim;
	    n_required = required[i].size ();
	    n_missing  = missing[i].size ();

	    if (DETAILED) {
		for (GUIDCharMapIter git = required[i].begin (); git != required[i].end (); ++git)
		    _required.push_back (git->first.c_str ());
		for (GUIDCharMapIter git = missing[i].begin (); git != missing[i].end (); ++git)
		    _missing.push_back (git->first.c_str ());
	    }

	    // fprintf (stderr, "i=%d required=%d missing=%d\n", i, n_required, n_missing);
	}

	// now output the 
	FILE *fh = OUT[i];
	if (DISTANCE) {
	    for (vector<const char *>::iterator it = _missing.begin ();
		 it != _missing.end (); ++it) {
		GUIDFloatMapIter fit = dist[i].find (*it);
		ASSERT(fit != dist[i].end ());
		fprintf (fh, "1\t%.3f\t%.2f\n", ftime, fit->second);
	    }
	    for (vector<const char *>::iterator it = _required.begin ();
		 it != _required.end (); ++it) {
		GUIDFloatMapIter fit = dist[i].find (*it);
		ASSERT(fit != dist[i].end ());
		fprintf (fh, "0\t%.3f\t%.2f\n", ftime, fit->second);
	    }
	}
	else if (!DETAILED) {
	    double ratio1 = 0.0;
	    if (n_required > 0)
		ratio1 = (double) n_missing / (double) n_required;
	    fprintf (fh, "%.3f\t%d\tNA\t%d\t%.5lf\tNA\n", ftime, n_missing, n_required, ratio1);
	}
	else {
	    fprintf (fh, "%.3f\t", ftime);
	    for (vector<const char *>::iterator it = _missing.begin ();
		 it != _missing.end (); ++it) {
		if (it != _missing.begin ())
		    fprintf (fh, ",");
		fprintf (fh, "%s", *it);
	    }
	    fprintf (fh, "\t");
	    for (vector<const char *>::iterator it = _required.begin ();
		 it != _required.end (); ++it) {
		if (it != _required.begin ())
		    fprintf (fh, ",");
		fprintf (fh, "%s", *it);
	    }
	    fprintf (fh, "\n");
	}
    }

    if (FUDGE > 0 && PREV_REQUIRED.size () > (uint32) FUDGE) {
	VecGUIDHash *pr = PREV_REQUIRED.front ();
	PREV_REQUIRED.erase (PREV_REQUIRED.begin ());
	delete pr;

	VecGUIDHash *pm = PREV_MISSING.front ();
	PREV_MISSING.erase (PREV_MISSING.begin ());
	delete pm;
    }  
}

int main (int argc, char *argv[])
{
    DBG_INIT (NULL);
    InitCPUMHz();
    Benchmark::init();

    ProcessOptions (options, argc, argv, true);

    if (DISTANCE)
	DETAILED = 1;
    if (SLOWDOWN > 1.0) { 
	SKIPTIME = (int) (SKIPTIME * SLOWDOWN);
	LENGTH = (int) (LENGTH * SLOWDOWN);
	FRAME_TIME = FRAME_TIME * SLOWDOWN;
    }

    for (int i = 0; i < NPATBUFS; i++)
	patbufs[i] = new char[1024];

    InitializeTypes ();
    OpenLogs (InterestLogs, "ObjectInterestLog.*");
    OpenLogs (DeltaLogs, "ObjectDeltaLog.*");
    OpenLogs (ReplicaLogs, "ReplicaLifetimeLog.*");

    if (InterestLogs.size () == 0 || DeltaLogs.size () == 0 || ReplicaLogs.size () == 0) 
	Debug::die ("no logs to process");

    // initialize the vectors with empty hashes...
    GUIDCharMap t;
    for (int i = 0, n =  DeltaLogs.size (); i < n; i++) {
	Store.push_back (t);
	ToDelete.push_back (t);
    }

    double _start = START;
    if (START < 0) {
	vector<double> t;
	GetHeadTimes (InterestLogs, &t);
	GetHeadTimes (DeltaLogs, &t);
	GetHeadTimes (ReplicaLogs, &t);

	_start = VecMin (&t);
    }

    double realstart = _start + SKIPTIME;
    fprintf (stderr, "** scrolling until start of sim: "); fflush (stderr);

    int pframe = 0;
    double pre_frame = _start;

    while (true) {
	pre_frame += FRAME_TIME;

	BeginFrame();

	ProcessInterests (pre_frame);
	ProcessDeltas (pframe, pre_frame);
	ProcessReplicas (pre_frame);

	if (pre_frame >= realstart)
	    break;

	pframe++;
	if (pframe % 100 == 0) {
	    fprintf (stderr, ","); 
	    if (pframe % 600 == 0) {
		fprintf (stderr, "%.3f", pre_frame);
	    }
	    fflush (stderr);
	}
    }

    {
	vector<double> t;
	GetHeadTimes (InterestLogs, &t);
	GetHeadTimes (DeltaLogs, &t);
	realstart = VecMin (&t);

	fprintf (stdout, "\nSTART: ");
	for (vector<double>::iterator it = t.begin (); it != t.end (); ++it)
	    fprintf (stdout, "%.3f ", *it);
	fprintf (stdout, "\n");
    }

    CreateOutFiles ();
    double frame_end = realstart;
    int vframe = 0;

    while (true) {
	frame_end = frame_end + FRAME_TIME;

	BeginFrame ();

	START(PROCESSING_IN);
	bool done;
	done = ProcessInterests (frame_end);
	ProcessDeltas (vframe, frame_end);
	ProcessReplicas (frame_end);
	STOP(PROCESSING_IN);

	TIME(ComputeCosts (vframe, realstart));

	vframe++;
	{
	    if (vframe % 100 == 0) {
		fprintf (stderr, ".");
		if (vframe % 1200 == 0) 
		    fprintf (stderr, "%.3f", frame_end);
		fflush (stderr);
	    }
	}

	if (done || frame_end > realstart + LENGTH) {
	    fprintf (stderr, "\n ===== DONE! =====\n");
	    break;
	}
    }

    Benchmark::print();
}

void OpenLogs (LogVec& logs, const char *pat)
{
    vector<string> files;
    glob (pat, &files);

    sort (files.begin (), files.end (), less_string ());
#if 0
    cerr << "Pattern=" << pat << endl;
    for (vector<string>::iterator it = files.begin (); it != files.end (); ++it)
	cerr << "\t" << *it << endl;
    cerr << "=======================================================" << endl;
#endif

    for (vector<string>::iterator it = files.begin (); it != files.end (); ++it)
	{
	    FILE *fp = open_log (*it);
	    if (!fp) {
		perror ("fopen or popen");
		exit (1);
	    }

	    Log *log = new Log (*it, fp);
	    logs.push_back (log);
	}
}

void GetHeadTimes (LogVec& logs, vector<double> *times)
{
    for (LogVecIter it = logs.begin (); it != logs.end (); ++it) {
	string l = (*it)->GetNext ();
	if (l == "") 
	    Debug::die ("no lines in log");

	if (!match_capture ("^(\\d+\\.\\d+)", l.c_str (), 1, (char **) patbufs)) 
	    Debug::die ("no time found in log line");

	double f;
	sscanf (patbufs[0], "%lf", &f);
	times->push_back (f);
    }
}

void ScrollUntilTime (double start, LogVec& logs)
{
    string l;

    for (LogVecIter it = logs.begin (); it != logs.end (); ++it) {
	fprintf (stderr, "."); fflush (stderr);

	while ((l = (*it)->GetNext ()) != "") {
	    match_capture ("^(\\d+\\.\\d+)", l.c_str (), 1, (char **) patbufs);
	    double f;
	    sscanf (patbufs[0], "%lf", &f);
	    if (f < start)
		continue;
	    else
		break;
	}

	if (l != "")
	    (*it)->AddResid (l);
    }

    fprintf (stderr, "done!\n");
}

double VecMin (vector<double> *tims)
{
    if (tims->size () == 0)
	Debug::die ("vector size = 0 in min");

    double m = (*tims)[0];
    int siz = tims->size ();
    for (int i = 0; i < siz; i++)
	if ((*tims)[i] < m)
	    m = (*tims)[i];
    return m;
}

void CreateOutFiles ()
{
    vector<string> files;
    glob ("ObjectDeltaLog.*", &files);
    sort (files.begin (), files.end (), less_string ());

    for (vector<string>::iterator it = files.begin (); it != files.end (); ++it) {
	if (!match_capture ("(\\d+\\.\\d+\\.\\d+\\.\\d+\\:\\d+)", it->c_str (), 1, (char **) patbufs))
	    Debug::die ("bad file name %s", it->c_str ());

	string of = OUTDIR;
	of += "/consistency."; of += patbufs[0]; of += ".out";

	FILE *fp = fopen (of.c_str (), "w");
	if (!fp)
	    Debug::die ("could not open file %s for writing", of.c_str ());

	OUT.push_back (fp);
    }	
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
