#!/usr/bin/perl
#
# Usage: PlotTimeSeries.pl [-g opts|-a epoch|-R]
#
# Quick and dirty script to plot a set of time series logs and display the
# plot. If no arguments are given, it plots glob("*.log").
#
# Options:
#
# -g opts  additional gnuplot commands (comma separated)
# -e epoch reaverage the time series average over this time epoch
# -R       don't rescale the time series to start from 0
# -K       show key
# -b point bandwidth breakdown for a random node in point
#          run this from tseries/$exp/
# -m point same as above but for merc-bwith breakdown
# -s       scatter plot


use strict;
use lib "$ENV{HOME}/Colyseus/scripts";
use lib "$ENV{HOME}/Colyseus/scripts/lib";
use ResultsConf;
use Getopt::Std;
use vars qw($opt_g $opt_e $opt_R $opt_b $opt_K $opt_f $opt_m $opt_o $opt_s $opt_H);

getopts("g:e:Rb:Kf:m:osH");

my @GPLOT_OPTS = (defined $opt_g ? split(/,/, $opt_g) : ());
my $AVG_EPOCH = $opt_e;
my $RESCALE   = !$opt_R;
my $BWIDTH_BREAKDOWN_POINT = $opt_b || $opt_m;
my $KEY = $opt_K;
my $FIELD = defined $opt_f ? $opt_f : 0;
my $OUTPUT = $opt_o;
my $SCATTER = $opt_s;
my $HOURS = $opt_H;

my @files;

if (!@ARGV) {
    @files = glob("Agg*.log.gz");
} else {
    @files = @ARGV;
}

if ($opt_b || $opt_m) {
    my @dirs;
    if ($opt_b) {
	@dirs = glob("*-bwidth");
    } else {
	@dirs = glob("*");
    }
    my @all = glob("$dirs[0]/$BWIDTH_BREAKDOWN_POINT/*.log");
    die "no logs in $dirs[0]/$BWIDTH_BREAKDOWN_POINT" if !@all;
    my $pick = $all[ int(rand(scalar @all)) ];
    $pick =~ s|^.*/||g;
    foreach my $dir (@dirs) {
	my $log = "$dir/$BWIDTH_BREAKDOWN_POINT/$pick";
	die "log doesn't exist: $log" if ! -f $log;
	push @files, $log;
    }
}

my $pipe = "";
if ($AVG_EPOCH) {
    $pipe .= "| ~/Colyseus/scripts/TimeSeriesAvg.pl -e $AVG_EPOCH ";
}
if ($RESCALE) {
    my $start = `~/Colyseus/scripts/MaxStartTime.pl -M -f1 @files`;
    chomp $start;
    die "bad files (can't get start)" if !$start;
    $pipe .= "| ~/Colyseus/scripts/TimeFilter.pl -f $FIELD -l $LENGTH -S '$start' -s $SKIPTIME -r ";
    $pipe .= " -w 3600" if $HOURS;
}

my $gpl = <<EOT;
set terminal postscript color eps 20
set xlabel "Time (sec)"
set ylabel "Atttribute"
unset key
#set size 0.8,0.8
set border 31 lw 1.4
set grid ytics lw 2
set autoscale xfix
#set yrange [0:]
EOT

if ($KEY) {
    $gpl .= "set key outside below Left reverse\n";
}

if (@GPLOT_OPTS) {
    foreach my $opt (@GPLOT_OPTS) {
	$gpl .= "$opt\n";
    }
}

$gpl .= "plot ";
my $index = 0;
foreach my $f (@files) {
    my $safe = $f;
    $safe =~ s|/|_|g;
    $safe =~ s|.gz||;
    
    my $tmp = "/tmp/$safe.dat";
    my $cmd = "cat $f $pipe > $tmp";    
    if ($f =~ /.gz$/) { 
	$cmd = "gunzip -c $f $pipe > $tmp";
    }
    print STDERR "$cmd\n";
    system("$cmd") && die "stoopid";

    $gpl .= ", " if $index++ != 0;
    $gpl .= "'$tmp' title '$tmp' w " . ($SCATTER ? " points " : " lines lw 2");
}
$gpl .= "\n";

$gpl .= "  ";

use POSIX qw(tmpnam);
my $of = tmpnam ();
if ($OUTPUT) {
   open G, "| gnuplot";
} else {
   open G, "| gnuplot > $of.ps";
}
print G $gpl;
close G;

open T, ">/tmp/plot.gnuplot";
print T $gpl;
close T;

system("gv $of.ps") if !$OUTPUT;
