#!/usr/bin/perl
#
# Make a random latency graph from the p2psim e2e-topology. There is a
# p2psim es2 graph at: Data/topologies/p2psim-kingdataset-29062004.data.
#
# More may be found (maybe) at:
# http://www.pdos.lcs.mit.edu/p2psim/
#
# Usage: P2PSimGraphToVHostLats.pl [options] <p2psim-e2e-file>
#
# Options:
#
# -a addr       increment for additional vservers (e.g., 0.0.0.1 will give you
#               127.0.0.3, 127.0.0.4, etc.)
# -h file       base addresses from file
# -e            emulab base addresses
# -l            localhost base address
# -n nservers   number of servers
# -v count      vservers per host
# -s scale      scale the latencies by this much
# -o kbps       outgoing bwidth per node (default: undef)
# -i kbps       incoming bwidth per node (default: undef)

use strict;
use lib "$ENV{HOME}/Colyseus/run";
use Travertine;
use Statistics::Descriptive;
use Getopt::Std;
use vars qw($opt_a $opt_b $opt_i $opt_o $opt_e $opt_l $opt_n $opt_h $opt_v $opt_s $opt_r);
use Net::DNS;

getopts("b:a:elh:n:v:s:i:o:r");

our @INCREMENT    = $opt_a ? split(/\./, $opt_i) : (0,0,0,1);
our $NUM_NODES    = $opt_n || 10;
our $NUM_VSERVERS = $opt_v || 1;
our $EMULAB       = $opt_e;
our $LOCAL        = $opt_l;
our $HOSTS_FILE   = $opt_h;
our $SCALE        = $opt_s || 1;
our $OBWIDTH      = defined $opt_o ? $opt_o*1000 : -1;
our $IBWIDTH      = defined $opt_i ? $opt_i*1000 : -1;
our $RESOLVE_IP   = $opt_r;

our @IPS = ();
our @HOSTS = ();

## read a hosts file, and lookup IPs for the hosts
if (defined $HOSTS_FILE) { 
    my $resolver = new Net::DNS::Resolver;
    @HOSTS = GetHosts ($HOSTS_FILE);
    foreach my $h (@HOSTS) { 
	my ($addr, $vbase) = split /\:/, $h;
	my $ip = $RESOLVE_IP ? ResolveIP ($resolver, $addr) : $addr;
	$ip = $addr if not defined $ip or $ip == "";
	push @IPS, [$ip,$vbase];
    }
}

if ($EMULAB) {
    @INCREMENT = (0,0,1,0);
    for (my $i=0; $i<$NUM_NODES; $i++) {
	my $oct = $i+2;
	my $ip = $RESOLVE_IP ? "10.1.1.$oct" : "node$i";
	my $vb = "10.$oct.1.1";
	push @IPS, [$ip,$vb];
    }
}

if ($LOCAL) {
    push @IPS, [$RESOLVE_IP ? "127.0.0.1" : "localhost","127.1.0.1"];
    $NUM_NODES = 1;
}

tdie "too many vservers $NUM_VSERVERS > 255" if $NUM_VSERVERS > 255;

our $stats = new Statistics::Descriptive::Full();

my @nodes = ();
my $inited = 0;
my %nodemap = ();
my @lats;

while (<>) {
    chomp $_;

    next if $_ =~ /^\s*\#/;

    if ($_ =~ /\s*node\s+(\d+)/) {
	push @nodes, $1; 
    }

    if ($_ =~ /(\d+),(\d+)\s+([\-\d\.]+)/) {
	my $from = $1;
	my $to = $2;
	my $lat = $3;

	if (!$inited) {
	    if (@nodes < $NUM_NODES*$NUM_VSERVERS) {
		die "not enough nodes in graph!";
	    }
	    permute(\@nodes);

	    my $index = 0;

	    for (my $i=0; $i<$NUM_NODES*$NUM_VSERVERS; $i++) {
		
		my $addr  = $IPS[$index]->[0];
		my $vbase = $IPS[$index]->[1];

		my @vip = split(/\./, $vbase);
		for (my $j=0; $j<4; $j++) {
		    $vip[$j] += $INCREMENT[$j]*($i % $NUM_VSERVERS);
		}

		my $name = [ $addr, join(".", @vip) ];
		$nodemap{$nodes[$i]} = $name;
		
		print "node $nodes[$i] $name->[0] $name->[1] $OBWIDTH $IBWIDTH\n";

		if ($i % $NUM_VSERVERS == $NUM_VSERVERS-1) {
		    $index++;
		}
	    }

	    $inited = 1;
	}

	next if !defined $nodemap{$from} || !defined $nodemap{$to};

	if ($lat >= 0) {
	    $lat   = sprintf("%.3f", $SCALE*$lat/1000); # to ms;
	    $stats->add_data( $lat );
	}
	push @lats, [ "$from,$to", $lat ];
    }
}

my $median_lat = $stats->median();

foreach my $p (@lats) {
    my ($pair, $lat) = @$p;
    $lat = $median_lat if $lat < 0;
    print "$pair $lat\n";
}

print STDERR "mean:   " . ($stats->mean()) . "\n";
print STDERR "stddev: " . ($stats->standard_deviation()) . "\n";
print STDERR "median: $median_lat\n";

sub permute($)
{   
    my $lines = shift;
    my $total = scalar @$lines;
    
    # first do a permutation
    for (my $i=0; $i<$total; $i++) {
	my $pick = int(rand($total - 1 - $i));
	my $last = $lines->[$total - 1 - $i];
	$lines->[$total - 1 - $i] = $lines->[$pick];
	$lines->[$pick] = $last;
    }
}

sub ResolveIP($$) {
    my ($res, $host) = @_;
    my $ip = undef;

    my $q = $res->search ($host);
    if ($q) {
	foreach my $rr ($q->answer) {
	    next unless $rr->type eq "A";
	    $ip = $rr->address;
	    last;
	}
    }
    else {
	print STDERR "lookup failed for ($host): ", $res->errorstring, "\n";
    }
    return $ip;
}

sub GetHosts($) {
    my $hfile = shift;
    my @hosts;

    open(H, "<$hfile") || die "can't open $hfile: $!";
    while (<H>) {
	chomp $_;
	next if $_ =~ /^\s*\#/;
	$_ =~ s/^\s+//;
	$_ =~ s/\s*\#.*$//;
	if ($_ =~ /[\w\-_\.]+/) {
	    push @hosts, $_;
	} else {
	    die "bad host in $hfile: $_";
	}
    }
    close(H);

    return @hosts;
}

