#!/usr/bin/perl
#
# Make a random latency graph from the p2psim e2e-topology. There is a
# p2psim es2 graph at: Data/topologies/p2psim-kingdataset-29062004.data.
#
# More may be found (maybe) at:
# http://www.pdos.lcs.mit.edu/p2psim/
#
# Usage: P2PSimGraphToEmulabLats.pl [options] <p2psim-e2e-file>
#
# Options:
#
# -l            generate for localhost (i.e., 1 node, bs = localhost:15000)
# -b addr:port  bootstrap server
# -x prefix     node name prefix (i.e., nodes are ${prefix}1, ${prefix}2, etc.)
# -p port       base mercury port for the first vserver.
# -n nservers   number of distinct nachines
# -v vservers   number of vservers per node (>= 1)
# -s scale      scale the latencies by this much
# -i            generate IP addresses for emulab
# -r rports     other ports, relative to the mercury port (default: 5000)
#               for each node (comma separated).
# -h hosts      use hosts from a file
#

use strict;
use Statistics::Descriptive;
use Getopt::Std;
use vars qw($opt_l $opt_b $opt_h $opt_x $opt_p $opt_n $opt_v $opt_s $opt_i $opt_r);
use Net::DNS;

getopts("lb:h:p:n:v:s:x:ir:");

our $LOCAL = $opt_l;
our $NODE_PREFIX  = $opt_x || "node"; # will be addressed by name "nodeXX"
our $INIT_PORT    = defined $opt_p ? $opt_p : 20000;  # merc port for the first vserver
our @RELATIVE_PORTS = defined $opt_r ? (split(/,/, $opt_r)) : (5000);
our $NUM_NODES    = defined $opt_n ? $opt_n : 10;
our $NUM_VSERVERS = defined $opt_v ? $opt_v : 1;
our $SCALE        = defined $opt_s ? $opt_s : 1;
our $GENERATE_IP  = defined $opt_i;
our $BOOTSTRAP    = defined $opt_b ? $opt_b : "node1:15000";
our $HOSTS_FILE   = $opt_h;
our @IPS = ();
our @HOSTS = ();

## read a hosts file, and lookup IPs for the hosts
if (defined $HOSTS_FILE) { 
    my $resolver = new Net::DNS::Resolver;
    @HOSTS = GetHosts ($HOSTS_FILE);
    foreach my $h (@HOSTS) { 
	my $ip = ResolveIP ($resolver, $h);
	$ip = $h if not defined $ip or $ip == "";
	push @IPS, $ip;
    }
}

# fix bootstrap
if (!defined $opt_b) {
    if ($GENERATE_IP) {
	if (defined $HOSTS_FILE) {  $BOOTSTRAP = "$IPS[0]:15000"; }
	elsif ($LOCAL) { $BOOTSTRAP = "127.0.0.1:15000"; }
	else { $BOOTSTRAP = "10.1.1.2:15000"; }
    }
    else {
	if (defined $HOSTS_FILE) { $BOOTSTRAP = "$HOSTS[0]:15000"; }
	elsif ($LOCAL) { $BOOTSTRAP = "127.0.0.1:15000"; }
	else { $BOOTSTRAP = "node0:15000"; }
    }
}

if ($LOCAL) {
    $NUM_NODES = 1;
    $NUM_VSERVERS = $NUM_NODES*$NUM_VSERVERS;
    $BOOTSTRAP = "127.0.0.1:15000";
}

our $stats = new Statistics::Descriptive::Full();

my @nodes = ();
my $inited = 0;
my %nodemap = ();
my @lats;

while (<>) {
    chomp $_;

    next if $_ =~ /^\s*\#/;

    if ($_ =~ /\s*node\s+(\d+)/) {
	push @nodes, $1; 
    }

    if ($_ =~ /(\d+),(\d+)\s+([\-\d\.]+)/) {
	my $from = $1;
	my $to = $2;
	my $lat = $3;

	if (!$inited) {
	    if (@nodes < $NUM_NODES*$NUM_VSERVERS) {
		die "not enough nodes in graph!";
	    }
	    permute(\@nodes);

	    my $index = 0;

	    for (my $i=0; $i<$NUM_NODES*$NUM_VSERVERS; $i++) {
		if ($i % $NUM_VSERVERS == 0) {
		    $index++;
		}
		my $port = $INIT_PORT + ($i % $NUM_VSERVERS);
		# merc, manager, xxx: master quake port? (50000)
		my @ports = ($port, map { $port + $_ } @RELATIVE_PORTS);
		my $name;
		
		if ($GENERATE_IP) {
		    if (defined $HOSTS_FILE) { $name = $IPS[$index - 1]; }
		    elsif ($LOCAL) { $name = "127.0.0.1"; }
		    else { $name = "10.1.1." . ($index + 1); }  # pretty hacky! XXX;
		}
		else {
		    if (defined $HOSTS_FILE) { $name = $HOSTS[$index - 1]; }
		    elsif ($LOCAL) { $name = "127.0.0.1"; }
		    else { $name = "$NODE_PREFIX" . ($index - 1); }
		}
		my @names = map { "$name:$_" } @ports;

		$nodemap{$nodes[$i]} = \@names;;
		print "node $nodes[$i] " . join(" ", @names) . "\n";
	    }
	    print "node 99999 $BOOTSTRAP\n";

	    $inited = 1;
	}

	next if !defined $nodemap{$from} || !defined $nodemap{$to};

	if ($lat >= 0) {
	    $lat   = sprintf("%.3f", $SCALE*$lat/1000); # to ms;
	    $stats->add_data( $lat );
	}
	push @lats, [ "$from,$to", $lat ];
    }
}

my $median_lat = $stats->median();

foreach my $p (@lats) {
    my ($pair, $lat) = @$p;
    $lat = $median_lat if $lat < 0;
    print "$pair $lat\n";
}
foreach my $k (keys %nodemap) {
    # no latency to bootstrap
    print "$k,99999 0\n";
}

print STDERR "mean:   " . ($stats->mean()) . "\n";
print STDERR "stddev: " . ($stats->standard_deviation()) . "\n";
print STDERR "median: $median_lat\n";

sub permute($)
{   
    my $lines = shift;
    my $total = scalar @$lines;
    
    # first do a permutation
    for (my $i=0; $i<$total; $i++) {
	my $pick = int(rand($total - 1 - $i));
	my $last = $lines->[$total - 1 - $i];
	$lines->[$total - 1 - $i] = $lines->[$pick];
	$lines->[$pick] = $last;
    }
}

sub ResolveIP($$) {
    my ($res, $host) = @_;
    my $ip = undef;

    my $q = $res->search ($host);
    if ($q) {
	foreach my $rr ($q->answer) {
	    next unless $rr->type eq "A";
	    $ip = $rr->address;
	    last;
	}
    }
    else {
	print STDERR "lookup failed for ($host): ", $res->errorstring, "\n";
    }
    return $ip;
}

sub GetHosts($) {
    my $hfile = shift;
    my @hosts;

    open(H, "<$hfile") || die "can't open $hfile: $!";
    while (<H>) {
	chomp $_;
	next if $_ =~ /^\s*\#/;
	$_ =~ s/^\s+//;
	$_ =~ s/\s*\#.*$//;
	if ($_ =~ /[\w\-_\.]+/) {
	    push @hosts, $_;
	} else {
	    die "bad host in $hfile: $_";
	}
    }
    close(H);

    return @hosts;
}

