#!/usr/bin/perl
#
# Calculate the time breakdown a subscription is generated and a new object
# is discovered, located, and constructed.
#
# $Id: NewSubDiscoveryTime.pl 2734 2006-03-21 01:23:08Z ashu $
#
# Usage: NewSubDiscoveryTime.pl DiscoveryLog.*
#
# -a  - show all (instead of mean/stddev/95%)
# -T  - track these many complete messages at MAX
# -S  - use this as the starttime (before skipping)
# -s  - skip time
# -l  - length of time
# -b  - binary logs
# -n  - ignore negative values < -0.1
#

use strict;
use Statistics::Descriptive;
use Getopt::Std;
use Time::HiRes qw(gettimeofday tv_interval);
use vars qw($opt_S $opt_a $opt_s $opt_b $opt_l $opt_T $opt_A $opt_w $opt_n);

my $basedir;
chomp ($basedir = `dirname $0`);
require "$basedir/Common.pl";

our $TOPDIR = "$basedir/../";
our %MsgTypes;

getopts("aS:s:l:bT:Aw:n");

our $SHOWALL  = defined $opt_a;
our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 0;
our $LENGTH   = defined $opt_l ? $opt_l : undef;
our $MAXSAMPLE = $opt_T || 5000;
our $LOG_BINARY = defined $opt_b;
our $AGGLOGS = 0; # defined $opt_A;
$LOG_BINARY = 0 if ($AGGLOGS);

our $SLOWDOWN = defined $opt_w ? $opt_w : 1;
our $IGNORE_NEGATIVE = defined $opt_n;

if ($SLOWDOWN > 1.0) { 
    $SKIPTIME *= $SLOWDOWN;
    $LENGTH *= $SLOWDOWN;
}

###############################################################################

# timing related stuff 
my %timers;
my %sttime;
my %iters;
sub rec {
    $timers{$_[0]} += $_[1];
    $iters{$_[0]}++;
}

sub start($) { 
    $sttime{$_[0]} = [gettimeofday()];
}
sub stop($) { 
    rec($_[0], tv_interval($sttime{$_[0]},[gettimeofday()])); 
}
sub print_timers () {   
    print STDERR "benchmark timers:\n";
    foreach my $t (sort { -($timers{$a} <=> $timers{$b}) } keys %timers) {
	print STDERR "$t\t$timers{$t}\t" . ($timers{$t}/$iters{$t}) . "\t$iters{$t}\n";
    }
}

ParseObjectLogsHeader();

our $UPDATE_DONE_INDEX = 13;         # if the loop fails, we still have something to bank on :)
our $ALIAS_INDEX = 6;
our $PUB_SEND_INDEX = 4;
our $SUB_SEND_INDEX = 1;
our $PUB_STORE_INDEX = 5;
our $MATCH_SEND_INDEX = 7;

foreach my $k (keys %MsgTypes) 
{
    $UPDATE_DONE_INDEX = $k if ($MsgTypes{$k} eq "UPDATE_DONE") ;
    $ALIAS_INDEX = $k if ($MsgTypes{$k} eq "ALIAS");    
    $PUB_SEND_INDEX = $k if ($MsgTypes{$k} eq "PUB_SEND");
    $PUB_STORE_INDEX = $k if ($MsgTypes{$k} eq "PUB_STORE");
    $SUB_SEND_INDEX = $k if ($MsgTypes{$k} eq "SUB_SEND");
    $MATCH_SEND_INDEX = $k if ($MsgTypes{$k} eq "MATCH_SEND");
}

my %tracked = ProcessLogs(@ARGV);
my %stats   = ();

sub IgnoreType 
{
    my $type = shift;

# XXX: check PUB_RECV
    my $ret = $MsgTypes{$type} eq 'ALIAS' ||
	$MsgTypes{$type} eq 'PUB_INIT' ||
	$MsgTypes{$type} eq 'PUB_SEND' ||
	$MsgTypes{$type} eq 'SUB_SEND' ||
	$MsgTypes{$type} eq 'PUB_RECV' ||
	$MsgTypes{$type} eq 'PUB_STORE' ||
	$MsgTypes{$type} eq 'PUB_ROUTE_RECV' ||
	$MsgTypes{$type} eq 'PUB_ROUTE_SEND' ||
	$MsgTypes{$type} eq 'SUB_ROUTE_RECV' ||
	$MsgTypes{$type} eq 'SUB_ROUTE_SEND';

    if ($AGGLOGS) { 
	$ret ||= $MsgTypes{$type} eq 'SUB_STORE';
	$ret ||= $MsgTypes{$type} eq 'PUB_RECV';
	$ret ||= $MsgTypes{$type} eq 'SUB_SEND';
    }
    return $ret;
}

print sprintf("%12s ", "ID");
foreach my $type (sort { $a <=> $b } keys %MsgTypes) { 
    next if IgnoreType($type);

    print sprintf("%12s ", $MsgTypes{$type});
    $stats{$type} = new Statistics::Descriptive::Full();
}
print sprintf("%12s ", "IDLE");
$stats{IDLE_TIME} = new Statistics::Descriptive::Full();
print sprintf("%12s", "TOTAL");
$stats{"TOTAL"} = new Statistics::Descriptive::Full();
print "\n";

# get rid of negative valued IDs, if appropriate ;
if ($IGNORE_NEGATIVE) {
OUTER:
    foreach my $t (keys %tracked) {
	my $start_time;
	my $last_time;

	foreach my $type (sort { $a <=> $b } keys %MsgTypes) {
	    next if IgnoreType($type);			

	    my $v;
	    if (!defined $start_time) {
		$start_time = $tracked{$t}->{$type};
		$v = 0;
	    } else {
		$v = $tracked{$t}->{$type} - $last_time;
	    }

	    if ($v < -0.1) {
		delete $tracked{$t};
		next OUTER;
	    }

	    $last_time = $tracked{$t}->{$type};
	}
    }
}

# now report;
foreach my $t (keys %tracked) {
    my $start_time;
    my $last_time;

    print sprintf("%12s ", $t) if $SHOWALL;
    foreach my $type (sort { $a <=> $b } keys %MsgTypes) {
	next if IgnoreType($type);			

	my $v;
	if (!defined $start_time) {
#die "first type not SUB_INIT: $MsgTypes{$type}($type)" 
#if ($MsgTypes{$type} ne "SUB_INIT");
	    $start_time = $tracked{$t}->{$type};
	    $v = 0;
	} else {
	    $v = $tracked{$t}->{$type} - $last_time;
	}

	$v /= $SLOWDOWN;      # re-scale the latencies;
	
	print sprintf("%12.3f ", $v) if $SHOWALL;
	$stats{$type}->add_data( $v );
	
	$last_time = $tracked{$t}->{$type};
    }

    $start_time /= $SLOWDOWN;
    $last_time /= $SLOWDOWN;

    # print route idle time 
    
    $tracked{$t}->{IDLE_TIME} /= $SLOWDOWN;
	
    print sprintf("%12.3f ", $tracked{$t}->{IDLE_TIME}) if $SHOWALL;
    $stats{IDLE_TIME}->add_data($tracked{$t}->{IDLE_TIME});

    print sprintf("%12.3f", $last_time - $start_time) if $SHOWALL;
    $stats{"TOTAL"}->add_data($last_time - $start_time);
    print "\n" if $SHOWALL;
}

foreach my $f ( [ "MEAN",   sub { $_[0]->mean() } ],
	[ "STDDEV", sub { $_[0]->standard_deviation() } ],
	[ "MEDIAN", sub { $_[0]->median() } ],
	[ "95 %",   sub { $_[0]->percentile(95) } ] ) {
    print sprintf("%12s ", $f->[0]) if !$SHOWALL;
    foreach my $type (sort { $a <=> $b } keys %MsgTypes) {
	next if IgnoreType($type);
	print sprintf("%12.3f ", &{$f->[1]}($stats{$type}) ) if !$SHOWALL;
    }
    print sprintf("%12.3f ", &{$f->[1]}($stats{IDLE_TIME})) if !$SHOWALL;
    print sprintf("%12.3f", &{$f->[1]}($stats{"TOTAL"})) if !$SHOWALL;
    print "\n" if !$SHOWALL;
}

sub show_progress { 
    my ($done, $total) = @_;
    print STDERR sprintf ("%.2f %% done \r", ($done * 100.0 / $total));
}

###############################################################################
# first-pass: get a list of IDs to track. sample them coz we can't track all of them.
sub SampleTrackingIDs
{
    my @files = @_;
    my $size = 0;
    my @to_track = ();

    print STDERR "first pass ... sampling tracking ids\n";

# first-pass: get a list of IDs to track. sample them coz we can't track all of them.
# the number of IDs is so huge that creating such a big hash-table would make perl go nuts (and out of memory)

    my $nf = 0;
    
    foreach my $f (@files) {
	if ($LOG_BINARY) {
	    open F, "$basedir/ParseDiscoveryLatLog $f | " or die "cant open pipe from file $f ($!)";
	}
	else {
	    if ($f =~ /.gz$/) { 
		open F, "gunzip -c $f | " or die " cant gunzip on $f: $!";
	    }
	    else {
		open(F, "<$f") || die "can't open $f ($!)";
	    }
	}
	
	$nf += 1;
	show_progress ($nf, scalar @files);

	start('FrameTime');
	my($start, $end);
	$start = $START;
	while (<F>) {
	    chomp $_;
	    m/(\d+\.\d+)\t(\d+)\t\d+\t([0-9A-F]{8}?)/;

	    my $time = $1;
	    my $type = $2;
	    my $id   = $3;

	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    if ($type == $UPDATE_DONE_INDEX) {  # avoid string comparisons
		if (int(rand(2)) == 1)  { # ok; we decide to sample
		    if ($size == $MAXSAMPLE) { 
			$to_track[int(rand($size))] = $id;
		    }
		    else {
			push @to_track, $id;
			$size++;
		    }
		}
	    }
	}
	stop ('FrameTime');
	close F;
    }

    print STDERR "sampled $size ids...\n";
    my %track_ids = ();
    map { $track_ids{$_} = {} } @to_track;
    return %track_ids;
}

sub ConstructAliasMap
{
    my $trackmap_ptr = shift;
    my $amap_ptr = shift;
    my $emap_ptr = shift;
    my @files = @_;

    my %pass_permitted = ();

# in the first pass, only permit the IDs we want to track to be 
# reverse-aliased. these are the leaves of the tree.
#
    foreach my $id (keys %$trackmap_ptr) {
	$pass_permitted{$id} = 0;
    }

# aliases can get created only 2 times; first time when/if the pub is sent out 
# to multiple hubs, and the second time at the rendezvous node.
#  
# so finding the reverse maps starting from the trackids will 
# need at max 2 passes 

    print STDERR "Constructing alias maps...\n";
    for (my $pass = 0; $pass < 2; $pass++) 
    {
	print STDERR "\tPass $pass\n";

	my $nf = 0;
	
	foreach my $f (@files) {
	    if ($LOG_BINARY) {
		open F, "$basedir/ParseDiscoveryLatLog $f | " or die "cant open pipe from file $f ($!)";
	    }
	    else {
		if ($f =~ /.gz$/) { 
		    open F, "gunzip -c $f | " or die " cant gunzip on $f: $!";
		}
		else {
		    open(F, "<$f") || die "can't open $f ($!)";
		}
	    }

	    $nf += 1;
	    show_progress ($nf, scalar @files);

	    my($start, $end);
	    $start = $START;
	    while (<F>) {
		chomp $_;
		m/(\d+\.\d+)\t(\d+)\t(\d+)\t([0-9A-F]{8}?)(.*)$/;

		my $time = $1;
		my $type = $2;
		my $hops = $3;
		my $id   = $4;
		my $alias= $5; $alias =~ s/\t//;

		if (!defined $start) {
		    $start = $time;
		}
		if ( $time < $start + $SKIPTIME ) {
		    next;
		}
		if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		    last;
		}
		if ($time > $end) {
		    $end = $time;
		}

# avoid the hash lookup
#				if (!defined $MsgTypes{$type}) {
#					warn "Unknown type: $type; processing file $f";
#					next;
#				}
#				$type = $MsgTypes{$type};

		next if ($type != $ALIAS_INDEX);
		next if ((!exists $pass_permitted{$alias}) || $pass_permitted{$alias} != $pass);

		$amap_ptr->{$id} = [] if !defined $amap_ptr->{$id};
		push @{$amap_ptr->{$id}}, $alias;
		$pass_permitted{$id} = $pass + 1;              # the next pass should record info for this guy;

		$emap_ptr->{$alias}->{rev_aliases} = [] if (!defined $emap_ptr->{$alias}->{rev_aliases});
		push @{$emap_ptr->{$alias}->{rev_aliases}}, $id;

	    }
	    close F;
	}
    }
}

sub CheckForCollisions 
{
    my $trackmap_ptr = shift;
    my $amap_ptr = shift;
    my $emap_ptr = shift;
    my @files = @_;

    print STDERR "Checking for collisions in the tracked IDs...\n";
    my %alias_creation_count = ();
    my %hops_tracker = ();

    my $nf = 0;
    
    foreach my $f (@files) {
	if ($LOG_BINARY) {
	    open F, "$basedir/ParseDiscoveryLatLog $f | " or die "cant open pipe from file $f ($!)";
	}
	else {
	    if ($f =~ /.gz$/) { 
		open F, "gunzip -c $f | " or die " cant gunzip on $f: $!";
	    }
	    else {
		open(F, "<$f") || die "can't open $f ($!)";
	    }
	}

	$nf += 1;
	show_progress ($nf, scalar @files);

	my($start, $end);
	$start = $START;

	while (<F>) {
	    chomp $_;
	    m/(\d+\.\d+)\t(\d+)\t(\d+)\t([0-9A-F]{8}?)(.*)$/;

	    my $time = $1;
	    my $type = $2;
	    my $hops = $3;
	    my $id   = $4;
	    my $alias= $5; $alias =~ s/\t//;

	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

            # look at this alias creation; if the created alias exists 
            # in our reverse alias map then it is of interest. 
	    
	    if ($type == $PUB_SEND_INDEX || $type == $SUB_SEND_INDEX || $type == $PUB_STORE_INDEX || $type == $MATCH_SEND_INDEX) 
	    {
		next if ((!exists $amap_ptr->{$id}) && (!exists $emap_ptr->{$id}));
		if ($type == $PUB_STORE_INDEX) { 
		    my $prev_hops = $hops_tracker{$id};
		    $hops_tracker{$id} = $hops;
		    next if (defined $prev_hops && $prev_hops != $hops);
		}
		$alias_creation_count{$id}++;
	    }
	}
	close F;
    }

    my $dups = 0;
    foreach my $id (keys %alias_creation_count) {

	if ($alias_creation_count{$id} > 1) {           
	    # warn "id $id was created $alias_creation_count{$id} times, throwing away... ";
	    $dups++;
	    delete $emap_ptr->{$id};

            # delete anything pointing to this alias...
	    my $aref = $amap_ptr->{$id};
	    if (defined $aref) {
		foreach my $a (@$aref) {
		    delete $emap_ptr->{$a};
		}
	    }
	    delete $amap_ptr->{$id};
	}
    }

    print STDERR "* threw $dups duplicate IDs away...\n";
}

sub RecordInitTimes 
{
    my $trackmap_ptr = shift;
    my $amap_ptr = shift;
    my $emap_ptr = shift;
    my @files = @_;
    my $nf = 0;

    print STDERR "Computing pubsub init times and routing delays...\n";
    foreach my $f (@files) {
	if ($LOG_BINARY) {
	    open F, "$basedir/ParseDiscoveryLatLog $f | " or die "cant open pipe from file $f ($!)";
	}
	else {
	    if ($f =~ /.gz$/) { 
		open F, "gunzip -c $f | " or die " cant gunzip on $f: $!";
	    }
	    else {
		open(F, "<$f") || die "can't open $f ($!)";
	    }
	}
	my($start, $end);
	$start = $START;

	$nf += 1;
	show_progress ($nf, scalar @files);

	while (<F>) {
	    chomp $_;
	    m/(\d+\.\d+)\t(\d+)\t(\d+)\t([0-9A-F]{8}?)(.*)$/;

	    my $time = $1;
	    my $type = $2;
	    my $hops = $3;
	    my $id   = $4;
	    my $alias= $5; $alias =~ s/\t//;

#		my ($time, $type, $hops, $id, $alias) = split(/\t/, $_);

	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    next if (!defined $amap_ptr->{$id});
	    if (!defined $MsgTypes{$type}) {
		warn "Unknown type: $type";
		next;
	    }
	    $type = $MsgTypes{$type};

	    if ($type eq 'SUB_INIT') {     # dont need to walk the reverse map back
		$emap_ptr->{$id}->{sub_init_time} = $time;
	    }
	    elsif ($type eq 'PUB_INIT') {  # dont need to walk the reverse map back
		$emap_ptr->{$id}->{pub_init_time} = $time;
	    }
	    elsif ($type eq 'PUB_ROUTE_RECV') {
		my $rid = GetTreeRoot($id, $emap_ptr);
		$emap_ptr->{$rid}->{tpr} = $time;
	    }
	    elsif ($type eq 'PUB_ROUTE_SEND') {
		my $rid = GetTreeRoot($id, $emap_ptr);
		$emap_ptr->{$rid}->{pub_idle_time} += ($time - $emap_ptr->{$rid}->{tpr});
	    }
	    elsif ($type eq 'SUB_ROUTE_RECV') {
		my $rid = GetTreeRoot($id, $emap_ptr);
		$emap_ptr->{$rid}->{tsr} = $time;
	    }
	    elsif ($type eq 'SUB_ROUTE_SEND') {
		my $rid = GetTreeRoot($id, $emap_ptr);
#print STDERR "adding idletime=" , ($time - $emap_ptr->{$rid}->{tsr});
		$emap_ptr->{$rid}->{sub_idle_time} += ($time - $emap_ptr->{$rid}->{tsr});
	    }
	}
	close F;
    }
}

#
# Walk the reverse alias map UPwards to get to the SUB_INIT or SUB_SEND 
# node. As long as the walk starts from somewhere BEFORE the matching happens, 
# it is guaranteed to have exactly one "start" point.
# 
sub GetTreeRoot {
    my $rid = shift;
    my $emap_ptr = shift;
    while (defined $emap_ptr->{$rid} && defined $emap_ptr->{$rid}->{'rev_aliases'}) {
	$rid = $emap_ptr->{$rid}->{'rev_aliases'}->[0];
    }

    return $rid;
}

sub MarkSubTravel 
{
    my $trackmap_ptr = shift;
    my $amap_ptr = shift;
    my $emap_ptr = shift;
    my @files = @_;
    my $nf = 0;

#########################################################;
#  For each sub, we need to the following steps:   
#
# 1.  find all updates that resulted due to this sub 
#      
# 2.  foreach of these updates, follow them back to the PUB_SEND which resulted 
#       in the update.
#       
# 3.  only look at those pubs whose PUB_SEND or PUB_INIT time <= SUB_INIT time
# 
# 4.  finally, for all the pubs that do qualify, report timings for each step. 
#

    print STDERR "tracking the desired ids now...\n";
    foreach my $f (@files) {
	if ($LOG_BINARY) {
	    open F, "$basedir/ParseDiscoveryLatLog $f | " or die "cant open pipe from file $f ($!)";
	}
	else {
	    if ($f =~ /.gz$/) { 
		open F, "gunzip -c $f | " or die " cant gunzip on $f: $!";
	    }
	    else {
		open(F, "<$f") || die "can't open $f ($!)";
	    }
	}
	$nf += 1;
	show_progress ($nf, scalar @files);
	
	my($start, $end);
	$start = $START;
	while (<F>) {
	    chomp $_;
	    m/(\d+\.\d+)\t(\d+)\t(\d+)\t([0-9A-F]{8}?)(.*)$/;

	    my $time = $1;
	    my $type = $2;
	    my $hops = $3;
	    my $id   = $4;
	    my $alias= $5; $alias =~ s/\t//;

	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    next if $MsgTypes{$type} eq 'ALIAS';

	    my @end_aliases = FindEndAlias($id, $amap_ptr);			

	    foreach my $end_alias (@end_aliases) {
		next if (!defined $trackmap_ptr->{$end_alias}); # we still need to check this because we are looking at all ids here and FindEndAlias does not return NULL if the alias_map is not defined for this id;

# follow the end-alias back and get the beginning points of the
# pubs and the subs associated with the update (equiv. the match); 
# 
		my ($pub_init, @sub_inits);
		($pub_init, @sub_inits) = GetPubsubInits($emap_ptr, $end_alias);

# if initial few seconds are not taken into account, some of the 
# initial phases for some IDs will not be present.
#
		if (!defined $pub_init) {
# warn "pubinit inconsistency for id($id) -> endalias($end_alias) !\n";
		    next;
		}
		if (scalar @sub_inits == 0) {
# warn "subinit inconsistency for id($id) -> endalias($end_alias) !\n";
		    next;
		}
		@sub_inits = sort { $a <=> $b } @sub_inits;

# check if this sub should be associated with the discovery of the 
# object.  if this sub was inited before the pub was sent, we
# will not care assuming the update really need not have gone
# to the subscriber. 
# 
		my $sub_idle = undef;
		if ($MsgTypes{$type} =~ /SUB/) {
		    my $root = GetTreeRoot($id, $emap_ptr);
		    my $sinit = $emap_ptr->{$root}->{sub_init_time};
		    if (!defined $sinit) {
# warn "subinit inconsistency!\n";
			next;
		    }

# if i was the first subscription after the pub was sent,
# then i should associate it with myself.
#
		    my $firstpast = -1;
		    foreach my $s (@sub_inits) {
			if ($s > $pub_init) {
			    $firstpast = $s;
			    last;
			}
		    }

		    next if ($firstpast == -1);     # every sub was sent before the pub! accidental object discovery...;
		    next if ($firstpast != $sinit);  # if my init time != firstpast, i should let go;

		    $sub_idle = $emap_ptr->{$root}->{sub_idle_time};
		    $sub_idle = 0 if !defined $sub_idle;
		}
# no else {} -- for other types only one path goes back. so, it's ok.

# ok; its fine for this entry to be associated to the end-alias
# and, because of a bug in the UPDATE/REG logging code, just record 
# the first seen entry
#
		my $t = $trackmap_ptr->{$end_alias}->{$type};
		if ((!defined $t) or $time < $t) {
		    $trackmap_ptr->{$end_alias}->{$type} = $time;
		    if (defined $sub_idle) {
			$trackmap_ptr->{$end_alias}->{IDLE_TIME} = $sub_idle;
		    }
		}
	    }
	}
	close F;
    }
}

sub GetPubsubInits 
{
    my ($emap_ptr, $id) = @_;
    my $pub_init = undef;
    my @sub_inits = ();

# walk up the reverse alias maps from the end-alias
# shouldnt this be done for each end-alias??
# 

    my @rev_aliases;

    if (defined $emap_ptr->{$id}->{rev_aliases}) {
	@rev_aliases = @{$emap_ptr->{$id}->{rev_aliases}};
    }
    my @new_rev_aliases = ();

    for (my $i = 0; $i < 2; $i++) {
	foreach my $revid (@rev_aliases) { 
	    $pub_init = $emap_ptr->{$revid}->{pub_init_time};

	    if (defined $emap_ptr->{$revid}->{sub_init_time}) {
		push @sub_inits, $emap_ptr->{$revid}->{sub_init_time};
	    }

	    if (defined $emap_ptr->{$revid}->{rev_aliases}) {
		push @new_rev_aliases, @{$emap_ptr->{$revid}->{rev_aliases}};
	    }

	}

	@rev_aliases = @new_rev_aliases;
	@new_rev_aliases = ();
    }

    return ($pub_init, @sub_inits);
}

sub ProcessLogs
{
    my @files = @_;

    my %to_track  = ();

    start('SampleTrackingIDs');
    %to_track = SampleTrackingIDs(@files);
    stop('SampleTrackingIDs');
    print_timers ();
    exit (1);

    my %alias_map = ();
    my %extrainfo_map = ();

    ConstructAliasMap(\%to_track, \%alias_map, \%extrainfo_map, @files);
    CheckForCollisions(\%to_track, \%alias_map, \%extrainfo_map, @files);
    RecordInitTimes(\%to_track, \%alias_map, \%extrainfo_map, @files);
    MarkSubTravel(\%to_track, \%alias_map, \%extrainfo_map, @files);

    print STDERR "clean up pass (have " . (scalar keys %to_track) . " ids) atm...\n";

# third-pass: remove ones that aren't complete
    foreach my $k (keys %to_track) {
	foreach my $t (keys %MsgTypes) {
	    next if IgnoreType($t);

	    if (!defined $to_track{$k}->{$t}) {
		# print STDERR "missing type key=$k, type=$MsgTypes{$t}\n";
		delete $to_track{$k};
		last;
	    }
	}
    }
    print STDERR "finally have " . (scalar keys %to_track) . " ids...\n";

    return %to_track;
}

sub FindEndAlias
{
    my $id = shift;
    my $alias_map = shift;
    my @res;

    if (exists $alias_map->{$id}) {
	my @alias = @{$alias_map->{$id}};

	foreach my $alias (@alias) {
	    splice @res, 0, 0, FindEndAlias($alias, $alias_map);
	}

	return @res;
    } else {
	return ($id);
    }
}

sub ParseObjectLogsHeader() 
{
# XXX This is kinda hacky and could break...
    my $index = 0;

    open(HEADER, "$TOPDIR/../Merc/mercury/ObjectLogs.h") || 
	die "can't open $TOPDIR/../Merc/mercury/ObjectLogs.h";

    my $begun_class = 0;
    my $begun_enum  = 0;

    while (<HEADER>) {
	chomp $_;

	$_ =~ s|//.*||g;

	if ($_ =~ /struct\s+DiscoveryLatEntry/) {
	    $begun_class = 1;
	}
	if ($begun_class && $_ =~ /enum \{/) {
	    $begun_enum = 1;
	    $_ =~ s/enum \{//;
	    }

	    if ($begun_enum) {
		if ($_ =~ /((?:[\w_]+(?:,\s*)?)+)/) {
		    my $names = $1;
		    my @names = split(/,\s*/, $names);
		    map { $MsgTypes{$index} = $_; $index++; } @names;
		}

		if ($_ =~ /\}/) {
		    last;
		}
	}   
	}

	close(HEADER);
    }
