#!/usr/bin/perl
#
# $Id$
#
# -S time  - use this as the starttime (before skipping)
# -s num   - skip the first x seconds of the trace
# -l num   - the length of time to aggregate over (ignore remainder)
# -e epoch - size of each aggregation epoch (sec -- floating point)
# -w       - time slow down

use strict;
use Getopt::Std;
use Statistics::Descriptive;
use vars qw($opt_s $opt_l $opt_e $opt_S $opt_w); 

my $basedir;
chomp ($basedir = `dirname $0`);
require "$basedir/Common.pl";

our $PROTO_OVERHEAD_BYTES = 28;

getopts("S:s:l:e:w:");

our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 120;
our $LENGTH   = defined $opt_l ? $opt_l : 600;
our $EPOCH    = defined $opt_e ? $opt_e : 1;
our $SLOWDOWN = defined $opt_w ? $opt_w : 1;

###############################################################################
ProcessInput(@ARGV);
###############################################################################

sub ProcessInput(@)
{
    my @files = @_;

    my @buckets;

    my($start, $end);
    $start = $START;
    #if ($start) {
    #	print STDERR "start time: $start\n";
    #}

    my $stat = new Statistics::Descriptive::Full ();
    foreach my $f (@files) {
	if ($f =~ /.gz$/) { 
	    open F, "gunzip -c $f | " or die $!;
	}
	else {
	    open(F, "<$f") || die "can't open $f";
	
	}

	while (<F>) {
	    chomp $_;
	    m/(\d+\.\d+)\t([^\s]+)\t([^\s]+)/;
	    
	    my $time = $1;
	    my $guid = $2;
	    my $dest = $3;
	    
	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
                #print STDERR "Got to end of log! ( $start $end $time $SKIPTIME $LENGTH )\n";
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    my $index = int( ($time - ($start+$SKIPTIME))/$EPOCH );
	    my $inc = 1;

	    $inc /= $EPOCH;
	    $buckets[$index] += $inc;
	}

	my $fstat = new Statistics::Descriptive::Sparse ();	
	for (my $i=0; $i<int(($end-$start-$SKIPTIME)/$EPOCH); $i++) {
	    my $time = $start + $SKIPTIME + $i*$EPOCH;
	    $buckets[$i] = 0 if !defined $buckets[$i];
	    $buckets[$i] *= $SLOWDOWN;

	    $fstat->add_data ($buckets[$i]);
	}

	$stat->add_data ($fstat->mean ());
    }

    foreach my $f ( [ "MEAN",   sub { $_[0]->mean() } ],
	[ "STDDEV", sub { $_[0]->standard_deviation() } ],
	[ "MEDIAN", sub { $_[0]->median() } ],
	[ "95 %",   sub { $_[0]->percentile(95) } ] ) {
	print sprintf ("%12s\t%12.3f\n", $f->[0], &{$f->[1]}($stat));
    }
}
