#!/usr/bin/perl
# Usage: MigrationSimulations.pl [options] (directory)
# Options:
#
# -s time   number of second to skip from beginning of trace
# -l time   number of seconds of trace to examine
# -w        slowdown
#

use strict;
use IO::File;
use Getopt::Std;
use Carp;
use Statistics::Descriptive;
use vars qw($opt_S $opt_s $opt_l $opt_w);

getopts("S:s:l:w:");

our $DIR = shift @ARGV;
$DIR = "." if not defined $DIR or $DIR eq "";

my $START    = defined $opt_S ? $opt_S : undef;
my $SKIPTIME = defined $opt_s ? $opt_s : 60;
my $LENGTH   = defined $opt_l ? $opt_l : 300;
our $SLOWDOWN = defined $opt_w ? $opt_w : 1.0;

our %InfoTypes    =  ( 'i' => 'OBJINFO' );
our %CreateTypes  =  ( 'm' => 'MCREATE',
		       'c' => 'DCREATE',
		       'p' => 'PCREATE',
		       'f' => 'FCREATE', );
our %StoreTypes   =  ( 's' => 'STORE' );
our %DestroyTypes =  ( 'y' => 'MDESTROY',
		       'd' => 'TDESTROY',
		       'e' => 'EDESTROY' );

my $FRAME_TIME = 0.1; # sec
if ($SLOWDOWN > 1.0) { 
    $SKIPTIME *= $SLOWDOWN;
    $LENGTH *= $SLOWDOWN;
    $FRAME_TIME *= $SLOWDOWN;
}
our ($WIDTH, $HEIGHT) = GetMapDims ();
our $NSERVERS = GetNServers ();
our $RECT_X_BLOCKSIZE = $WIDTH / $NSERVERS;

my %GSMAP = ();
my @SERVER_MIGRATIONS = ();
foreach my $i (0 .. ($NSERVERS - 1)) {
    $SERVER_MIGRATIONS [$i] = [];
}

my($start, $end);
$start = $START;

# this log records what areas each object is interested in
foreach my $file (glob "$DIR/ObjectInterestLog.*") {
    my $fh = OpenFile ($file);
    $file =~ m/$DIR\/ObjectInterestLog.(.*).log/;
    my $line;
    
    print STDERR "processing $file...\n";
    while (defined ($line = <$fh>)) {
	chomp $line;
	if ($line =~ /^(\d+\.\d+)\t\d+\t([\d\.\:]+)\t(\w)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)/) {
	    my $time = $1;
	    my $guid = $2;
	    my $type = $3;
	    my @pos  = ($4,$5,$6);

	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }
	    my $new_server = getServerForGuid ($guid, \@pos);
	    
	    if (exists $GSMAP{$guid} && $GSMAP{$guid} ne $new_server) {
		my $aref = $SERVER_MIGRATIONS [$GSMAP {$guid}];
		push @$aref, $time;
	    }
	    $GSMAP {$guid} = $new_server;
	}
    }    
}

my $stat = new Statistics::Descriptive::Full ();
foreach my $i (0 .. ($NSERVERS - 1)) {
    my @migtimes = sort { $a <=> $b } @{$SERVER_MIGRATIONS [$i]};
    if (scalar @migtimes > 0) {
	my $diff = ($migtimes [-1] - $migtimes [0]) / $SLOWDOWN;
	next if ($diff == 0);
	$stat->add_data ((scalar @migtimes) / $diff);
    }
}

foreach my $f ( [ "MEAN",   sub { $_[0]->mean() } ],
	[ "STDDEV", sub { $_[0]->standard_deviation() } ],
	[ "MEDIAN", sub { $_[0]->median() } ],
	[ "95 %",   sub { $_[0]->percentile(95) } ] ) {
    print sprintf ("%12s\t%12.3f\n", $f->[0], &{$f->[1]}($stat));
}

sub getServerForGuid ($$) {
    my $guid = shift;
    my $ppos = shift;
    
    # just divide it on based on the x-range. this will create rectangles with 
    # aspect ration x:y = 1:2, approximately. but this will work ONLY FOR RECT MAPS.
    return int ($ppos->[0] / $RECT_X_BLOCKSIZE);
}

sub OpenFile ($) { 
    my $file = shift;
    my $fh;
    if ($file =~ /.gz$/) {
	$fh = new IO::File ("gunzip -c $file | ");
    }
    else {
	$fh = new IO::File("<$file");
    }
    return $fh;
}
sub GetMapDims () { 
    my $file = (glob "$DIR/OutputLog.*") [-1];
    my $map  = `grep -- --map $file`;
    chomp $map;
    $map =~ /([\w]+_maps\/\d+-players)/;
    $map = $1;

    my $blurb = `cat $ENV{HOME}/Data/$map/dims`;
    chomp $blurb;
    $blurb =~ /(\d+) (\d+)/;
    return ($1, $2);
}

sub GetNServers () {
    my @v = glob ("$DIR/OutputLog.*");
    return scalar (@v) - 1;
}

