#!/usr/bin/perl
#
# Generate aggregate bandwidth stats for a MessageLog
#
# -S time  - use this as the starttime (before skipping)
# -s num   - skip the first x seconds of the trace
# -l num   - the length of time to aggregate over (ignore remainder)
# -e epoch - size of each aggregation epoch (sec -- floating point)
# -d dir   - {in, out}
# -t type  - {bwidth, proto, pps}
#            bandwidth without protocol overhead,
#            bandwidth with protocol overhead,
#            packets per second
# -m       - only mercury packets
# -o       - only non-mercury (object manager) packets
# -r       - only routing traffic
# -c       - only matching traffic
# -M type  - only messages of this type
# -a       - absolute count (not rate per second)
# -b       - the log is binary
# -w       - time slow down (scales the bwidth costs by X)

use strict;
use Getopt::Std;
use vars qw($opt_s $opt_l $opt_e $opt_t $opt_d $opt_m $opt_o $opt_r $opt_c $opt_a $opt_M $opt_b $opt_S $opt_w); 

my $basedir;
chomp ($basedir = `dirname $0`);
require "$basedir/Common.pl";

our $PROTO_OVERHEAD_BYTES = 28;

getopts("S:s:l:e:d:t:moaM:brcw:");

our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 120;
our $LENGTH   = defined $opt_l ? $opt_l : 600;
our $EPOCH    = defined $opt_e ? $opt_e : 1;
our $DIR      = defined $opt_d ? $opt_d : "out";
our $TYPE     = defined $opt_t ? $opt_t : "bwidth";
our $SLOWDOWN = defined $opt_w ? $opt_w : 1;

our $ABSOLUTE  = defined $opt_a; 
our $ONLY_MERC = defined $opt_m;
our $ONLY_OM   = defined $opt_o;
our $ONLY_ROUTING = defined $opt_r;
our $ONLY_MATCHING = defined $opt_c;
our $MSGTYPE   = $opt_M;
our $LOG_BINARY = defined $opt_b;

###############################################################################

our %MsgTypes = ParseMessageHeader($0);

our $USE_FILTER = 0;
our %FILTER_TYPES;
if ($MSGTYPE) {
    $USE_FILTER = 1;
    map { $FILTER_TYPES{$_} = 1 } split(/,/, $MSGTYPE);
} elsif ($ONLY_ROUTING) {
    $USE_FILTER = 1;
    foreach my $type (keys %MsgTypes) {
	if (IsMercMsg($type)) {
	    if ($MsgTypes{$type} !~ /MATCH/) {
		$FILTER_TYPES{$MsgTypes{$type}} = 1;
	    }
	}
    }
} elsif ($ONLY_MATCHING) {
    $USE_FILTER = 1;
    foreach my $type (keys %MsgTypes) {
	if (IsMercMsg($type)) {
	    if ($MsgTypes{$type} =~ /MATCH/) {
		$FILTER_TYPES{$MsgTypes{$type}} = 1;
	    }
	}
    }
}

ProcessInput(@ARGV);

###############################################################################

sub ProcessInput(@)
{
    my @files = @_;

    my @buckets;

    my($start, $end);
    $start = $START;
    #if ($start) {
    #	print STDERR "start time: $start\n";
    #}

    foreach my $f (@files) {
	if ($LOG_BINARY) {
	    open F, "$basedir/ParseMessageLog $f |" or die "can't open pipe from file $f";
	}
	else {
	    open(F, "<$f") || die "can't open $f";
	}

	while (<F>) {
	    chomp $_;
	    m/([0-9A-F]{8}?)\t(\d+\.\d+)\t(\d+)\t(\d+)\t(\d+)\t(\d+)/;

	    my $nonce = $1;
	    my $time = $2;
	    my $dir = $3;
	    my $proto = $4;
	    my $type = $5;
	    my $size = $6;
	    
#	    my ($nonce, $time, $dir, $proto, $type, $size) = split(/\t/, $_);
	    
	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
                #print STDERR "Got to end of log! ( $start $end $time $SKIPTIME $LENGTH )\n";
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    if (IsMercMsg ($type)) {
		next if ($ONLY_OM);
	    }
	    else {
		next if ($ONLY_MERC);
	    }
	    if ($USE_FILTER && !($FILTER_TYPES{$MsgTypes{$type}})) {
		next;
	    }
	    
	    $dir = $dir == 0 ? "in" : "out";

	    if ($DIR ne $dir) {
		next;
	    }

	    my $index = int( ($time - ($start+$SKIPTIME))/$EPOCH );
	    my $inc;

	    if ($TYPE eq 'bwidth') {                
		$inc = $size*8.0/1000;
	    } elsif ($TYPE eq 'proto') {
		$inc = ($size+$PROTO_OVERHEAD_BYTES)*8.0/1000;
	    } elsif ($TYPE eq 'pps') {
		$inc = 1;
	    } else {
		die "unknown type: $TYPE!";
	    }
	    if (!$ABSOLUTE) {
		$inc /= $EPOCH;
	    }


	    $buckets[$index] += $inc;
	}
    }

    for (my $i=0; $i<int(($end-$start-$SKIPTIME)/$EPOCH); $i++) {
	my $time = $start + $SKIPTIME + $i*$EPOCH;
	$buckets[$i] = 0 if !defined $buckets[$i];
	$buckets[$i] *= $SLOWDOWN;
	print "$time\t$buckets[$i]\n";
    }
}
