#!/usr/bin/perl
#
# Generate aggregate bandwidth stats for a MessageLog
#
# -S time  - use this as the starttime (before skipping)
# -s num - skip the first x seconds of the trace
# -l num - the length of time to aggregate over (ignore remainder)
# -b     - the log is binary

use strict;
use Getopt::Std;
use vars qw($opt_S $opt_s $opt_l $opt_b);

my $basedir;
chomp ($basedir = `dirname $0`);
require "$basedir/Common.pl";

our $PROTO_OVERHEAD_BYTES = 28;

getopts("S:s:l:b");

our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 0;
our $LENGTH   = defined $opt_l ? $opt_l : undef;
our $LOG_BINARY = defined $opt_b;

###############################################################################

our %MsgTypes = ParseMessageHeader($0);
#foreach my $k (sort {$a <=> $b } keys %MsgTypes) {
#    print "$k\t$MsgTypes{$k}\n";
#}
#exit 1;
our %stats = ProcessInput(@ARGV);

print sprintf("%-28s %15s   %15s   %15s\n",
	      "Type", "Cost (Kbps)", "Count (pps)", "Cost w/ Proto");
foreach my $s (sort { $a cmp $b } keys %stats) {
    print sprintf("%-28s %15.3f   %15.3f   %15.3f\n", 
		  $s, $stats{$s}->[0], $stats{$s}->[1], $stats{$s}->[2]);
}

###############################################################################

sub ProcessInput(@)
{
    my @files = @_;

    my($start, $end);
    $start = $START;
    my %stats;

    # [ size, count, with_overhead ]
    $stats{"IN"}  = [0,0];
    $stats{"OUT"} = [0,0];
    $stats{"IN:TCP"} = [0,0];
    $stats{"IN:UDP"} = [0,0];
    $stats{"OUT:TCP"} = [0,0];
    $stats{"OUT:UDP"} = [0,0];

    foreach my $f (@files) {
	if ($LOG_BINARY) {
	    open F, "$basedir/ParseMessageLog $f |" or die "can't open pipe from file $f";
	}
	else {
	    open(F, "<$f") || die "can't open $f";
	}

	while (<F>) {
#	    print;
	    chomp $_;
	    m/([0-9A-F]{8}?)\t(\d+\.\d+)\t(\d+)\t(\d+)\t(\d+)\t(\d+)/;

	    my $nonce = $1;
	    my $time = $2;
	    my $dir = $3;
	    my $proto = $4;
	    my $type = $5;
	    my $size = $6;
	    
#my ($nonce, $time, $dir, $proto, $type, $size) = split(/\t/, $_);
# print "nonce=$nonce time=$time dir=$dir proto=$proto type=$MsgTypes{$type} size=$size\n";
	    
	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) { print STDERR "before\n";
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) { print STDERR "done\n";
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }
	    
	    $dir   = $dir   == 0 ? "IN" : "OUT";
	    $proto = $proto == 0 ? "TCP" : "UDP";
	    $type  = $MsgTypes{$type} || $type;
	    
	    $stats{"$dir:$type"} = [0,0] if ! defined $stats{"$dir:$type"};
	    
	    $stats{"$dir:$type"}->[0] += $size;
	    $stats{"$dir:$type"}->[1]++;
	    
	    $stats{$dir}->[0] += $size;
	    $stats{$dir}->[1]++;
	    
	    $stats{"$dir:$proto"}->[0] += $size;
	    $stats{"$dir:$proto"}->[1]++;
	}
    }

    my $interval = $end - $start;
    foreach my $s (keys %stats) {
	$stats{$s}->[0] = $stats{$s}->[0]*8/(1000*$interval);
	$stats{$s}->[1] = $stats{$s}->[1]/$interval;
	# suppose we including IP/UDP overhead
	$stats{$s}->[2] = 
	    $stats{$s}->[0] + $stats{$s}->[1]*$PROTO_OVERHEAD_BYTES*8/1000;
    }

    return %stats;
}
