#!/usr/bin/perl
#
# Get the maximum start time of a series of logs
#
# -f num field in the log (tab delimited) of time (default: 1)
# -m     logs are binary message logs
# -d     logs are binary discovery lat logs
# -M     min instead of max

use strict;
use Cwd;
use IO::File;
use Getopt::Std;
use vars qw($opt_m $opt_d $opt_f $opt_M);

getopts("mdf:M");

my $BINARY_MSGLOG = defined $opt_m;
my $BINARY_DISLOG = defined $opt_d;
my $TIME_FIELD    = defined $opt_f ? $opt_f-1 : 0;

my @logs  = OpenFiles(@ARGV);
my @times = GetHeadTimes(@logs);
my $max   = max( @times );
my $min   = min( @times );
my $diff  = sprintf "%.3f", $max - $min;

if ($opt_M) {
	print "$min\n";
} else {
	print "$max\n";
}
#print "$min\t$max\t$diff\n";

exit 0;

sub OpenFiles {
    my @logs = @_;
    my @ret;

    foreach my $file (@logs) {
	my $fh;

	if ($BINARY_MSGLOG) {
	    $fh = new IO::File("~/Colyseus/scripts/ParseMessageLog $file |");
	} elsif ($BINARY_DISLOG) {
	    $fh = new IO::File("~/Colyseus/scripts/ParseDiscoveryLatLog $file |");
	} else {
	    if ($file =~ /.gz$/) { 
		$fh = new IO::File ("gunzip -c $file | ");
	    }
	    else {
		$fh = new IO::File("<$file");
	    }
	}
	
	die "can't open $file: curwd=" . (cwd) . "$!" if !$fh;
	push @ret, $fh;
    }

    return @ret;
}

sub GetHeadTimes {
    my @logs = @_;
    my @ret;

    foreach my $log (@logs) {
	my $line = <$log>;
	if (!$line) {
	    warn "no lines in log! $log";
	    next;
	}
	chomp $line;
	my @fields = split(/\t/, $line);
	if ($fields[$TIME_FIELD] =~ /(\d+\.\d+)/) {
	    push @ret, $1;
	} else {
	    die "bad format: $line";
	}
    }

    return @ret;
}

sub min {
    my @vals = @_;
    my $min = shift @vals;

    foreach my $v (@vals) {
	if ($min > $v) {
	    $min = $v;
	}
    }

    return $min;
}

sub max {
    my @vals = @_;
    my $max = shift @vals;

    foreach my $v (@vals) {
	if ($max < $v) {
	    $max = $v;
	}
    }

    return $max;
}
