#!/usr/bin/perl
#
# Converts lists of GUIDs into their classtypes (p, m, i, o)
#
# Usage: MapGUIDListToType.pl [options] @files
#
# -f field(s) to replace (separated by commas, starting from 1)
# -d delimiter
# -g directory containing GUIDLogs.

use strict;
use Getopt::Std;
use vars qw($opt_f $opt_d $opt_g);

getopts("f:g:d:");

our @FIELDS = defined $opt_f ? map { $_ = $_ - 1 } split(/,/, $opt_f) : (0);
our $DELIM  = defined $opt_d ? $opt_d : "\t";
our $DIR    = defined $opt_g ? $opt_g : ".";

our %PLAYER_TYPES = ( "player" => 1, "bot" => 1 );
our %MISSILE_TYPES = ( "missile" => 1, "bolt" => 1, "grenade" => 1, "rocket" => 1 );
our %ITEM_TYPES = ( "item_health" => 1, "item_health_large" => 1, 
		    "weapon_chaingun" => 1, "weapon_grenadelauncher" => 1,
		    "weapon_hyperblaster" => 1, "weapon_machinegun" => 1,
		    "weapon_railgun" => 1, "weapon_rocketlauncher" => 1,
		    "weapon_shotgun" => 1, "weapon_supershotgun" => 1 );

###############################################################################

our %GUID_MAP;

my @guidlogs = glob("$DIR/GUIDLog.*");
die "no guid logs" if @guidlogs == 0;

print STDERR "processing guid logs: ";

foreach my $log (@guidlogs) {
    if ($log =~ /\.gz$/) {
	open(L, "zcat $log |") || die "can't pipe $log: $!";
    } else {
    	open(L, "<$log") || die "can't open $log: $!";
    }
    while (<L>) {
	if ($_ =~ /\d+\.\d+\t([^\t]+)\t([^\t]+)\t(\w)/) {
	    my ($guid, $t, $act) = ($1, $2, $3);
	    my $stype;

	    if ($PLAYER_TYPES{$t}) {
		$stype = "p";
	    } elsif ($MISSILE_TYPES{$t}) {
		$stype = "m";
	    } elsif ($ITEM_TYPES{$t}) {
		$stype = "i"
	    } else {
		$stype = "o";
	    }

	    $GUID_MAP{$guid} = $stype;
	}
    }
    close(L);
    print STDERR ".";
}

print STDERR "\n";

###############################################################################

while (<>) {
    chomp $_;
    my @fields = split(/$DELIM/, $_);

    foreach my $i (@FIELDS) {
	my @guids = split(/,/, $fields[$i]);

	my ($players, $missiles, $items, $others, $unknown) =
	    (0,0,0,0,0);

	foreach my $g (@guids) {
	    my $t = $GUID_MAP{$g};
	    if (!defined $t) {
		$unknown++;
	    } elsif ($t eq 'p') {
		$players++;
	    } elsif ($t eq 'm') {
		$missiles++;
	    } elsif ($t eq 'i') {
		$items++;
	    } else {
		$others++;
	    }
	}

	$fields[$i] = "$players,$missiles,$items,$others,$unknown";
    }

    print (join("$DELIM", @fields) . "\n");
}
