#!/usr/bin/perl
use strict;
use Getopt::Std;
use Statistics::Descriptive;
use vars qw($opt_s $opt_S $opt_l $opt_w);

our $basedir;
chomp ($basedir = `dirname $0`);
require "$basedir/Common.pl";

getopts ("s:S:l:w:");

our $TOPDIR = "$basedir/../";
our %MsgTypes;

getopts("aS:s:l:bT:Aw:n");

our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 0;
our $LENGTH   = defined $opt_l ? $opt_l : undef;
our $LOG_BINARY = 0;
our $AGGLOGS = 0; # defined $opt_A;
$LOG_BINARY = 0 if ($AGGLOGS);

our $SLOWDOWN = defined $opt_w ? $opt_w : 1;

if ($SLOWDOWN > 1.0) { 
    $SKIPTIME *= $SLOWDOWN;
    $LENGTH *= $SLOWDOWN;
}
###############################################################################

ParseObjectLogsHeader();

our $SUB_STORE_INDEX;
our $PUB_STORE_INDEX;
our $SUB_ROUTE_RECV_INDEX;

foreach my $k (keys %MsgTypes) 
{
    $PUB_STORE_INDEX = $k if ($MsgTypes{$k} eq "PUB_STORE");
    $SUB_STORE_INDEX = $k if ($MsgTypes{$k} eq "SUB_STORE");
    $SUB_ROUTE_RECV_INDEX = $k if ($MsgTypes{$k} eq "SUB_ROUTE_RECV");
}

ProcessLogs(@ARGV);

sub show_progress { 
    my ($done, $total) = @_;
    print STDERR sprintf ("%.2f %% done \r", ($done * 100.0 / $total));
}

sub ProcessLogs(@)
{
    my @files = @_;
    my %ss_tracker = ();

    my $nf = 0;

    print STDERR "Computing pubsub init times and routing delays...\n";
    foreach my $f (@files) {
	if ($LOG_BINARY) {
	    open F, "$basedir/ParseDiscoveryLatLog $f | " or die "cant open pipe from file $f ($!)";
	}
	else {
	    if ($f =~ /.gz$/) { 
		open F, "gunzip -c $f | " or die " cant gunzip on $f: $!";
	    }
	    else {
		open(F, "<$f") || die "can't open $f ($!)";
	    }
	}
	my($start, $end);
	$start = $START;

	show_progress ($nf++, scalar @files);

	while (<F>) {
	    chomp $_;
	    m/(\d+\.\d+)\t(\d+)\t(\d+)\t([0-9A-F]{8}?)(.*)$/;

	    my $time = $1;
	    my $type = $2;
	    my $hops = $3;
	    my $id   = $4;

	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    if ($type == $SUB_STORE_INDEX) { 
		$ss_tracker{$id}++;
		print STDOUT "s\t$hops\n";   # useful for DHT only due to a logging bug in the code (LINEAR_SUBs are not logged in DiscoveryLogs);
	    }
	    elsif ($type == $PUB_STORE_INDEX) {
		print STDOUT "p\t$hops\n";   # useful for DHT and mercury;
	    }
	    elsif ($type == $SUB_ROUTE_RECV_INDEX) { 
		print STDOUT "r\t$hops\n";   # useful for mercury;
	    }	    
	}
	close F;
    }
    print STDOUT "=============================================\n";
    foreach my $id (keys %ss_tracker) {
	print STDOUT "h\t$ss_tracker{$id}\t$id\n";
    }
}


############################################################################

sub ParseObjectLogsHeader() 
{
# XXX This is kinda hacky and could break...
    my $index = 0;

    open(HEADER, "$TOPDIR/../Merc/mercury/ObjectLogs.h") || 
	die "can't open $TOPDIR/../Merc/mercury/ObjectLogs.h";

    my $begun_class = 0;
    my $begun_enum  = 0;

    while (<HEADER>) {
	chomp $_;

	$_ =~ s|//.*||g;

	if ($_ =~ /struct\s+DiscoveryLatEntry/) {
	    $begun_class = 1;
	}
	if ($begun_class && $_ =~ /enum \{/) {
	    $begun_enum = 1;
	    $_ =~ s/enum \{//;
        }

	if ($begun_enum) {
	    if ($_ =~ /((?:[\w_]+(?:,\s*)?)+)/) {
		my $names = $1;
		my @names = split(/,\s*/, $names);
		map { $MsgTypes{$index} = $_; $index++; } @names;
	    }
	    
	    if ($_ =~ /\}/) {
		last;
	    }
	}   
    }

    close(HEADER);
}
