////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <map>
#include <hash_map.h>
#include <vector>
#include <string>
#include <list>
#include <util/debug.h>
#include <stdlib.h>
#include <util/Options.h>
#include <gameapi/common.h>
#include <util/GPL/callback.h>
#include <util/Benchmark.h>
#include <mercury/ObjectLogs.h>

#include "putils.cxx"

double START;
float SLOWDOWN;
int SKIPTIME;
int LENGTH;

// XXX hopcounts are currently bogus for range subs, anyways.
//
typedef unsigned long psid_t;

struct hash_psid_t {
    size_t operator () (const psid_t& a) const {
	return hash<psid_t> () (a);
    }
};

struct eq_psid_t { 
    bool operator () (const psid_t& a, const psid_t& b) const {
	return a == b;
    }
};	

typedef hash_map<psid_t, int, hash_psid_t, eq_psid_t> IDintMap;
typedef IDintMap::iterator IDintMapIter;

typedef vector<psid_t> VS;
typedef VS::iterator VSIter;
typedef VS *PVS;

typedef hash_map<psid_t, PVS, hash_psid_t, eq_psid_t> IDPVSMap;
typedef IDPVSMap::iterator IDPVSMapIter;

OptionType options[] = {
    { 'S', "start", OPT_DBL, "", &START, "-1.0", NULL },
    { 'l', "length", OPT_INT, "", &LENGTH, "600", NULL  },
    { 's', "skiptime", OPT_INT, "", &SKIPTIME, "120", NULL  },
    { 'w', "slowdown", OPT_FLT, "", &SLOWDOWN, "0.0", NULL },
    { 0, 0, 0, 0, 0, 0, 0 },
};

vector<string> DisLatLogs;         // list of all logs
IDintMap       TrackingIDs;        // the sampled IDs we want to track
IDPVSMap       AliasMap;           // all alias mappings
IDintMap       PassPermitted;      // temporary hashtable for pruning alias map
IDintMap       AllowedIDs;         // IDs to filter from the log, finally

void collect_hops () {
    cerr << "looking for SUB_STORE and PUB_STORE [" <<
	DisLatLogs.size () << " logs]" << endl;
    int nf = 0;

    for (vector<string>::iterator it = DisLatLogs.begin (); it != DisLatLogs.end (); ++it) {
	FILE *fp = open_log (*it);
	if (!fp) 
	    Debug::die ("could not open log %s", it->c_str ()); 

	show_progress (nf++, DisLatLogs.size ());

	double start = START, end = 0, tim;
	int type;
	int hops;
	psid_t id;
	psid_t alias;

	START (FILETIME);
	memset (line, 0, sizeof (line));
	while (fgets (line, sizeof (line), fp)) {
	    if (!match_capture (
				"(\\d+\\.\\d+)\\t(\\d+)\\t(\\d+)\\t([0-9A-F]{8}?).*",
				(char *) line,
				4, /* npats */
				(char **) patbufs)) {
		Debug::warn ("bad log line: %s", line);
		continue;
	    }

	    sscanf (patbufs[0], "%lf", &tim);
	    type = atoi (patbufs[1]);
	    hops = atoi (patbufs[2]);
	    id = (psid_t) strtoll (patbufs[3], NULL, 16);

	    if (start < 0) 
		start = tim;
	    if (tim < start + SKIPTIME) {
		continue;
	    }
	    if (LENGTH > 0 && tim > start + SKIPTIME + LENGTH) { 
		// fprintf (stderr, "%d %.3f %.3f %.3f %d\n", LENGTH, tim, start, START, SKIPTIME);
		break;
	    }
	    if (tim > end)
		end = tim;

	    if (type == DiscoveryLatEntry::SUB_STORE) 
		fprintf (stdout, "s\t%d\n", hops);
	    else if (type == DiscoveryLatEntry::PUB_STORE) 
		fprintf (stdout, "p\t%d\n", hops);
	}
	close_log (fp, *it);
	STOP (FILETIME);
    }
}

int main (int argc, char *argv[])
{
    DBG_INIT (NULL);
    InitCPUMHz();
    Benchmark::init();

    int nargc = ProcessOptions (options, argc, argv, false);
    if (SLOWDOWN > 1.0) {
	SKIPTIME = (int) (SKIPTIME * SLOWDOWN);
	LENGTH = (int) (LENGTH * SLOWDOWN);
    }

    for (int i = 1; i < nargc; i++) {
	DisLatLogs.push_back (argv[i]);
    }

    for (int i = 0; i < NPATBUFS; i++)
	patbufs[i] = new char[1024];

    TIME (collect_hops ());
    Benchmark::print ();
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
