#!/usr/bin/perl
#
# Look at frame intervals for our exps.
#
# $Id: FrameRates.pl 2495 2005-11-22 23:46:04Z jeffpang $
# usage: FrameRates.pl (in directory containing QuakeStatsLog.*s)
#

use strict;
use Statistics::Descriptive;
use Getopt::Std;
use vars qw($opt_S $opt_s $opt_l $opt_w $opt_f);

getopts("S:s:l:w:f");

my $skiptime  = defined $opt_s ? $opt_s : 120;
my $len       = defined $opt_l ? $opt_l : 8*60;
my $starttime = undef;
our $STARTTIME = defined $opt_S ? $opt_S : undef;
our $SLOWDOWN = defined $opt_w ? $opt_w : 1.0;
if ($SLOWDOWN > 1.0) { 
    $skiptime *= $SLOWDOWN;
    $len *= $SLOWDOWN;
}
our $IPREGEX = $opt_f ? "(\\d+\\.\\d+\\.\\d+\\.\\d+)\\:" : "(\\d+)\\:(\\d+)";

my @astats = qw(MEDIAN MEAN STDDEV NINEFIVE NINEZERO);

my %astats;
for my $i (@astats) {
    $astats{$i} = new Statistics::Descriptive::Full();
}

sub sort_by_emulab_ip { 
    $a =~ /$IPREGEX/;
    my ($a_num, $a_port) = ($1, $2);
    $b =~ /$IPREGEX/;
    my ($b_num, $b_port) = ($1, $2);
    return ($a_num <=> $b_num) || ($a_port <=> $b_port);
}

if ($opt_f) {
    print STDERR sprintf "%15s\tMEDIAN\tMEAN\tSTDDEV\t99%\t95%\n", "NODE";
} else {
    print STDERR "NODE\tMEDIAN\tMEAN\tSTDDEV\t99%\t95%\n";
}
foreach my $file (sort sort_by_emulab_ip glob("QuakeStatsLog.*")) {
    $file =~ /$IPREGEX/;
    my ($num, $port) = ($1, $2);
    $num-- if !$opt_f; # to node num
    $port = $port % 100; # to vserver index
    if ($opt_f) {
	print STDERR sprintf "%15s\t", $num;
    } else {
	print STDERR "$num:$port\t";
    }
    if ($file =~ /.gz$/) { 
	open P, "gunzip -c $file | " or die "cant gunzip on $file: $!";
    }
    else {
	open(P, "$file") or die "Can't open $file";
    }

    my $stats = new Statistics::Descriptive::Full();

    $starttime = $STARTTIME;

    my $anythingthere = 0;
    while (<P>) {
	$anythingthere = 1;
	chomp $_;
	$_ =~ /^(\d+\.\d+)\t(\d+\.\d+)/;
	my ($time, $fr) = ($1,$2);
	if (!defined $starttime) {
	    $starttime = $time;
	}
	if ($time < $starttime+$skiptime) {
	    next;
	}
	if ($time > $starttime+$skiptime+$len) {
	    last;
	}
	
	$stats->add_data( ($fr/$SLOWDOWN) );
    }

    if (!$anythingthere || $stats->count() == 0) { 
	print STDERR "NO DATA\n";
	next;
    }
    p($stats->median()); 
    $astats{MEDIAN}->add_data( $stats->median() );
    p($stats->mean()); 
    $astats{MEAN}->add_data( $stats->mean() );
    p($stats->standard_deviation()); 
    $astats{STDDEV}->add_data( $stats->standard_deviation() );
    p($stats->percentile(99)); 
    $astats{NINEFIVE}->add_data( scalar $stats->percentile(99) );
    p($stats->percentile(95)); 
    $astats{NINEZERO}->add_data( scalar $stats->percentile(95) );
    print STDERR "\n";
}

print STDERR "\n";

# print "MEDIAN\t";
# for my $i (@astats) {
#    p($astats{$i}->median());
# }
# print "\n";
print "MEAN\t";
for my $i (@astats) {
    print sprintf("%.1f\t", ($astats{$i}->mean()));
}
print "\n";

sub p {
    print STDERR sprintf("%.1f\t", shift());
}
