#!/usr/bin/perl
#
# Go through all the discovery logs and track an object discovery ID
# This is more of a debugging script to figure out how the 
# particular components for the ID interacted 
# 
# $Id: Follow.pl 2734 2006-03-21 01:23:08Z ashu $
# 
# Usage: Follow.pl DiscoveryLog.*
#
# -i  - ID to track (needed)
# -a  - show all (instead of mean/stddev/95%)
# -T  - track these many complete messages at MAX
# -s  - skip time
# -l  - length of time
# -b  - binary logs
#
#

use strict;
use Statistics::Descriptive;
use Getopt::Std;
use vars qw($opt_a $opt_i $opt_s $opt_b $opt_l $opt_T);

my $basedir;
chomp ($basedir = `dirname $0`);
require "$basedir/Common.pl";

our $TOPDIR = "$basedir/../";
our %MsgTypes;

getopts("ai:s:l:bT:");

our $ID = $opt_i;
our $SHOWALL  = defined $opt_a;
our $SKIPTIME = defined $opt_s ? $opt_s : 0;
our $LENGTH   = defined $opt_l ? $opt_l : undef;
our $MAXSAMPLE = $opt_T || 50;
die "ID must be given" unless defined $ID;
our $LOG_BINARY = defined $opt_b;

###############################################################################

ParseObjectLogsHeader($0);

my %tracked = ProcessLogs(@ARGV);

sub IgnoreType 
{
    my $type = shift;
    return $MsgTypes{$type} eq 'ALIAS' ||
	$MsgTypes{$type} eq 'PUB_INIT' ||
	$MsgTypes{$type} eq 'PUB_SEND' ||
	$MsgTypes{$type} eq 'PUB_STORE' ||
	$MsgTypes{$type} eq 'PUB_ROUTE_RECV' ||
	$MsgTypes{$type} eq 'PUB_ROUTE_SEND' ||
	$MsgTypes{$type} eq 'SUB_ROUTE_RECV' ||
	$MsgTypes{$type} eq 'SUB_ROUTE_SEND';
}

sub ConstructAliasMap
{
    my $trackmap_ptr = shift;
    my $amap_ptr = shift;
    my $emap_ptr = shift;
    my @files = @_;

    my %pass_permitted = ();

# in the first pass, only permit the IDs we want to track to be 
# reverse-aliased. these are the leaves of the tree.
#
    foreach my $id (keys %$trackmap_ptr) {
	$pass_permitted{$id} = 0;
    }

#   aliases can get created only 2 times; first time when/if the pub is sent out 
#   to multiple hubs, and the second time at the rendezvous node.
#   
#   so finding the reverse maps starting from the trackids will 
#   need at max 2 passes 

    print STDERR "Constructing alias maps...\n";
    for (my $pass = 0; $pass < 2; $pass++) 
    {
	print STDERR "\tPass $pass\n";

	foreach my $f (@files) {
	    if ($LOG_BINARY) {
		open F, "$basedir/ParseDiscoveryLatLog $f | " or die "cant open pipe from file $f ($!)";
	    }
	    else {
		if ($f =~ /.gz$/) { 
		    open F, "gunzip -c $f | " or die " cant gunzip on $f: $!";
		}
		else {
		    open(F, "<$f") || die "can't open $f ($!)";
		}
	    }
	    my($start, $end);
	    while (<F>) {
		chomp $_;
		m/(\d+\.\d+)\t(\d+)\t(\d+)\t([0-9A-F]{8}?)(.*)$/;

		my $time = $1;
		my $type = $2;
		my $hops = $3;
		my $id   = $4;
		my $alias= $5; $alias =~ s/\t//;

#		my ($time, $type, $hops, $id, $alias) = split(/\t/, $_);

		if (!defined $start) {
		    $start = $time;
		}
		if ( $time < $start + $SKIPTIME ) {
		    next;
		}
		if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		    last;
		}
		if ($time > $end) {
		    $end = $time;
		}

		if (!defined $MsgTypes{$type}) {
		    warn "Unknown type: $type; processing file $f";
		    next;
		}
		$type = $MsgTypes{$type};
		next if ($type ne 'ALIAS');		
		next if ((!exists $pass_permitted{$alias}) || $pass_permitted{$alias} != $pass);

		print "ALIAS\t$id\t$alias\n";
		$amap_ptr->{$id} = [] if !defined $amap_ptr->{$id};
		push @{$amap_ptr->{$id}}, $alias;
		if (!defined $pass_permitted{$id}) {
		    $pass_permitted{$id} = $pass + 1;              # the next pass should record info for this guy;
		}

		$emap_ptr->{$alias}->{rev_aliases} = [] if (!defined $emap_ptr->{$alias}->{rev_aliases});
		push @{$emap_ptr->{$alias}->{rev_aliases}}, $id;

	    }
	    close F;

	}
    }
}

#
# Walk the reverse alias map UPwards to get to the SUB_INIT or SUB_SEND 
# node. As long as the walk starts from somewhere BEFORE the matching happens, 
# it is guaranteed to have exactly one "start" point.
# 
sub GetTreeRoot {
    my $rid = shift;
    my $emap_ptr = shift;
    while (defined $emap_ptr->{$rid} && defined $emap_ptr->{$rid}->{'rev_aliases'}) {
	$rid = $emap_ptr->{$rid}->{'rev_aliases'}->[0];
    }

    return $rid;
}

sub MarkSubTravel 
{
    my $trackmap_ptr = shift;
    my $amap_ptr = shift;
    my $emap_ptr = shift;
    my @files = @_;

    print STDERR "tracking the desired ids now...\n";
    foreach my $f (@files) {
	if ($LOG_BINARY) {
	    open F, "$basedir/ParseDiscoveryLatLog $f | " or die "cant open pipe from file $f ($!)";
	}
	else {
	    if ($f =~ /.gz$/) { 
		open F, "gunzip -c $f | " or die " cant gunzip on $f: $!";
	    }
	    else {
		open(F, "<$f") || die "can't open $f ($!)";
	    }
	}
	my($start, $end);
	while (<F>) {
	    chomp $_;
	    m/(\d+\.\d+)\t(\d+)\t(\d+)\t([0-9A-F]{8}?)(.*)$/;

	    my $time = $1;
	    my $type = $2;
	    my $hops = $3;
	    my $id   = $4;
	    my $alias= $5; $alias =~ s/\t//;

	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }

	    next if $MsgTypes{$type} eq 'ALIAS';

	    my @end_aliases = FindEndAlias($id, $amap_ptr);			

	    foreach my $end_alias (@end_aliases) {
		next if (!defined $trackmap_ptr->{$end_alias}); 
		my $t = $trackmap_ptr->{$end_alias}->{$type};

		if ((!defined $t) or $time < $t) {
		    print sprintf("%-25s %14.3f\t%12s\t%12s\n", $f, $time, $id, $MsgTypes{$type});
		    $trackmap_ptr->{$end_alias}->{$type} = $time;
		}
	    }
	}
	close F;
    }
}

sub ProcessLogs
{
    my @files = @_;

    my %to_track  = ();
    $to_track{$ID} = {};

    my %alias_map = ();
    my %extrainfo_map = ();

    ConstructAliasMap(\%to_track, \%alias_map, \%extrainfo_map, @files);
    MarkSubTravel(\%to_track, \%alias_map, \%extrainfo_map, @files);

    return %to_track;
}

sub FindEndAlias
{
    my $id = shift;
    my $alias_map = shift;
    my @res;

    if (exists $alias_map->{$id}) {
	my @alias = @{$alias_map->{$id}};

	foreach my $alias (@alias) {
	    splice @res, 0, 0, FindEndAlias($alias, $alias_map);
	}

	return @res;
    } else {
	return ($id);
    }
}

sub ParseObjectLogsHeader() 
{
# XXX This is kinda hacky and could break...
    my $index = 0;

    open(HEADER, "$TOPDIR/../Merc/mercury/ObjectLogs.h") || 
	die "can't open $TOPDIR/../Merc/mercury/ObjectLogs.h";

    my $begun_class = 0;
    my $begun_enum  = 0;

    while (<HEADER>) {
	chomp $_;

	$_ =~ s|//.*||g;

	if ($_ =~ /struct\s+DiscoveryLatEntry/) {
	    $begun_class = 1;
	}
	if ($begun_class && $_ =~ /enum \{/) {
	    $begun_enum = 1;
	    $_ =~ s/enum \{//;
	}

	if ($begun_enum) {
	    if ($_ =~ /((?:[\w_]+(?:,\s*)?)+)/) {
		my $names = $1;
		my @names = split(/,\s*/, $names);
		map { $MsgTypes{$index} = $_; $index++; } @names;
	    }

	    if ($_ =~ /\}/) {
		last;
	    }
	}   
    }

    close(HEADER);
}
