#!/usr/bin/perl
#
# Simulate the bandwidth cost each frame if all nodes were run on a central
# server at the given framerate.
#
# usage: ClientServerSim.pl (in directory containing Object{Interest,Delta}Logs
#
# -b  calculate broadcast costs (1 file per server in bcast.$ip:$port.log
#     with time\tout\tin
#
# XXX HACK: remember to pass in a log with all the items logged in the
# beginning and with timestamps BEFORE the beginning of the processed logs.

# R::Tree:
# http://search.cpan.org/~ajolma/Tree-R-0.05/
# Algorithm::QuadTree:
# http://search.cpan.org/~aqumsieh/Algorithm-QuadTree-0.1/

use strict;
use IO::File;
use Getopt::Std;
use Tree::R;
#use Algorithm::QuadTree;
use POSIX qw(ceil floor);

use Time::HiRes qw(gettimeofday tv_interval);
use vars qw($opt_b $opt_S $opt_s $opt_l $opt_o $opt_x $opt_y $opt_w);

getopts("bS:s:l:o:x:y:w:");

#my @test = ([0,[1,0]],[0,[2,0]],[0,[3,0]],[0,[3,0]],[0,[3,0]],[0,[6,0]],[0,[7,0]]);
#my $lb = lower_bound(\@test, 7);
#my $ub = upper_bound(\@test, 7);
#
#print "$lb $ub\n";
#exit 0;

## BCAST PARAMS
my $BCAST = defined $opt_b;

my $START    = defined $opt_S ? $opt_S : undef;
my $SKIPTIME = defined $opt_s ? $opt_s : 120;
my $LENGTH   = defined $opt_l ? $opt_l : 10*60;
my $OUTDIR   = defined $opt_o ? $opt_o : ".";
our $SLOWDOWN = defined $opt_w ? $opt_w : 1.0;
#my @XRANGE = defined $opt_x ? split(",", $opt_x) : (0,700*4);
#my @YRANGE = defined $opt_y ? split(",", $opt_y) : (0,700*4);

my $FRAME_TIME = 0.1; # sec
if ($SLOWDOWN > 1.0) { 
    $SKIPTIME *= $SLOWDOWN;
    $LENGTH *= $SLOWDOWN;
    $FRAME_TIME *= $SLOWDOWN;
}

my $MONS_AS_PLAYERS = 0;

my $PER_FRAME_OVERHEAD = 28+4+1+4; # proto+seqno+cum+numdeltas
my $GUID_SIZE    = 4+2+4; # same as colyseus
my $BITMASK_SIZE = 4; # per object (bytes)
my $DELETE_COST  = 4; # cost per object delete (bytes)

# 1/2 of our interest TTLs
my %TTLS = ( 'i' => 5,
	     'p' => 3,
	     'o' => 3,
	     'm' => 0.5 );

my @InterestLogs = OpenFiles( glob("ObjectInterestLog.*") );
my @DeltaLogs    = OpenFiles( glob("ObjectDeltaLog.*") ); 

die "no logs: int(@InterestLogs) del(@DeltaLogs)!" 
    if @InterestLogs==0 || @DeltaLogs==0;

# get the max start-time
my $_start = defined $START ? $START : min( GetHeadTimes(@InterestLogs, @DeltaLogs) );
print STDERR "_start=$_start START=$START\n";
my $realstart = $_start + $SKIPTIME;

# scroll logs until that time to synchronize
print STDERR "** scrolling lines to starttime: $realstart: ";
my @ResidualInterestLines = ScrollUntilTime($realstart, @InterestLogs);
my @ResidualDeltaLines    = ScrollUntilTime($realstart, @DeltaLogs);

print STDERR "\n";

# $guid => cost_so_far
my %Clients = ();
# $guid => [ lastvframe_up, type, [pos], [[min],[max]], delta, full, stat, {ints} ]
my %Objects = ();

my ($VF, $TYPE, $POS, $BBOX, $DELTA, $FULL, $STAT, $INTS) = 
    (0,1,2,3,4,5,6,7);

my $frame_end = $realstart;
my $vframe = 0;

###############################################################################

my $CURRENT_SERVER = undef;
my @BCAST_OUT;
my %GUID_MAP;
if ($BCAST) {
    foreach my $file (glob("ObjectDeltaLog.*")) {
	#next if $file =~ /10\.1\.1\.2\:/; # skip master!
	$file =~ /(\d+\.\d+\.\d+\.\d+\:\d+)/;
	#print "$1\n";
	die "bad filename: $file" if ! $1;
	my $fh = new IO::File(">$OUTDIR/bcast.$1.out");
	die "can't create bcast.$1.out" if !$fh;
	push @BCAST_OUT, $fh;
    }
}

print STDERR "** running sim: ";

my %timers;
my %iters;
sub rec {
    $timers{$_[0]} += $_[1];
    $iters{$_[0]}++;
}

while (1) {
    $frame_end = $frame_end+$FRAME_TIME;

    #start();
    BeginFrame();
    #rec('BeginFrame', stop());
    
    #start();
    ProcessInterestLines(@ResidualInterestLines);
    @ResidualInterestLines = ProcessInterests($frame_end);
    #rec('ProcessInterestLines', stop());

    #start();
    ProcessDeltaLines(@ResidualDeltaLines);
    @ResidualDeltaLines = ProcessDeltas($frame_end);
    #rec('ProcessDeltaLines', stop());

    #start();
    ComputeCosts();
    $vframe++;
    #rec('ComputeCosts', stop());

    if ($vframe % 10 == 0) {
	print STDERR ".";
	if ($vframe % 500 == 0) {
	    print STDERR $frame_end;
	}
    }

    if (! @ResidualDeltaLines ) {
	last;
    }

    last if ($frame_end > $realstart + $LENGTH);
}
print STDERR "\n ===== DONE! =====\n";

print STDERR "benchmark timers:\n";
foreach my $t (sort { -($timers{$a} <=> $timers{$b}) } keys %timers) {
    print STDERR "$t\t$timers{$t}\t" . ($timers{$t}/$iters{$t}) . "\t$iters{$t}\n";
}

###############################################################################

sub GetObj
{
    my $guid = shift;

    if (!defined $Objects{$guid}) {
	$Objects{$guid} = [undef,undef,undef,[],undef,undef,undef,{}];
    }

    return $Objects{$guid};
}

sub BBoxContains($$)
{
    my $box = shift;
    my $pt  = shift;

    for (my $i=0; $i<3; $i++) {
	if ($box->[0]->[$i] > $pt->[$i] || 
	    $box->[1]->[$i] < $pt->[$i]) {
	    return 0;
	}
    }

    return 1;
}


###############################################################################

sub BeginFrame
{
    foreach my $g (keys %Objects) {
	my $ref = $Objects{$g};

	# delete it if it was deleted in the last frame.
	if ($ref->[$STAT] eq 'd') {
	    delete $Objects{$g};
	}

	# clear its delta
	$ref->[$DELTA] = undef;
	$ref->[$STAT] = undef;

	# garbage collect stale interests
	my $ints = $ref->[$INTS];
	foreach my $k (keys %$ints) {
	    #print "here: $g->$k $vframe < $ints->{$k}\n";
	    if ($ints->{$k} <= $vframe) {
		#print "here: $g->$k $vframe < $ints->{$k}\n";
		delete $ints->{$k};
	    }
	}
    }
}

sub ComputeCosts
{
    my $time = $realstart + $vframe*$FRAME_TIME;
    my $framecost = 0;

    if ($BCAST) {

	my @bcast_out;
	my @bcast_in;

	my $counted = 0;
	foreach my $o (keys %Objects) {
	    my $oref = $Objects{$o};
	    my $pos = $oref->[$POS];
	    next if !$pos;

	    next if !defined $GUID_MAP{$o};

	    my $cost;
	    if ($oref->[$STAT] eq 'd') {
		# was a deletion
		$cost = $DELETE_COST + $GUID_SIZE;
	    } else {
		$cost = $oref->[$DELTA];
	    }

	    $counted++;
	    
	    for (my $i=0; $i<@BCAST_OUT; $i++) {
		if ($i == $GUID_MAP{$o}) {
		    $bcast_out[$i] += (scalar(@BCAST_OUT)-1)*$cost;
		} else {
		    $bcast_in[$i] += $cost;
		}
	    }
	}

	for (my $i=0; $i<@BCAST_OUT; $i++) {
	    if ($bcast_out[$i]) {
		$bcast_out[$i] += $PER_FRAME_OVERHEAD*(scalar(@BCAST_OUT)-1);
	    } else {
		$bcast_out[$i] = 0;
	    }
	    if ($bcast_in[$i]) {
		$bcast_in[$i] += $PER_FRAME_OVERHEAD*(scalar(@BCAST_OUT)-1);
	    } else {
		$bcast_in[$i] = 0;
	    }
	    my $fh = $BCAST_OUT[$i];
	    
	    print $fh "$time\t$bcast_out[$i]\t$bcast_in[$i]\n";
	}

	return;
    }

    #print "clients: " . (keys %Clients) . "\n";

##

# quad-tree filtering:
#    our $qtree = new Algorithm::QuadTree(-xmin  => $XRANGE[0],
#					 -xmax  => $XRANGE[1],
#					 -ymin  => $YRANGE[0],
#					 -ymax  => $YRANGE[1],
#					 -depth => 8);

# r-tree filtering:
    our $rtree = new Tree::R;

# sort on x filtering:
#    our @stree;

    #start();
    my $s = 0;
    foreach my $o (keys %Objects) {
	my $oref = $Objects{$o};
	my $pos = $oref->[$POS];
	next if !$pos;

#	warn "object $o has bad pos: ($pos->[0], $pos->[1])\n"
#	    if ($pos->[0] < $XRANGE[0] ||
#		$pos->[0] > $XRANGE[1] ||
#		$pos->[1] < $YRANGE[0] ||
#		$pos->[1] > $YRANGE[1]);
	
#	$qtree->add($o, $pos->[0], $pos->[1], $pos->[0], $pos->[1]);
	$rtree->insert($o, $pos->[0], $pos->[1], $pos->[0], $pos->[1]);
	$s += $pos->[0];
	
#	push @stree, [$o, $pos];
    }
    #rec('insert', stop());
    #print STDERR "perl; s=$s\n";

#    @stree = sort { $a->[1]->[0] <=> $b->[1]->[0] } @stree;

##

    my $index = 0;
    my $p = 0;
    foreach my $c (keys %Clients) {
	my $cref = $Objects{$c};
	next if !$cref || !$cref->[$POS];
	my $bbox = $cref->[$BBOX];
	my $client_framecost = 0;


# print STDERR "guid=$c clients=$index\n";
# naive filtering:
#	foreach my $o (keys %Objects) {
#	    my $oref = $Objects{$o};
#	    my $pos = $oref->[$POS];
#	    next if !$pos;

#	    start();
#	    my $bbox_contains = BBoxContains($bbox, $pos);
#	    rec('bbox_contains', stop());
#	    start();
#	    my $old_interest  = $cref->[$INTS]->{$o};
#	    rec('old_interest', stop());

##

# quad-tree filtering:
#	my $bbox_contains = $qtree->getEnclosedObjects($bbox->[0]->[0], 
#						       $bbox->[0]->[1],
#						       $bbox->[1]->[0], 
#						       $bbox->[1]->[1]);

# r-tree filtering:
	my $bbox_contains = [];
	#start();
	$rtree->query_completely_within_rect($bbox->[0]->[0], 
					     $bbox->[0]->[1],
					     $bbox->[1]->[0], 
					     $bbox->[1]->[1],
					     $bbox_contains);
	#rec('query', stop());

# sort on x filtering:
#	my $bbox_contains = [];
#	stree_in_rect(\@stree, $bbox, $bbox_contains);

	# xxx for quake or 3D we need to add a z-dim check here!
	$bbox_contains = { map { $_ => 1 } @$bbox_contains };
	my $old_interest  = $cref->[$INTS];

	
	$p += scalar (keys %$bbox_contains);
	$index++;
	
	my %all = map { $_ => 1 } (keys %$bbox_contains, keys %$old_interest);
	foreach my $o (keys %all) {
	    my $oref = $Objects{$o};
	    my $pos = $oref->[$POS];
	    next if !$pos;
	    my $bbox_contains = $bbox_contains->{$o};
	    my $old_interest = $old_interest->{$o};
##

	    if ($bbox_contains || $old_interest) {
		# this guy is interested!
		my $cost;

		#start();

		if ($cref->[$INTS]->{$o}) {
		    # was interested before, just send delta
		    if (!defined $oref->[$DELTA] ) {
			if ($oref->[$STAT] eq 'd') {
			    # was a deletion
			    $cost = $DELETE_COST + $GUID_SIZE;
			    delete $cref->[$INTS]->{$o};
                            # print STDERR "here1\n";
			} else {
			    $cost = 0;
			    # print STDERR "here2\n";
			}
		    } else {
			$cost = $oref->[$DELTA] + $BITMASK_SIZE + $GUID_SIZE;
			# print STDERR "here3\n";
		    }
		} else {
		    # is newly interested in object
		    if (!defined $oref->[$FULL] ) {
			# print STDERR "here5\n";
			$cost = 0;
		    } else {
			$cost = $oref->[$FULL] + $BITMASK_SIZE + $GUID_SIZE;
			# print STDERR "here4\n";
		    }
		}

		# if interested this frame, refresh its ttl
		if ($bbox_contains && $oref->[$STAT] ne 'd') {
		    my $ttl = $TTLS{$oref->[$TYPE]}*(1/$FRAME_TIME);
		    die if !defined $ttl;
		    $cref->[$INTS]->{$o} = $vframe + $ttl;
		    #print "here: $oref->[$TYPE] $ttl\n";
		}

#		print  STDERR "\tperl; $o $cost full=$oref->[$FULL] delta=$oref->[$DELTA]\n";
		$client_framecost += $cost;

		#rec('bbox_contains_true', stop());
	    }
	}

	# XXX HACK -- we're missing items! so get them from a static log
	#foreach my $i (keys %ITEM_MAP) {
	#    my $pos = $ITEM_MAP{$i};
        #  
	#    my $bbox_contains = BBoxContains($bbox, $pos);
	#    my $old_interest  = $cref->[$INTS]->{$i};
        #
	#    if ($bbox_contains && !$old_interest) {
	#	# this guy is newly interested!
	#	$client_framecost += $ITEM_SIZE + $BITMASK_SIZE + $GUID_SIZE;
	#	#print "here: $c->$i\n";
	#    } else {
	#	# assume items don't make updates that are worth recording...
	#    }
        #
	#    # if interested this frame, refresh its ttl
	#    if ($bbox_contains) {
	#	my $ttl = $TTLS{'i'}*(1/$FRAME_TIME);
	#	$cref->[$INTS]->{$i} = $vframe + $ttl;
	#    }
	#}

	if ($client_framecost > 0) {
	    $client_framecost += $PER_FRAME_OVERHEAD;
	}

	# print STDERR "perl; client=$c vframe=$vframe framecost=$client_framecost\n";
	$Clients{$c} += $client_framecost;
	$framecost   += $client_framecost;
	my $num_ints = scalar keys %{$Objects{$c}->[$INTS]};

	#print "$time\t$c\t$client_framecost\t$num_ints\n";
    }
#print STDERR "perl; clients=$index total bbox_contains=$p\n";
    print "$time\t$framecost\n";
}

###############################################################################

sub ProcessInterestLine
{
    my $line = shift;

    if ($line =~ /^\d+\.\d+\t\d+\t([\d\.\:]+)\t(\w)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)\t([-\d\.]+),([-\d\.]+),([-\d\.]+)/) {
	my $guid = $1;
	my $type = $2;
	my @pos  = ($3,$4,$5);
	my @min  = ($6,$7,$8);
	my @max  = ($9,$10,$11);

	if ($type eq 'p'||
	    ($MONS_AS_PLAYERS && $type eq 'o')) {
	    if (!exists $Clients{$guid}) {
		$Clients{$guid} = 0;
	    }
	}

	if (defined $CURRENT_SERVER) {
	    $GUID_MAP{$guid} = $CURRENT_SERVER;
	}

	my $ref = GetObj($guid);
	$ref->[$POS] = \@pos;
	$ref->[$TYPE] = $type;
	$ref->[$BBOX]->[0] = \@min;
	$ref->[$BBOX]->[1] = \@max;

# print STDERR sprintf ("perl; type=$type pos=%.3f,%.3f,%.3f\n", $pos[0], $pos[1], $pos[2]);
    } else {
	warn "bad interest line: $line\n";
    }
}

sub ProcessInterestLines
{
    my @lines = @_;

    $CURRENT_SERVER = undef; # dunno

    foreach my $line (@lines) {
	ProcessInterestLine($line);
    }
}

sub ProcessInterests
{
    my $endtime = shift;
    my @residual;

    my $index = 0;
    foreach my $f (@InterestLogs) {
	$CURRENT_SERVER = $index++;
	
	while (<$f>) {
	    chomp $_;
	    $_ =~ /^(\d+\.\d+)/;
	    my $time = $1;
	    if ($time >= $endtime) {
		push @residual, $_;
		last;
	    }
	    ProcessInterestLine($_);
	}
    }

    return @residual;
}

###############################################################################

sub ProcessDeltaLine
{
    my $line  = shift;
    my $force = shift;

    if ($line =~ /^\d+\.\d+\t\d+\t([\d\.\:]+)\t(\w)\t(\d+)\t(\d+)/) {
	my $guid = $1;
	my $status = $2;
	my $delta = $3;
	my $full = $4;

	my $ref = GetObj($guid);

	if (!$force && defined $ref->[$VF] && $ref->[$VF] >= $vframe) {
	    # already did an update for this obj during this vframe,
	    # save this update and apply it the next vframe
	    
	    return $line;
	}

	$ref->[$VF]    = $vframe;
	$ref->[$STAT]  = $status;
	$ref->[$DELTA] = $delta;
	$ref->[$FULL]  = $full;

	# print STDERR "perl* reading delta info force={$force} for ($guid) full=($full) delta=($delta)\n";
    } else {
	warn "bad interest line: $line\n";
    }

    return undef;
}

sub ProcessDeltaLines
{
    my @lines = @_;

    foreach my $line (@lines) {
	ProcessDeltaLine($line, 1);
    }
}

sub ProcessDeltas
{
    my $endtime = shift;
    my @residual;

    my $index = 0;

    foreach my $f (@DeltaLogs) {
	$CURRENT_SERVER = $index++;
	while (<$f>) {
	    chomp $_;
	    $_ =~ /^(\d+\.\d+)/;
	    my $time = $1;
	    if ($time >= $endtime) {
		push @residual, $_;
		last;
	    }
	    # might get 2 updates for same object in 1 "approx" frame
	    # if so, just delay one of them until next frame
	    
	    my $resid = ProcessDeltaLine($_);
	    push @residual, $resid if $resid;
	}
    }

    return @residual;
}

###############################################################################

sub OpenFiles {
    my @logs = @_;
    my @ret;

    foreach my $file (sort { $a cmp $b } @logs) {
	#next if $file =~ /10\.1\.1\.2\:/; # skip master!
	my $fh;

	if ($file =~ /.gz$/) { 
	    $fh = new IO::File ("gunzip -c $file | ");
	}
	else {
	    $fh = new IO::File("<$file");
	}
	die "can't open $file: $!" if !$fh;
	push @ret, $fh;
    }

    return @ret;
}


sub GetHeadTimes {
    my @logs = @_;
    my @ret;

    foreach my $log (@logs) {
	my $line = <$log>;
	die "no lines in log!" if !$line;
	chomp $line;
	$line =~ /^(\d+\.\d+)/;
	push @ret, $1;
    }

    return @ret;
}

sub ScrollUntilTime {
    my $time = shift;
    my @logs = @_;
    my @ret;

    foreach my $log (@logs) {
	my $line;

	print STDERR ".";

	while ($line = <$log>) {
	    chomp $line;
	    $line =~ /^(\d+\.\d+)/;
	    my $t = $1;
	    #push @ret, $line;
	    if ($t < $time) {
		next;
	    } else {
		last;
	    }
	}
	push @ret, $line if $line;
    }

    print STDERR "done!\n";

    return @ret;
}

sub min {
    my @vals = @_;
    my $min = shift @vals;

    foreach my $v (@vals) {
	if ($min > $v) {
	    $min = $v;
	}
    }

    return $min;
}

sub max {
    return $_[0] > $_[1] ? $_[0] : $_[1];
}

###############################################################################

our $_time;

sub start()
{
    $_time = [gettimeofday()];
}

sub stop()
{
    return tv_interval($_time,[gettimeofday()]);
}

# find the smallest index such that $i < stree[index]
sub lower_bound
{
    my ($stree,$val,$dmin,$dmax) = @_;

    my $min = $dmin ? $dmin : 0;
    my $max = $dmax ? $dmax : (scalar @$stree)-1;
    my $mid;

    if ($val < $stree->[$min]->[1]->[0] ||
	$val > $stree->[$max]->[1]->[0]) {
	return undef;
    }

    while ($min < $max) {
	$mid = $max - floor(($max-$min)/2);

	if ($val >= $stree->[$mid]->[1]->[0]) {
	    $min = $mid+1;
	} elsif ($val < $stree->[$mid]->[1]->[0]) {
	    $max = $mid-1;
	} else {
	    # there may be duplicate values
	    #while ($mid > $min && $stree->[$mid-1]->[1]->[0] == $val) {
	    #--$mid;
	    #}
	    #return $mid;
	    die;
	}
    }

    return $val >= $stree->[$max]->[1]->[0] ? $max+1 : $max;
}

# find the largest index such that stree[index] < $i
sub upper_bound
{
    my ($stree,$val,$dmin,$dmax) = @_;

    my $min = $dmin ? $dmin : 0;
    my $max = $dmax ? $dmax : (scalar @$stree)-1;
    my $mid;

    if ($val < $stree->[$min]->[1]->[0] ||
	$val > $stree->[$max]->[1]->[0]) {
	return undef;
    }

    while ($min < $max) {
	$mid = $min + floor(($max-$min)/2);

	if ($val > $stree->[$mid]->[1]->[0]) {
	    $min = $mid+1;
	} elsif ($val <= $stree->[$mid]->[1]->[0]) {
	    $max = $mid-1;
	} else {
	    # there may be duplicate values, so search each
	    # half of the range separately
	    #while ($mid < $max && $stree->[$mid+1]->[1]->[0] == $val) {
	    #++$mid;
	    #}
	    #return $mid;
	    die;
	}
    }

    #if ($val == $stree->[$max]->[1]->[0]) {
    #return $max;
    #}

    return $val <= $stree->[$max]->[1]->[0] ? $max-1 : $max;
}

sub stree_in_rect
{
    my ($stree, $rect, $res) = @_;

    my $minindex = undef;
    my $maxindex = undef;
    
    $minindex = lower_bound($stree, $rect->[0]->[0]);
    $maxindex = upper_bound($stree, $rect->[1]->[0]);

    return if (!defined $minindex || !defined $maxindex);

    for (my $i=$minindex; $i<=$maxindex; $i++) {
	push @$res, $stree->[$i]->[0] if
	    BBoxContains($rect, $stree->[$i]->[1]);
    }
    return;
}
