////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <map>
#include <hash_map.h>
#include <vector>
#include <string>
#include <list>
#include <util/debug.h>
#include <util/Options.h>
#include <gameapi/common.h>
#include <util/GPL/callback.h>
#include <util/RTree.h>

#include "putils.cxx"

double START;
int SKIPTIME;
int LENGTH;
char  OUTDIR[1024];
bool  BCAST;
float SLOWDOWN;
float FRAME_TIME;

// count the number of updates to new objects (that come within sub)
// range vs ones to existing objects a client is already maintaining.
bool NEW_VS_EXISTING;

const int DIMENSION = 2;    // how many co-ordinates are used { 1 | 2 | 3 }

OptionType options[] = {
    { 'b', "bcast", OPT_NOARG | OPT_BOOL, "", &BCAST, "0", (void *) "1"}, 
    { 'o', "outdir", OPT_STR, "", OUTDIR, ".", NULL },
    { 'S', "start", OPT_DBL, "", &START, "-1.0", NULL },
    { 'l', "length", OPT_INT, "", &LENGTH, "600", NULL  },
    { '/', "frametime", OPT_FLT, "frame-time in seconds", &FRAME_TIME, "0.1", NULL },
    { 's', "skiptime", OPT_INT, "", &SKIPTIME, "120", NULL  },
    { 'w', "slowdown", OPT_FLT, "", &SLOWDOWN, "1.0", NULL },
    { 'N', "newvsexisting", OPT_NOARG|OPT_BOOL, "", &NEW_VS_EXISTING, "0", (void *) "1"},
    { 0, 0, 0, 0, 0, 0, 0 },
};

// data type definitions
struct obj;
typedef hash_map<string, struct obj *, hash_string_1, eq_string> GUIDObjMap;
typedef GUIDObjMap::iterator GUIDObjMapIter;

GUIDObjMap Objects;

typedef hash_map<string, int, hash_string_1, eq_string> GUIDCliMap;
typedef GUIDCliMap::iterator GUIDCliMapIter;

GUIDCliMap Clients;

typedef hash_map<string, float, hash_string_1, eq_string> ObjInts;
typedef ObjInts::iterator ObjIntsIter;

typedef hash_map<string, int, hash_string_1, eq_string> ObjSvrMap;
typedef ObjSvrMap::iterator ObjSvrMapIter;

ObjSvrMap  GUID_MAP;

struct obj {
    int VF;
    char TYPE;
    vec3_t POS;
    vec3_t BBOX_MIN;
    vec3_t BBOX_MAX;
    char STAT;
    int DELTA;
    int FULL;
    string guid;
    ObjInts INTS;

    void GetExtent (Rect<double> *extent) {
	for (int i = 0; i < DIMENSION; i++) 
	    extent->min[i] = extent->max[i] = POS[i];
    }
};

map<char, double> TTLS;
int PER_FRAME_OVERHEAD = 28 + 61 /* 4 + 1 + 4 */; // using our huge overhead numbers now
int GUID_SIZE = 4 + 2 + 4;
int BITMASK_SIZE = 36 /* 4 */;      // using our huge overhead numbers now
int DELETE_COST = 4;
bool MONS_AS_PLAYERS = false;

vector<FILE *> InterestLogs;
vector<FILE *> DeltaLogs;

vector<FILE *> BCAST_OUT;

void init_bcast_output ()
{
    vector<string> files;
    glob ("ObjectDeltaLog.*", &files);
    for (vector<string>::iterator it = files.begin (); it != files.end (); ++it) {
	if (!match_capture ("(\\d+\\.\\d+\\.\\d+\\.\\d+\\:\\d+)", (char *) it->c_str (), 1, (char **) patbufs))
	    Debug::die (merc_va ("bad filename: %s", it->c_str ()));

	string s = OUTDIR;
	s.append (merc_va ("/bcast.%s.out", patbufs[0]));

	FILE *fp = fopen (s.c_str (), "w");
	if (!fp) 
	    Debug::die ("cant open file for writing");

	BCAST_OUT.push_back (fp);
    }
}

struct obj *GetObj (char *g)
{
    string s (g);
    GUIDObjMapIter it = Objects.find (s);

    if (it == Objects.end ()) { 
	struct obj *n = new struct obj ();
	n->VF = -1;
	n->TYPE = '\0';
	n->STAT = '\0';
	n->DELTA = -1;
	n->FULL = -1;
	n->guid = g;

	for (int i = 0; i < 3; i++) 
	    n->POS[i] = n->BBOX_MIN[i] = n->BBOX_MAX[i] = sqrt (-1.0);    // create a NaN

	Objects.insert (GUIDObjMap::value_type (n->guid, n));    
	return n;
    }
    return it->second;
}

void BeginFrame (int vframe)
{
    for (GUIDObjMapIter it = Objects.begin (); it != Objects.end (); /* ++it */) {
	struct obj *o = it->second;
	// cerr << "guid=" << it->first << endl;
	if (o->STAT == 'd') {
	    GUIDObjMapIter oit (it);
	    ++oit;
	    Objects.erase (it);

	    delete o;
	    it = oit;
	}
	else {
	    o->DELTA = -1;
	    o->STAT = '\0';

	    // garbage collect stale interests
	    for (ObjIntsIter intit = o->INTS.begin (); intit != o->INTS.end (); /* ++intit */) {
		if (intit->second <= vframe) {
		    ObjIntsIter ointit (intit);
		    ++ointit;

		    o->INTS.erase (intit);
		    intit = ointit;
		}
		else {
		    ++intit;
		}
	    }
	    ++it;
	}
    }
}

void print_obj (struct obj *o)
{
    fprintf (stderr, "type=%c pos=%.3f,%.3f,%.3f\n", o->TYPE, o->POS[0], o->POS[1], o->POS[2]);
}

void ProcessInterestLine (string& interest_line, int current_server)
{
    if (!match_capture (
			"^\\d+\\.\\d+\\t\\d+\\t([\\d\\.\\:]+)\\t(\\w)\\t([-\\d\\.]+),([-\\d\\.]+),([-\\d\\.]+)\\t([-\\d\\.]+),([-\\d\\.]+),([-\\d\\.]+)\\t([-\\d\\.]+),([-\\d\\.]+),([-\\d\\.]+)",
			(char *) interest_line.c_str (), 
			11, /* npats */
			(char **) patbufs)) {
	Debug::warn ("bad interest line: %s", interest_line.c_str ());
	return;
    }

    char *guid = patbufs[0];
    char type = *(patbufs[1]);

    struct obj *o = GetObj (guid);

    if (type == 'p' || (MONS_AS_PLAYERS && type == 'o')) {
	if (Clients.find (o->guid) == Clients.end ()) {
	    Clients[o->guid] = 0;
	}
    }
    if (current_server >= 0) {
	GUID_MAP[o->guid] = current_server;
    }

    o->TYPE = type;
    o->POS[0] = atof (patbufs[2]);
    o->POS[1] = atof (patbufs[3]);
    o->POS[2] = atof (patbufs[4]);

    o->BBOX_MIN[0] = atof (patbufs[5]);
    o->BBOX_MIN[1] = atof (patbufs[6]);
    o->BBOX_MIN[2] = atof (patbufs[7]);

    o->BBOX_MAX[0] = atof (patbufs[8]);
    o->BBOX_MAX[1] = atof (patbufs[9]);
    o->BBOX_MAX[2] = atof (patbufs[10]);

    // print_obj (o);
}

void ProcessInterestLines (vector<string>& residual_interests)
{
    for (vector<string>::iterator it = residual_interests.begin (); it != residual_interests.end (); ++it)
	ProcessInterestLine (*it, -1 /* current_server */);
}

void ProcessInterests (double frame_end, vector<string> *residual_interests)
{
    int index = 0;

    for (vector<FILE *>::iterator it = InterestLogs.begin (); it != InterestLogs.end (); ++it) {
	memset (line, 0, sizeof (line));
	while (fgets (line, sizeof (line), *it)) {
	    match_capture ("^(\\d+\\.\\d+)", line, 1, (char **) patbufs);
	    double f;
	    sscanf (patbufs[0], "%lf", &f);

	    if (f >= frame_end) {
		residual_interests->push_back (line);
		break;
	    }

	    string s (line);
	    ProcessInterestLine (s, index);
	}
	index++;
    }
}

bool ProcessDeltaLine (string& delta_line, int vframe, bool force)
{
    if (!match_capture (
			"^\\d+\\.\\d+\\t\\d+\\t([\\d\\.\\:]+)\\t(\\w)\\t(\\d+)\\t(\\d+)",
			(char *) delta_line.c_str (),
			4,
			(char **) patbufs)) {
	Debug::warn ("bad delta line: %s", delta_line.c_str ());
	return false;
    }

    char *guid = patbufs[0];
    char status = *(patbufs[1]);
    int delta = atoi (patbufs[2]);
    int full = atoi (patbufs[3]);

    struct obj *o = GetObj (guid);
    if (!force && o->VF >= vframe) 
	return true;

    o->VF = vframe;
    o->STAT = status;
    o->DELTA = delta;
    o->FULL = full;

    // fprintf (stderr, "* reading delta info [force=%d] for (%s) full=(%d) delta=(%d)\n",
    //       force, o->guid, o->FULL, o->DELTA);
    return false;
}

void ProcessDeltaLines (vector<string>& residual_deltas, int vframe)
{
    for (vector<string>::iterator it = residual_deltas.begin (); it != residual_deltas.end (); ++it)
	ProcessDeltaLine (*it, vframe, true);
}

void ProcessDeltas (double frame_end, int vframe, vector<string> *residual_deltas)
{
    for (vector<FILE *>::iterator it = DeltaLogs.begin (); it != DeltaLogs.end (); ++it) {
	memset (line, 0, sizeof (line));
	while (fgets (line, sizeof (line), *it)) {
	    match_capture ("^(\\d+\\.\\d+)", line, 1, (char **) patbufs);
	    double f;
	    sscanf (patbufs[0], "%lf", &f);

	    if (f >= frame_end) {
		residual_deltas->push_back (line);
		break;
	    }

	    string s (line);
	    if (ProcessDeltaLine (s, vframe, false)) 
		residual_deltas->push_back (line);
	}
    }
}

void BCastComputeCosts (int vframe, double realstart)
{
    double tim = realstart + vframe * FRAME_TIME;
    int framecost = 0;

    vector<int> bcast_outs, bcast_ins;
    for (int i = 0, n = BCAST_OUT.size (); i < n; i++) 
	bcast_outs.push_back (0), bcast_ins.push_back (0);

    int counted = 0;

    for (GUIDObjMapIter it = Objects.begin (); it != Objects.end (); ++it) {
	struct obj *o = it->second;
	if (isnan (o->POS[0]))
	    continue;

	ObjSvrMapIter svit = GUID_MAP.find (it->first);
	if (svit == GUID_MAP.end ())
	    continue;

	counted++;
	int cost = 0;
	if (o->STAT == 'd')
	    cost = DELETE_COST + GUID_SIZE;
	else
	    cost = o->DELTA;

	for (int i = 0, n = BCAST_OUT.size (); i < n; i++) {
	    if (i == svit->second) {
		bcast_outs[i] += (n - 1) * cost;
	    }
	    else
		bcast_ins[i] += cost;
	}	
    }

    for (int i = 0, n = BCAST_OUT.size (); i < n; i++) {
	// fprintf (stderr, "vframe=%d i=%d rc=%d avg=%.3f\n", vframe, i, rc[i], (rc[i] == 0 ? -1 : tc[i]/rc[i]));
	if (bcast_outs[i] > 0)
	    bcast_outs[i] += PER_FRAME_OVERHEAD * (n - 1);
	if (bcast_ins[i] > 0)
	    bcast_ins[i] += PER_FRAME_OVERHEAD * (n - 1);

	fprintf (BCAST_OUT[i], "%.3f\t%d\t%d\n", tim, bcast_outs[i], bcast_ins[i]);
    }
}

GUIDCliMap bbox_contains;
GUIDCliMap all;

void collect_matches (struct obj *o) {
    bbox_contains[o->guid] = 1;
    all[o->guid] = 1;
}

void _clear_hash_table (GUIDCliMap& hm) {
    hm.clear ();
}

struct less_const_char_t {
    bool operator () (const char *a, const char *b) { 
	return strcmp (a, b) < 0;
    }
};

void CliSerComputeCosts (int vframe, double realstart)
{
    double tim = realstart + vframe * FRAME_TIME;
    int framecost = 0;

    RTree<struct obj, double> rtree (DIMENSION);
    Rect<double> trect (DIMENSION);

    // insert all objects into the rtree
    for (GUIDObjMapIter it = Objects.begin (); it != Objects.end (); ++it) {
	struct obj *o = it->second;
	if (isnan (o->POS[0]))
	    continue;

	rtree.Insert (o);
    }

    // int p = 0;
    int index = 0;

    // check visible objects for each client, according his bbox
#ifdef SORT_CLIENTS
    list<const char *> tclist;
    for (GUIDCliMapIter it = Clients.begin (); it != Clients.end (); ++it) {
	tclist.push_back (it->first);
    }
    tclist.sort (less_const_char_t ());
#endif

#ifdef SORT_CLIENTS
    for (list<const char *>::iterator it = tclist.begin (); it != tclist.end (); ++it) 
#else
	for (GUIDCliMapIter it = Clients.begin (); it != Clients.end (); ++it) 
#endif
	    {
		// cerr << "guid=" << *it << " client=" << index << endl;
#ifdef SORT_CLIENTS
		GUIDObjMapIter oit = Objects.find (*it);
#else
		GUIDObjMapIter oit = Objects.find (it->first);
#endif

		if (oit == Objects.end () || isnan (oit->second->POS[0]))
		    continue;

		struct obj *c = oit->second;
		int client_framecost;

		client_framecost = 0;
		for (int i = 0; i < DIMENSION; i++)
		    trect.min[i] = c->BBOX_MIN[i], trect.max[i] = c->BBOX_MAX[i];

		// clear the hash tables 
		_clear_hash_table (bbox_contains);
		_clear_hash_table (all);

		// check for overlaps this frame
		rtree.GetOverlaps (&trect, wrap (collect_matches));

		// p += bbox_contains.size ();

		// add our old_interests
		for (ObjIntsIter oint = c->INTS.begin (); oint != c->INTS.end (); ++oint) {
		    GUIDObjMapIter pit = Objects.find (oint->first);
		    if (pit == Objects.end ()) 
			continue;

		    all[pit->second->guid] = 1;
		}
		index++;

		for (GUIDCliMapIter lit = all.begin (); lit != all.end (); ++lit) {
		    struct obj *oref = Objects[lit->first];
		    ASSERT (oref != NULL);
		    int cost = 0;

		    ObjIntsIter oit2 = c->INTS.find (oref->guid);
		    if (oit2 != c->INTS.end ()) {
			if (NEW_VS_EXISTING) {
			    fprintf(stdout, "%s\t%c\t1\n", c->guid.c_str(), oref->TYPE);
			}
			if (oref->DELTA < 0) { 
			    if (oref->STAT == 'd') {
				// fprintf (stderr, "here1\n");
				cost = DELETE_COST + GUID_SIZE;
				c->INTS.erase (oit2);
			    }
			    else {
				// fprintf (stderr, "here2\n");
			    }
			}
			else {
			    // fprintf (stderr, "here3\n");
			    cost = oref->DELTA + BITMASK_SIZE + GUID_SIZE;
			}
		    }
		    else {  // newly interested 
			if (NEW_VS_EXISTING) {
			    fprintf(stdout, "%s\t%c\t0\n", c->guid.c_str(), oref->TYPE);
			}
			if (oref->FULL >= 0) {
			    // fprintf (stderr, "here4\n");
			    cost = oref->FULL + BITMASK_SIZE + GUID_SIZE;
			}
			else {
			    // fprintf (stderr, "here5\n");
			}
		    }

		    GUIDCliMapIter it2 = bbox_contains.find (oref->guid);	    
		    if (it2 != bbox_contains.end () && (oref->STAT != 'd')) { 
			float ttl = TTLS[oref->TYPE] * (1.0 / FRAME_TIME);

			ObjIntsIter doit = c->INTS.find (oref->guid);
			if (doit != c->INTS.end ()) {
			    doit->second = vframe + ttl;
			}
			else {
			    c->INTS[oref->guid] = vframe + ttl;
			}
		    }

		    // fprintf (stderr, "\tcpp; guid=%s %d full=%d delta=%d\n", oref->guid, cost, oref->FULL, oref->DELTA);
		    client_framecost += cost;
		}

		if (client_framecost > 0)
		    client_framecost += PER_FRAME_OVERHEAD;
		// fprintf (stderr, "client=%s vframe=%d framecost=%d\n", c->guid, vframe, client_framecost);
#ifdef SORT_CLIENTS
#else
		it->second += client_framecost;
#endif
		framecost += client_framecost;
	    }

    // cerr << "clients=" << index << " total bbox_contains=" << p << endl;
    if (!NEW_VS_EXISTING)
	fprintf (stdout, "%.3f\t%d\n", tim, framecost);
}

int main (int argc, char *argv[])
{
    DBG_INIT (NULL);
    InitCPUMHz();
    Benchmark::init();

    ProcessOptions (options, argc, argv, true);

    if (SLOWDOWN > 1.0) { 
	SKIPTIME = (int) (SKIPTIME * SLOWDOWN);
	LENGTH = (int) (LENGTH * SLOWDOWN);
	FRAME_TIME = FRAME_TIME * SLOWDOWN;
    }

    for (int i = 0; i < NPATBUFS; i++)
	patbufs[i] = new char[1024];

    // init TTLS
    TTLS['i'] = 5;
    TTLS['p'] = 3;
    TTLS['o'] = 3;
    TTLS['m'] = 0.5;

    // open logs
    OpenLogs (InterestLogs, "InterestLog.*");
    OpenLogs (DeltaLogs, "DeltaLog.*");

    if (InterestLogs.size () == 0 || DeltaLogs.size () == 0) 
	Debug::die ("no logs to process");

    double _start = START;
    if (START < 0) 
	_start = GetMinStart (InterestLogs, DeltaLogs);

    double realstart = _start + SKIPTIME;

    cerr << "** scrolling lines to starttime: " << merc_va ("%.3f", realstart) << endl;
    vector<string> residual_interests, residual_deltas;

    ScrollUntilTime (realstart, InterestLogs, &residual_interests);
    ScrollUntilTime (realstart, DeltaLogs, &residual_deltas);    

    cerr << "done!" << endl;

    if (BCAST) { 
	init_bcast_output ();
    }

    cerr << "** running sim: "; cerr.flush ();

    double frame_end = realstart;
    int vframe = 0;
    while (true) {
	frame_end += FRAME_TIME;

	// cerr << "begin frame " << endl;
	BeginFrame (vframe);

	// cerr << "process interest lines" << endl;
	// process interests
	ProcessInterestLines (residual_interests);
	residual_interests.clear ();
	ProcessInterests (frame_end, &residual_interests);

	// cerr << "process delta lines " << endl;
	// process deltas
	ProcessDeltaLines (residual_deltas, vframe);
	residual_deltas.clear ();
	ProcessDeltas (frame_end, vframe, &residual_deltas);

	START(FRAMETIME);
	if (BCAST) 
	    BCastComputeCosts (vframe, realstart);
	else
	    CliSerComputeCosts (vframe, realstart);
	STOP(FRAMETIME);

	vframe++;
	if (vframe % 10 == 0) {
	    cerr << "."; cerr.flush ();
	    if (vframe % 500 == 0) {
		cerr << merc_va ("%.3f", frame_end); cerr.flush ();
	    }
	}

	if (residual_deltas.size () == 0)
	    break;
	if (frame_end > realstart + LENGTH)
	    break;	
    }

    cerr << endl << "===== DONE! =====" << endl;
    CloseLogs (InterestLogs, "InterestLogs.*");
    CloseLogs (DeltaLogs, "DeltaLogs.*");

    Benchmark::print ();
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
