#!/usr/bin/perl
#
# Checks error of latencies generated by ping server and a topology 
# (in intel/defrag)
#

use strict;
use Getopt::Std;
use vars qw($opt_t $opt_o);

getopts("t:o");

my $topo = $opt_t || die "require -t topo";
my $old  = $opt_o;

my %id_map;
my %lat_map;

print STDERR "loading topo: $topo...\n";

open(T, "<$topo") || die "can't open $topo: $!";
while (<T>) {
    chomp;
    
    if (!$old) {
	if (/^node\s+(\d+)\s+([^\t ]+)\s+([^\t ]+)\s+([^\t ]+)\s+([^\t ]+)/) {
	    my ($id,$addr,$vaddr) = ($1,$2,$3,$4,$5);
	    $id_map{$vaddr} = $id;
	    
	    $lat_map{"$id,$id"} = 0;
	}
    } else {
	if (/^node\s+(\d+)\s+([^\t ]+:\d+)/) {
	    my ($id, $addr) = ($1,$2);
	    $id_map{$addr} = $id;
	    
	    $lat_map{"$id,$id"} = 0;
	}
    }

    if (/^(\d+),(\d+)\s+([\d\.]+)/) {
	$lat_map{"$1,$2"} = int($3/2)*2;
	$lat_map{"$2,$1"} = int($3/2)*2;
    }
}
close(T);

print STDERR "checking logs...\n";

foreach my $f (@ARGV) {
    if ($f =~ /\.gz$/) {
	open(F, "zcat $f |") or die "can't open $f";
    } else {
	open(F, "<$f") or die "can't open $f";
    }
    while (<F>) {
	chomp;
	if (/\d+\.\d+\t([\d\.]+:\d+)\t([\d\.]+:\d+)\t(\d+|timeout)/) {
	    my ($src, $dst, $rtt) = ($1,$2,$3);
	    next if $rtt eq 'timeout';
	    
	    if (!$old) {
		$src =~ s/:\d+$//;
		$dst =~ s/:\d+$//;
	    }
	    
	    my $id1 = $id_map{$src};
	    my $id2 = $id_map{$dst};
	    
	    warn "missing id for $src" if (!defined $id1);
	    warn "missing id for $dst" if (!defined $id2);
	    next if !defined $id1 || !defined $id2;
	    
	    my $actual = $lat_map{"$id1,$id2"};
	    if (!defined $actual) {
		warn "missing lat for $id1,$id2";
		next;
	    }
	    my $absolute = abs($rtt - $actual);
	    my $relative = $actual == 0 ? ($absolute == 0 ? 0 : 1) : abs($absolute)/$actual;
	    
	    print "$src,$dst\t$absolute\t$relative\n";
	}
    }
    close(F);
}
