#!/usr/bin/perl
#
# Generate aggregate bandwidth stats for a client-server and broadcast log;
# 
# $Id: CSBMsgLogTmSeries.pl 2289 2005-10-16 22:07:10Z ashu $
#
# -s num   - skip the first x seconds of the trace
# -l num   - the length of time to aggregate over (ignore remainder)
# -e epoch - size of each aggregation epoch (sec -- floating point)
# -t type  - client-server or broadcast
# -T       - synchronize all the files; use the skip time as the time from the first timestamp you observe in all the files

use strict;
use Getopt::Std;
use vars qw($opt_s $opt_l $opt_e $opt_t $opt_T $opt_w); 
use Statistics::Descriptive;

my $basedir;
chomp ($basedir = `dirname $0`);
require "$basedir/Common.pl";
our $PROTO_OVERHEAD_BYTES = 28;

getopts("s:l:e:t:Tw:");

our $SKIPTIME = defined $opt_s ? $opt_s : 0;
our $LENGTH   = defined $opt_l ? $opt_l : undef;
our $EPOCH    = defined $opt_e ? $opt_e : 1;
our $TYPE     = defined $opt_t ? $opt_t : 'c';
our $DO_SYNC    = defined $opt_T;
our $SLOWDOWN = defined $opt_w ? $opt_w : 1;
if ($SLOWDOWN > 1.0) { 
    $SKIPTIME *= $SLOWDOWN;
    if (defined $LENGTH) {
	$LENGTH *= $SLOWDOWN;
    }
}

###############################################################################

our %MsgTypes = ParseMessageHeader($0);
my ($expstats, $ninefive, $ninenine) = ProcessInput(@ARGV);

printf("%-15s %12s %12s %12s %12s %12s\n", "Type", "mean", "stddev", "min", "max", "median");
print "-"x80, "\n";
foreach my $dir ("OUT", "IN") {
# we print the mean and median of each of these statistics.

    printf ("%-15s %12.3f %12.3f %12.3f %12.3f %12.3f\n", "$dir:MEAN", 
            $expstats->{$dir}->mean(),
            $expstats->{$dir}->standard_deviation(),
            $expstats->{$dir}->min(),
            $expstats->{$dir}->max(),
	    $expstats->{$dir}->median());
    printf ("%-15s %12.3f %12.3f %12.3f %12.3f %12.3f\n", "$dir:95%", 
            $ninefive->{$dir}->mean(),
            $ninefive->{$dir}->standard_deviation(),
            $ninefive->{$dir}->min(),
            $ninefive->{$dir}->max(),
	    $ninefive->{$dir}->median());
    printf ("%-15s %12.3f %12.3f %12.3f %12.3f %12.3f\n", "$dir:99%", 
            $ninenine->{$dir}->mean(),
            $ninenine->{$dir}->standard_deviation(),
            $ninenine->{$dir}->min(),
            $ninenine->{$dir}->max(),
	    $ninenine->{$dir}->median());
}

###############################################################################

sub ProcessInput(@)
{
    my @files = @_;
    my $start;

    if ($DO_SYNC) {
	$start = GetAbsoluteStart(@files);
    }
    
    my %expstats = ();
    my %ninefive = ();
    my %ninenine = ();
    foreach my $dir ("IN", "OUT") {
        $expstats{$dir} = new Statistics::Descriptive::Full();
	$ninefive{$dir} = new Statistics::Descriptive::Full();
	$ninenine{$dir} = new Statistics::Descriptive::Full();
    }
    
    foreach my $f (@files) {
	my $fstats = ProcessFile($f, $start);

	foreach my $dir ("IN", "OUT") {
            $expstats{$dir}->add_data($fstats->{$dir}->trimmed_mean(0.05));
	    $ninefive{$dir}->add_data(scalar $fstats->{$dir}->percentile(95));
	    $ninenine{$dir}->add_data(scalar $fstats->{$dir}->percentile(99));
	}
    }
    
    return (\%expstats, \%ninefive, \%ninenine);
}

#
# Make one pass through all the files to get the first time observed in the logs 
#
sub GetAbsoluteStart(@) 
{
    my @files = @_;
    my $absolute_start = 0;
    foreach my $f (@files) {
        open(F, "<$f") || die "can't open $f";

	$_ = <F>;    # read first line;
	m/(\d+\.\d+)/;
	
	my $time = $1;
	if ($absolute_start == 0 or $time < $absolute_start) {
	    $absolute_start = $time;
	}
	close F;
    }

     #print STDERR "absolute start = $absolute_start\n";
    return $absolute_start;	
}

sub Percentile($$) {
    my ($stats, $perc) = @_;

    my @a = $stats->get_data();
    @a = sort { $a <=> $b } @a;
    return $a[int($perc/(scalar @a))];
}

# record stats for each bucket -
# ultimately, we need to report mean, median, 95th and stddev for this guy
#
sub ProcessFile($$)
{
    my $f = shift;
    my $start = shift;
    my $end;
    
    print STDERR "processing file $f...\n";
    my %stats = ();
    foreach my $dir ("IN", "OUT") {
        $stats{$dir} = new Statistics::Descriptive::Full();
    }
    
    open(F, "<$f") || die "can't open $f";

    my $print = 1;
    my $prev_index = -1;
    my $index = -1;
    
    my %buckets = ();
    foreach my $dir ("IN", "OUT") {
        $buckets{$dir} = 0;
    }
    
    my $s_epoch = $EPOCH * $SLOWDOWN;
    while (<F>) {
	chomp $_;
        my $time;
        my $in;
        my $out;
        
        if ($TYPE eq 'c') { # client-server log 
            m/(\d+\.\d+)\t(\d+)/;
            $time = $1;
            $in = $out = $2;
        }
        else {
            m/(\d+\.\d+)\t(\d+)\t(\d+)/;
            $time = $1;
            $in = $2;
            $out = $3;
        }

	if (!defined $start) {
	    $start = $time;
	}
	if ( $time < $start + $SKIPTIME ) {
	    next;
	}
	if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
	    last;
	}
	if ($time > $end) {
	    $end = $time;
	}

	$index = int( ($time - ($start+$SKIPTIME))/$s_epoch );
	
        $in  = ($in * 8 / 1000)/$EPOCH;
        $out = ($out * 8 / 1000)/$EPOCH;

        $buckets{"IN"} += $in;
        $buckets{"OUT"} += $out;
	
	if ($index != $prev_index) {
	    $prev_index = $index;
	    foreach my $dir ("IN", "OUT") {
                $stats{$dir}->add_data($buckets{$dir});
                $buckets{$dir} = 0;
	    }
	}
    }
    if ($index != $prev_index) {
	foreach my $dir ("IN", "OUT") {
            $stats{$dir}->add_data($buckets{$dir});
            $buckets{$dir} = 0;
	}
    }
    close F;

    return \%stats;
}


