#!/usr/bin/perl
#
# Generate aggregate bandwidth stats for a MessageLog
#
# $Id: AggMsgLogAggregate.pl 2267 2005-10-15 01:24:27Z jeffpang $
#
# -S time  - use this as the starttime (before skipping)
# -s num - skip the first x seconds of the trace
# -l num - the length of time to aggregate over (ignore remainder)
# -b     - the log is binary

use strict;
use Getopt::Std;
use vars qw($opt_S $opt_s $opt_l $opt_w);

my $basedir;
chomp ($basedir = `dirname $0`);
require "$basedir/Common.pl";

our $PROTO_OVERHEAD_BYTES = 28;

getopts("S:s:l:w:");

our $START    = defined $opt_S ? $opt_S : undef;
our $SKIPTIME = defined $opt_s ? $opt_s : 120;
our $LENGTH   = defined $opt_l ? $opt_l : 600;
our $SLOWDOWN = defined $opt_w ? $opt_w : 1;

###############################################################################

our %MsgTypes = ParseMessageHeader($0);
#foreach my $k (sort {$a <=> $b } keys %MsgTypes) {
#    print "$k\t$MsgTypes{$k}\n";
#}
#exit 1;
our %stats = ProcessInput(@ARGV);

print sprintf("%-28s %15s   %15s   %15s\n",
	      "Type", "Cost (Kbps)", "Count (pps)", "Cost w/ Proto");
foreach my $s (sort { $a cmp $b } keys %stats) {
    print sprintf("%-28s %15.3f   %15.3f   %15.3f\n", 
		  $s, $stats{$s}->[0], $stats{$s}->[1], $stats{$s}->[2]);
}

###############################################################################

sub ProcessInput(@)
{
    my @files = @_;

    my($start, $end);
    $start = $START;
    my %stats;

    # [ size, count, with_overhead ]
    $stats{"IN"}  = [0,0];
    $stats{"OUT"} = [0,0];

    foreach my $f (@files) {
        if ($f =~ /\.gz$/) {
	    open(F, "zcat $f |") || die "can't pipe $f";
        } else {
	    open(F, "<$f") || die "can't open $f";
        }

        #print STDERR "doing $f...\n";

	while (<F>) {
	    #print;
	    chomp $_;
	    m/(\d+\.\d+)\t(\d+)\t(\d+)\t(\d+)\t(\d+)\t\s+(\d+\.\d+)/ or warn "bad line: $_" && next;
	    
	    my $time = $1;
	    my $dir = $2;
	    my $type = $3;
	    my $size = $4;
	    my $samples = $5;
	    my $avghopcount = $6;

	    #print "time=$time dir=$dir type=$MsgTypes{$type} size=$size samples=$samples\n";
	    
	    if (!defined $start) {
		$start = $time;
	    }
	    if ( $time < $start + $SKIPTIME ) {
		next;
	    }
	    if (defined $LENGTH && $time > $start + $SKIPTIME + $LENGTH) {
		last;
	    }
	    if ($time > $end) {
		$end = $time;
	    }
	    
	    $dir   = $dir   == 0 ? "IN" : "OUT";
	    $type  = $MsgTypes{$type} || $type;
	    
	    $stats{"$dir:$type"} = [0,0] if ! defined $stats{"$dir:$type"};
	    
	    $stats{"$dir:$type"}->[0] += $size;
	    $stats{"$dir:$type"}->[1] += $samples;
	    
	    $stats{$dir}->[0] += $size;
	    $stats{$dir}->[1] += $samples;
	}
    }

    my $interval = $end - $start - $SKIPTIME;
    $interval /= $SLOWDOWN;         # this will scale bandwidth up;
    
    foreach my $s (keys %stats) {
	$stats{$s}->[0] = $stats{$s}->[0]*8/(1000*$interval);
	$stats{$s}->[1] = $stats{$s}->[1]/$interval;
	# suppose we including IP/UDP overhead
	$stats{$s}->[2] = 
	    $stats{$s}->[0] + $stats{$s}->[1]*$PROTO_OVERHEAD_BYTES*8/1000;
    }

    return %stats;
}
