#!/usr/bin/perl
#
# ResolveHosts.pl
#
# This script resolves the planetlab hostnames in planetlab/hostnames.txt
# and writes the IPs into planetlab/hosts.txt. It pings the host to ensure
# that it is online. If the name does not resolve or the host does not
# respond, then it is skipped.
#

use strict;
use lib "./planetlab";
use Travertine;
use PlanetLabConf;
use Getopt::Std;
use vars qw($opt_R);

getopts("R");

$Travertine::DEBUG_SHOW_STACKTRACE = 0;

$HOSTS_FILE = defined $ARGV[0] ? $ARGV[0] : "planetlab/hostnames.txt";
our @hosts  = GetPlanetLabHosts();

tinfo "## resolving hostnames.txt...";

my @ips = ParallelExec3(10, sub {
    my $h = shift;
    my $ip = join(".", unpack 'C4', (gethostbyname $h)[4]);

    ## require domain name resolves
    if (!$ip) {
	tinfo "WARNING: $h did not resolve, skipping";
    } else {
	goto done if $opt_R;
	## require that it is pingable
	my $out = `ping -w 5 -c 1 $ip 2>&1`;
	if ($out =~ /100\% loss/) {
	    tinfo "WARNING: $h ($ip) appears to be dead, skipping";
	    $ip = undef;
	} else {
	    ## require that we can ssh into it
	    my $ssh = new Travertine::SSHTunnel($USERNAME, $ip);
	    my $err;
	    ($out, $err, undef) = $ssh->exec("echo ok", undef, 30);
	    if ($out !~ /ok/) {
		tinfo "WARNING: $h ($ip) won't let us ssh in, it said: $err";
		$ip = undef;
	    }
	}
    }

done:
    return $ip;
}, @hosts);

if ($opt_R) {
	*H = *STDOUT;
} else {
	open(H, ">planetlab/hosts.txt") or die "can't open hosts.txt: $!";
}
for (my $i=0; $i<@hosts; $i++) {
    print H "$ips[$i]\t# $hosts[$i]\n" if $ips[$i];
}
close(H);

tinfo "## done!";
